<?php
namespace App\Http\Controllers;

use App;
use App\Models\Article;
use App\Models\Translation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\View;
use Intervention\Image\ImageManagerStatic as Image;
use Redirect;

class ArticleController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            } else {
                $settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }
        
        // List of languages
        $translations = Translation::orderBy('id', 'ASC')->get();
         
        foreach ($translations as $language) {
            $language_id[$language->code] = $language->id;
            $language_code[$language->id]=$language->icon;
            if ( $settings['root_language'] == '1' && $language->code == $settings['site_language'] ) {
            $language_prefix[$language->id] = '/';
            } else {
            $language_prefix[$language->id] = $language->code.'/';
            }
        }
        
        // Array Data
        $array_data = array(
            'language_code' => $language_code,
            'language_prefix' => $language_prefix,
            'site_language_id' => $language_id[$settings['site_language']],
        );
        
        // Pass data to views
        View::share(['settings' => $settings, 'translations' => $translations, 'array_data' => $array_data]);
    }

    /** Index */
    public function index()
    {
        // List of articles
        $articles = Article::orderBy('id', 'DESC')->paginate(15);

        // Return view
        return view('adminlte::articles.index', compact('articles'));
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::articles.create');
    }

    /** Store */
    public function store(Request $request)
    {
         // Check if slug exists
         $slug_check = Article::where('slug', $request->get('slug'))->first();

         // Return error message if slug is in use
         if ($slug_check != null) {
             return Redirect::back()->withErrors(__('admin.slug_in_use'));
         }
         
        $this->validate($request, [
            'title' => 'required|max:255',
            'description' => 'required|max:755',
            'details' => 'required',
            'image' => 'required',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
        ]);

        $article = new Article;
        $article->slug = $request->get('slug');
        $article->title = $request->get('title');
        $article->description = $request->get('description');
        $article->description = strip_tags($article->description);
        $article->custom_title = $request->get('custom_title');
        $article->custom_description = $request->get('custom_description');
        $article->language = $request->get('language');
        
        $article->details = $request->get('details');
      
        preg_match_all('/src="(data:image\/[^;]+;base64[^"]+)"/', $article->details, $result, PREG_PATTERN_ORDER);

        $total_images = count($result[0]);

        for ($x = 0; $x <= $total_images-1; $x++) {
                $data = substr($result[0][$x], strpos($result[0][$x], ',') + 1);
                $data = base64_decode($data);
            if ($this->save_as_webp == '1') {
                    $file_name = time() . rand(11111, 99999) . '.webp';
            } else {
                    $file_name = time() . rand(11111, 99999) . '.png';
            }
            \File::put(public_path(). '/images/uploads/' . $file_name, $data);
            $imagefile=asset('images/uploads').'/'.$file_name;
            $article->details = str_replace($result[0][$x], "src=\"$imagefile\" class=\"img-fluid\"", $article->details);
            $article->details = preg_replace('/(\<img[^>]+)(style\=\"[^\"]+\")([^>]+)(>)/', '${1}${3}${4}', $article->details);
        }

        // Check if the picture has been uploaded
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/articles/' . $file_name);
            Image::make($image)->resize(770, 450)->save($location, $this->image_quality);
            $cache_location = public_path('images/cache/' . $file_name);
            Image::make($image)->resize(275, 161)->save($cache_location, $this->image_quality);
            $article->image = $file_name;
        }

        $article->save();

        if ($request->get('slug') == null) {
            $article->slug = null;
            $article->update(['title' => $article->title]);
        }

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/articles');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Clear cache
        Cache::flush();

        // Redirect to article edit page
        return redirect()->route('articles.edit', $article->id)
            ->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve article details
        $article = Article::find($id);

        // Return 404 page if article not found
        if ($article == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::articles.edit', compact('article', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
         // Check if slug exists
         $slug_check = Article::where('slug', $request->get('slug'))->where('id', '!=', $id)->first();

         // Return error message if slug is in use
         if ($slug_check != null) {
             return Redirect::back()->withErrors(__('admin.slug_in_use'));
         }
         
        $this->validate($request, [
            'title' => 'required|max:255',
            'description' => 'required|max:755',
            'details' => 'required',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
        ]);

        // Retrieve article details
        $article = Article::find($id);
        $article->slug = $request->get('slug');
        $article->title = $request->get('title');
        $article->description = $request->get('description');
        $article->description = strip_tags($article->description);
        $article->custom_title = $request->get('custom_title');
        $article->custom_description = $request->get('custom_description');
        $article->language = $request->get('language');

        $article->details = $request->get('details');
      
        preg_match_all('/src="(data:image\/[^;]+;base64[^"]+)"/', $article->details, $result, PREG_PATTERN_ORDER);

        $total_images = count($result[0]);

        for ($x = 0; $x <= $total_images-1; $x++) {
                $data = substr($result[0][$x], strpos($result[0][$x], ',') + 1);
                $data = base64_decode($data);
            if ($this->save_as_webp == '1') {
                    $file_name = time() . rand(11111, 99999) . '.webp';
            } else {
                    $file_name = time() . rand(11111, 99999) . '.png';
            }
            \File::put(public_path(). '/images/uploads/' . $file_name, $data);
            $imagefile=asset('images/uploads').'/'.$file_name;
            $article->details = str_replace($result[0][$x], "src=\"$imagefile\" class=\"img-fluid\"", $article->details);
            $article->details = preg_replace('/(\<img[^>]+)(style\=\"[^\"]+\")([^>]+)(>)/', '${1}${3}${4}', $article->details);
        }

        // Check if the picture has been changed
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/articles/' . $file_name);
            Image::make($image)->resize(770, 450)->save($location, $this->image_quality);
            $old_image = public_path('images/articles/' . $article->image);
            $cache_location = public_path('images/cache/' . $file_name);
            Image::make($image)->resize(275, 161)->save($cache_location, $this->image_quality);
            $article->image = $file_name;
            File::delete($old_image);
        }

        // Ping Google
        if ($article->isDirty() && $this->ping_google == '1') {
            $sitemap_url = asset('sitemap/articles');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        if ($request->get('slug') == null) {
            $article->slug = null;
            $article->update(['title' => $article->title]);
        }
        
        $article->save();

        // Clear cache
        Cache::flush();

        // Redirect to article edit page
        return redirect()
            ->route('articles.edit', $article->id)
            ->with('success', __('admin.data_updated'));
    }

    /** Destroy */
    public function destroy($id)
    {
        // Retrieve article details
        $article = Article::find($id);

        if (!empty($article->image)) {
            if (file_exists(public_path() . '/images/articles/' . $article->image)) {
                unlink(public_path() . '/images/articles/' . $article->image);
            }
            if (file_exists(public_path() . '/images/cache/' . $article->image)) {
                unlink(public_path() . '/images/cache/' . $article->image);
            }
        }

        $article->delete();

        // Clear cache
        Cache::flush();

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/articles');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Redirect to list of articles
        return redirect()
            ->route('articles.index')
            ->with('success', __('admin.data_deleted'));
    }

}
