<?php

namespace App\Http\Controllers\Auth;

use App;
use App\Http\Controllers\Controller;
use DB;
use GuzzleHttp\Client;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Redirect;
use View;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
     */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */

    protected function redirectTo()
    {
        return env('ADMIN_URL');
    }

    /**
     * Create a new controller instance.
     *
     * @return void
     */

    public function __construct()
    {
        if(env('ADMIN_IP_ADDRESS') != '' && env('ADMIN_IP_ADDRESS') != \Request::ip()) {
            abort(404);
        }
        
        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $settings[$setting->name] = $setting->value;
        }

        $this->google_recaptcha_secret_key = $settings['google_recaptcha_secret_key'];
        $this->enable_google_recaptcha = $settings['enable_google_recaptcha'];

        App::setLocale(app('site_lang'));
        $this->middleware('guest')->except('logout');

        // Pass data to views
        View::share(['settings' => $settings]);
    }

    /**
     * Validate the user login request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    protected function validateLogin(Request $request)
    {
        $this->validate($request, [
            $this->username() => 'required|string',
            'password' => 'required|string',
            'g-recaptcha-response' => 'sometimes|nullable|string',
        ]);

        if ($this->enable_google_recaptcha == '1') {

            // Google reCAPTCHA validation
            $secret = $this->google_recaptcha_secret_key;
            $recaptcha_data = $request->get('g-recaptcha-response');

            $client = new \GuzzleHttp\Client();
            $response = $client->request('POST', 'https://www.google.com/recaptcha/api/siteverify', [
                'form_params' => [
                    'secret' => $secret,
                    'response' => $recaptcha_data,
                ],
            ]);

            $response = $response->getBody();

            $responseData = json_decode($response, true);

            if ($responseData['success'] == false) {
                // If Google reCAPTCHA validation fails, return error response
                return $this->CustomsendFailedLoginResponse($request);
            }
        }
    }

    // Custom Error Message for Google reCAPTCHA
    protected function CustomsendFailedLoginResponse(Request $request)
    {
        throw ValidationException::withMessages([
            trans('general.recaptcha_error'),
        ]);
    }

}
