<?php

namespace App\Http\Controllers;

use App\Mail\Confirmmail;
use App\Mail\Newpassword;
use App\Mail\Passwordreset;
use App\Mail\Submissionreceived;
use App\Models\Bank;
use App\Models\Category;
use App\Models\Link;
use App\Models\Package;
use App\Models\Page;
use App\Models\Partner;
use App\Models\Submission;
use Auth;
use Brick\PhoneNumber\PhoneNumber;
use Brick\PhoneNumber\PhoneNumberParseException;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Str;
use Intervention\Image\ImageManagerStatic as Image;
use Mail;

class FrontendMemberController extends Controller
{
    public function __construct()
    {
        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            } else {
                $settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        // List of languages
        $languages = DB::table('translations')->OrderBy('sort', 'ASC')->get();

        $lang_codes = [];
        foreach ($languages as $language) {
            if ($language->code == $settings['site_language']) {
                $site_language_id = $language->id;
            }
            array_push($lang_codes, $language->code);
            $language_id[$language->code] = $language->id;
            $language_icon[$language->code] = $language->icon;
            $language_direction[$language->id] = $language->text_direction;
            if ($settings['root_language'] == '1' && $language->code == $settings['site_language']) {
                $menu_language_prefix[$language->id] = '/';
            } else {
                $menu_language_prefix[$language->id] = $language->code . '/';
            }
            $locale_tags[$language->id] = $language->locale_code;
        }

        if ($settings['root_language'] == '1') {

            if (request()->segment(1) == $settings['site_language'] && request()->segment(2) == null) {
                Redirect::to(asset('/'), 301)->send();
            }

            if (request()->segment(1) == $settings['site_language']) {
                abort(404);
            }

            if (in_array(request()->segment(1), $lang_codes) && request()->segment(1) != $settings['site_language']) {
                $this->language_id = $language_id[request()->segment(1)];
                $language_code = request()->segment(1);
                $language_icon_code = $language_icon[request()->segment(1)];
            } else {
                $this->language_id = $language_id[$settings['site_language']];
                $language_code = null;
                $language_icon_code = $language_icon[$settings['site_language']];
            }

        } else {

            if (!in_array(request()->segment(1), $lang_codes) && request()->segment(1) == null) {
                Redirect::to($settings['site_language'], 301)->send();
            }

            if (!in_array(request()->segment(1), $lang_codes)) {
                abort(404);
            }

            $language_code = request()->segment(1);
            $this->language_id = $language_id[$language_code];

            $language_icon_code = $language_icon[$language_code];
        }

        $this->language_prefix = $language_code . '/';

        // List of language slugs
        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            if ($setting->language == $this->language_id) {
                $this->$setting_name = $setting->value;
                $settings[$setting->name] = $setting->value;
            }
        }
        if ($language_direction[$this->language_id] == '1') {
            $this->language_direction = null;
        } else {
            $this->language_direction = 'rtl-';
        }

        // Open Graph Locale Tag
        $locale_tag = $locale_tags[$this->language_id];

        // List of pages in footer
        $pages = Page::OrderBy('sort', 'ASC')->where('language', $this->language_id)->get();

        // List of categories in footer
        $this->categories = Category::where('language', $this->language_id)->orderBy('title', 'ASC')->get();

        // List of partners in footer
        $partners = Partner::where('language', $this->language_id)->orderBy('sort', 'ASC')->get();

        // List of bank accounts
        $this->banks = Bank::where('enabled', 1)->orderBy('sort', 'ASC')->get();

        // List of packages
        $this->packages = Package::OrderBy('sort', 'ASC')->where('available', 1)->get();

        // List of payment methods
        $payment_methods = DB::table('payment_methods')->where('enabled', 1)->orderBy('title', 'ASC')->get();

        $category_slug = [];

        foreach ($this->categories as $category) {
            $category_slug[$category->id] = $category->slug;
        }

        $this->mail_site_title = $settings['site_title' . '-' . $site_language_id];
        $this->mail_from = $_ENV['MAIL_FROM_ADDRESS'];

        // Pass data to views
        View::share(['settings' => $settings, 'pages' => $pages, 'categories' => $this->categories, 'partners' => $partners, 'locale_tag' => $locale_tag, 'category_slug' => $category_slug, 'link_types' => link_types(), 'packages' => $this->packages, 'payment_methods' => $payment_methods, 'banks' => $this->banks, 'languages' => $languages, 'language_prefix' => $this->language_prefix, 'language_icon_code' => $language_icon_code, 'menu_language_prefix' => $menu_language_prefix]);
    }

    /** Index */
    public function index()
    {
        // Redirect to My Links page
        return Redirect::to($this->language_prefix . 'member-area/my-links');
    }

    /** Member Login Page  */
    public function login()
    {
        if (Auth::check()) {
            return Redirect::to($this->language_prefix . 'member-area');
        }

        // Meta tags
        meta_tags(__('general.member_login') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // Return view
        return response()->view('' . $this->language_direction . 'frontend::member.login');
    }

    /** Member Login Function */
    public function dologin(Request $request)
    {
        if ($this->enable_google_recaptcha == '1') {
            $dataa = recaptcha_check($request->input('g-recaptcha-response'), $this->google_recaptcha_secret_key);

            if ($dataa == 'error') {
                return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.recaptcha_error'));
            }
        }

        // Validate the info, create rules for the inputs
        $rules = array(
            'email' => 'required|email',
            'password' => 'required|min:6',
        );

        // Run the validation rules on the inputs from the form
        $validator = Validator::make($request->all(), $rules);

        // If the validator fails, redirect back to the form
        if ($validator->fails()) {
            return Redirect::to($this->language_prefix . 'member-area/login')->withErrors($validator->getMessageBag()->toArray());
        } else {

            $email = $request->input('email');
            $password = $request->input('password');
            $remember_me = $request->has('remember_me') ? true : false;

            // Attempt to do the login
            if (Auth::attempt(['email' => $email, 'password' => $password, 'user_type' => '2'], $remember_me)) {

                $member_details = Auth::user();

                if ($member_details->disabled == '1') {
                    Auth::logout();
                    return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.account_disabled'));
                } else if ($member_details->activated == '0') {
                    Auth::logout();
                    return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.account_not_activated'));
                } else {
                    $ip_address = $request->ip();
                    DB::update("update users set last_ip = '$ip_address' WHERE id = $member_details->id");
                    return Redirect::to($this->language_prefix . 'member-area');
                }

            } else {
                // If validation not successful, send back to form
                return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.check_username_and_password'));
            }
        }
    }

    /** Member Register Page */
    public function register()
    {
        // Return 404 page if registration is disabled
        $this->enable_registration == 0 ? abort(404) : '';

        // Meta tags
        meta_tags(__('general.member_register') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // Return view
        return response()->view('' . $this->language_direction . 'frontend::member.register');
    }

    /** Member Register Function */
    public function doregister(Request $request)
    {
        // Return 404 page if registration is disabled
        $this->enable_registration == 0 ? abort(404) : '';

        if ($this->enable_google_recaptcha == '1') {
            $dataa = recaptcha_check($request->input('g-recaptcha-response'), $this->google_recaptcha_secret_key);

            if ($dataa == 'error') {
                return Redirect::to($this->language_prefix . 'member-area/register')->withErrors(__('general.recaptcha_error'));
            }
        }

        // Validate the info, create rules for the inputs
        $rules = array(
            'email' => 'required|email',
            'name' => 'required',
            'password' => 'required|min:6|confirmed',
            'password_confirmation' => 'required|min:6',
            'terms' => 'accepted',
        );

        // Run the validation rules on the inputs from the form
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return Redirect::to($this->language_prefix . 'member-area/register')->withErrors($validator->getMessageBag()->toArray());
        }

        // Check if user exists
        $user_check = DB::table('users')->where('email', $request->get('email'))->first();

        // Return error message if user exists
        if ($user_check != null) {
            return Redirect::to($this->language_prefix . 'member-area/register')->withErrors("Member exists.");
        } else {

            $rand_string = Str::random(60);
            $verification_token = Str::lower($rand_string);
            $link = asset($this->language_prefix . 'member-area/verify-email/' . $verification_token);

            // Insert Record
            $last_user_id = DB::table('users')->insertGetId(array(
                'name' => request()->name,
                'email' => request()->email,
                'password' => Hash::make($request->password),
                'user_type' => '2',
                'verification_token' => $verification_token,
                'ip' => $request->ip(),
                'last_ip' => $request->ip(),
                'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
            ));

            $email = $request->get('email');

            // Insert Notification Record
            insert_notification('1', request()->email);

            Mail::to($email)->send(new Confirmmail($link, $this->mail_site_title, $this->mail_from));
            return Redirect::to($this->language_prefix . 'member-area/register')->withSuccess(__('general.account_created_successfully'));

        }

    }

    /** Password Reset Page */
    public function passwordreset()
    {
        // Meta tags
        meta_tags(__('general.forgot_password') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // Return view
        return response()->view('' . $this->language_direction . 'frontend::member.reset');
    }

    /** Password Reset Function */
    public function dopasswordreset(Request $request)
    {
        if ($this->enable_google_recaptcha == '1') {
            $dataa = recaptcha_check($request->input('g-recaptcha-response'), $this->google_recaptcha_secret_key);

            if ($dataa == 'error') {
                return Redirect::to($this->language_prefix . 'member-area/password-reset')->withErrors(__('general.recaptcha_error'));
            }
        }

        // Validate the info, create rules for the inputs
        $rules = array(
            'email' => 'required|email',
        );

        // Run the validation rules on the inputs from the form
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return Redirect::to($this->language_prefix . 'member-area/password-reset')->withErrors($validator->getMessageBag()->toArray());
        }

        // Check if user exists
        $user_check = DB::table('users')->where('disabled', 0)->where('activated', 1)->where('user_type', 2)->where('email', $request->get('email'))->first();

        // Return error message if user does not exists
        if ($user_check == null) {
            return Redirect::to($this->language_prefix . 'member-area/password-reset')->withErrors(__('general.user_not_exist_or_activated'));
        } else {

            // Random Password Reset Token
            $rand_string = Str::random(60);
            $password_reset_token = Str::lower($rand_string);
            $link = asset($this->language_prefix . 'member-area/password-request/' . $password_reset_token);

            $email = $request->get('email');

            DB::update("update users set password_reset_token = '$password_reset_token' WHERE id = $user_check->id");

            Mail::to($email)->send(new Passwordreset($link, $this->mail_site_title, $this->mail_from));

            return Redirect::to($this->language_prefix . 'member-area/password-reset')->withSuccess(__('general.password_reset_email'));
        }

    }

    /** Password Sender Function */
    public function passwordrequest()
    {
        $slug = request()->slug;

        // Check if the password reset token is valid
        $password_reset_query = DB::table('users')->where('password_reset_token', $slug)->where('activated', 1)->first();
        if ($password_reset_query == null) {
            return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.invalid_reset_link'));
        } else {
            DB::update("update users set password_reset_token = NULL WHERE id = $password_reset_query->id");

            $email = $password_reset_query->email;

            $new_password = Str::random(6);
            $new_password_hash = Hash::make($new_password);

            $login_link = asset($this->language_prefix . 'member-area/login');

            DB::update("update users set password = '$new_password_hash' WHERE id = $password_reset_query->id");

            Mail::to($email)->send(new Newpassword($new_password, $login_link, $this->mail_site_title, $this->mail_from));

            return Redirect::to($this->language_prefix . 'member-area/login')->withSuccess(__('general.password_sent_to_mail'));
        }

    }

    /** Verify E-mail Function */
    public function doverify()
    {
        $slug = request()->slug;

        // Check if the verification token is valid
        $verify_query = DB::table('users')->where('verification_token', $slug)->where('activated', 0)->first();

        // Check if user activated
        if ($verify_query == null) {
            return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.invalid_verification_link'));
        } else {
            DB::update("update users set activated = '1' WHERE id = $verify_query->id");
            DB::update("update users set verification_token = NULL WHERE id = $verify_query->id");
            return Redirect::to($this->language_prefix . 'member-area/login')->withSuccess(__('general.account_activation_success'));
        }
    }

    /** My Account Page */
    public function myaccount()
    {
        // Meta tags
        meta_tags(__('general.my_account') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        $member = Auth::user();
        return view('' . $this->language_direction . 'frontend::member.myaccount')->with('member', $member);
    }

    /** Account Update Function */
    public function domyaccount(Request $request)
    {
        // Validate the info, create rules for the inputs
        $rules = array(
            'name' => 'required',
            'current_password' => 'nullable|min:6|required_with:new_password,new_password_confirmation',
            'new_password' => 'nullable|required_with:current_password,new_password_confirmation|min:6|different:current_password',
            'new_password_confirmation' => 'same:new_password',
        );

        // Run the validation rules on the inputs from the form
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return Redirect::to($this->language_prefix . 'member-area/my-account')->withErrors($validator->getMessageBag()->toArray());
        }

        $member = Auth::user();
        $name = $request->input('name');
        $current_password = $request->input('current_password');
        $password = $request->input('new_password');
        $hashed_password = Hash::make($password);

        DB::update("update users set name = '$name' WHERE id = $member->id");

        if (!empty($current_password)) {
            if (Hash::check($current_password, $member->password)) {
                DB::update("update users set password = '$hashed_password' WHERE id = $member->id");
                return Redirect::to($this->language_prefix . 'member-area/my-account')->withSuccess(__('general.password_changed'));
            } else {
                return Redirect::to($this->language_prefix . 'member-area/my-account')->withErrors(__('general.password_dont_match'));
            }

        }

        return Redirect::to($this->language_prefix . 'member-area/my-account')->withSuccess(__('general.account_updated'));
    }

    /** My Orders Page */
    public function my_orders()
    {
        // Meta tags
        meta_tags(__('general.my_orders') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        $member = Auth::user();

        $orders = DB::table('orders')
            ->where('user', $member->id)
            ->orderBy('id', 'desc')
            ->paginate(15);

        return view('' . $this->language_direction . 'frontend::member.myorders')->with('member', $member)->with('orders', $orders);
    }

    /** Order Details Page */
    public function order_details()
    {
        $order_id = request()->id;

        $member = Auth::user();

        // Check if the order exists
        $order_query = DB::table('orders')
            ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
            ->select('submissions.*', 'orders.*')
            ->where('orders.id', $order_id)
            ->where('orders.user', $member->id)
            ->first();

        // Return 404 page if order not found
        $order_query == null ? abort(404) : '';

        // List of categories
        $categories_list = Category::get();

        // Meta tags
        meta_tags(__('general.order_details') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        return view('' . $this->language_direction . 'frontend::member.order')->with('member', $member)->with('order_query', $order_query)->with('categories_list', $categories_list);
    }

    /** Payment Notification Form */
    public function payment_notification()
    {
        $order_id = request()->id;

        $member = Auth::user();

        // Check if the order exists
        $order_query = DB::table('orders')->where('id', $order_id)->where('payment_method', '2')->where('user', $member->id)->first();

        // Return 404 page if order not found
        $order_query == null ? abort(404) : '';

        // Meta tags
        meta_tags(__('general.bank_payment_notification') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        return view('' . $this->language_direction . 'frontend::member.payment_notification')->with('member', $member)->with('order_query', $order_query);
    }

    /** My Links Page */
    public function my_links()
    {
        // Meta tags
        meta_tags(__('general.my_links') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        $member = Auth::user();

        $links = DB::table('links')
            ->leftJoin('packages', 'links.package', '=', 'packages.id')
            ->select('packages.price AS price', 'packages.duration AS duration', 'links.*')
            ->where('owner', $member->id)
            ->orderBy('links.id', 'desc')
            ->paginate(15);

        return view('' . $this->language_direction . 'frontend::member.mylinks')->with('member', $member)->with('links', $links)->with('durations', submission_durations());
    }

    /** Packages Page */
    public function packages()
    {
        // Return 404 page if submission form is disabled
        $this->enable_submission_form == 0 ? abort(404) : '';

        // Meta tags
        meta_tags(__('general.packages') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        $member = Auth::user();

        return view('' . $this->language_direction . 'frontend::member.packages')->with('member', $member)->with('durations', submission_durations());
    }

    /** Package Details Page */
    public function package()
    {
        // Return 404 page if submission form is disabled
        $this->enable_submission_form == 0 ? abort(404) : '';

        $package_id = request()->id;

        // Check if the package exists
        $package_query = DB::table('packages')->where('id', $package_id)->where('available', 1)->first();

        // Return 404 page if package not found
        $package_query == null ? abort(404) : '';

        // Meta tags
        meta_tags($package_query->title . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // List of categories
        $submission_categories = Category::where('parent_id', 0)->orderBy('sort', 'ASC')->get();

        $member = Auth::user();

        return view('' . $this->language_direction . 'frontend::member.package')->with('member', $member)->with('package_query', $package_query)->with('submission_categories', $submission_categories);
    }

    /** Link Details Page */
    public function link_details()
    {
        $id = request()->id;

        $member = Auth::user();

        // Check if the link exists
        $link_query = Link::
            LeftJoin('packages', 'links.package', '=', 'packages.id')
            ->select('packages.address AS allow_address', 'packages.deep_links_count AS allowed_deep_links_count', 'packages.deep_links AS allow_deep_links', 'packages.bypass_redirect AS allow_bypass_redirect', 'packages.bypass_details AS allow_bypass_details', 'packages.visible_url AS allow_visible_url', 'packages.reciprocal_link', 'packages.phone_number AS allow_phone_number', 'packages.email AS allow_email', 'links.*')
            ->where('links.id', $id)
            ->where('links.owner', $member->id)
            ->orderBy('links.id', 'desc')
            ->first();

        // Return 404 page if package not found
        $link_query == null ? abort(404) : '';

        $deep_link_data = unserialize($link_query->deep_links);

        if (!$deep_link_data) {
            $deep_link_data = array(
                "link_1" => array("title" => null, "url" => null, "type" => null),
                "link_2" => array("title" => null, "url" => null, "type" => null),
                "link_3" => array("title" => null, "url" => null, "type" => null),
            );
        }

        $other_contacts = unserialize($link_query->other_contacts);

        if (!$other_contacts) {
            $other_contacts = array(
                "facebook_url" => array("url" => null),
                "twitter_url" => array("url" => null),
                "instagram_url" => array("url" => null),
                "linkedin_url" => array("url" => null),
                "whatsapp_number" => array("number" => null),
                "telegram_url" => array("url" => null),
            );
        }

        // Meta tags
        meta_tags(__('general.link_details') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // List of categories
        $submission_categories = Category::where('parent_id', 0)->where('language', $link_query->language)->orderBy('sort', 'ASC')->get();

        return view('' . $this->language_direction . 'frontend::member.details')->with('member', $member)->with('link_query', $link_query)->with('deep_link_data', $deep_link_data)->with('other_contacts', $other_contacts)->with('submission_categories', $submission_categories);
    }

    /** Link Update Function */
    public function link_update(Request $request)
    {
        // Return 404 page if users are not allowed to update their links
        $this->allow_user_updates == 0 ? abort(404) : '';

        $member = Auth::user();

        // Check if the link exists
        $link = Link::where('id', request()->link_id)->where('owner', $member->id)->first();

        // Return 404 page if package not found
        $link == null ? abort(404) : '';

        // Package query
        $package_query = Package::where('id', $link->package)->first();

        // Run the validation rules on the inputs from the form
        $rulesList = [
            "title" => "required",
            "description" => "required|min:25|max:755",
            "url" => "required|url", "image" => "sometimes|image|mimes:jpeg,png,jpg,gif,svg,webp|max:1024|dimensions:min_width=200,min_height=200",
            'categories' => ['required', 'array', 'min:1'],
            'categories.*' => ['required', 'integer', 'exists:categories,id'],
            "deep_link_1_url" => "nullable|url",
            "deep_link_2_url" => "nullable|url",
            "deep_link_3_url" => "nullable|url",
        ];

        if ($package_query->phone_number == 1) {
            $rulesList["phone_number"] = "nullable|max:30";
        }

        if ($package_query->email == 1) {
            $rulesList["email"] = "nullable|email";
        }

        $validator = Validator::make($request->all(), $rulesList);

        if ($validator->fails()) {
            return back()->withInput()->withErrors($validator->getMessageBag()->toArray());
        }

        $link->title = $request->get('title');
        $link->url = $request->get('url');
        $link->description = $request->get('description');
        $link->phone_number = $request->get('phone_number');
        $link->email = $request->get('email');
        $link->visible_url = $request->get('visible_url');
        $link->address = $request->get('address');
        $link->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $link->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $link->bypass_type = $package_query->bypass_redirect_type;
        $link->bypass_details_type = $package_query->bypass_details_type;

        $deep_links = array(

            "link_1" => array(
                "title" => $request->get('deep_link_1_title'),
                "url" => $request->get('deep_link_1_url'),
                "type" => $package_query->deep_links_type,
            ),

            "link_2" => array(
                "title" => $request->get('deep_link_2_title'),
                "url" => $request->get('deep_link_2_url'),
                "type" => $package_query->deep_links_type,
            ),

            "link_3" => array(
                "title" => $request->get('deep_link_3_title'),
                "url" => $request->get('deep_link_3_url'),
                "type" => $package_query->deep_links_type,
            ),
        );

        $link->deep_links = serialize($deep_links);

        $other_contacts = array(

            "facebook_url" => array(
                "url" => $request->get('facebook_url'),
            ),

            "twitter_url" => array(
                "url" => $request->get('twitter_url'),
            ),

            "instagram_url" => array(
                "url" => $request->get('instagram_url'),
            ),

            "linkedin_url" => array(
                "url" => $request->get('linkedin_url'),
            ),

            "whatsapp_number" => array(
                "number" => $request->get('whatsapp_number'),
            ),

            "telegram_url" => array(
                "url" => $request->get('telegram_url'),
            ),
        );

        $link->other_contacts = serialize($other_contacts);

        // Check if the site icon is changed
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $file_name = time() . '.' . $image->getClientOriginalExtension();
            $location = public_path('images/links/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location);
            $old_image = public_path('images/links/' . $link->image);
            $link->image = $file_name;
            File::delete($old_image); // Remove old site icon
        }

        $link->save();
        $link->slug = null;
        $link->update(['title' => $link->title]);

        $link->categories()->sync((array) $request->input('categories'));

        // Redirect to link edit page
        return back()->with('success', __('general.link_updated'));
    }

    /** Link Submission Function */
    public function submit_link(Request $request)
    {
        // Check if the URL is blacklisted
        $blacklist = preg_split("/\r\n|\n|\r/", $this->domain_blacklist);
        foreach ($blacklist as $site_url) {
            if (parse_url(request()->url, PHP_URL_HOST) != '') {
                if (strpos(parse_url($site_url, PHP_URL_HOST), parse_url(request()->url, PHP_URL_HOST)) !== false) {
                    return back()->withInput()->withErrors(__('general.url_in_blacklist'));
                }
            }
        }

        // Check if phone number is valid
        if ($request->get('phone_number') != null) {
            try {
                $number = PhoneNumber::parse($request->get('phone_number'));
            } catch (PhoneNumberParseException $e) {
                return back()->withInput()->withErrors($e->getMessage());
            }
        }

        // Check if the link exists in the database
        $link_check = Link::where('url', $request->get('url'))->first();

        // Return error message if the link exists in the database
        if ($link_check != null) {
            return back()->withInput()->withErrors(__('general.link_already_in_database'));
        }

        // Check if the link is in approval process
        $submission_check = Submission::leftJoin('orders', 'submissions.id', '=', 'orders.submission_id')
            ->select('orders.package AS package', 'submissions.*')
            ->where('url', $request->get('url'))->first();

        // Return error message if the link is in approval process
        if ($submission_check != null) {
            return back()->withInput()->withErrors(__('general.link_awaiting_approval'));
        }

        $package_id = request()->package_id;

        $member = Auth::user();

        // Package query
        $package_query = Package::where('id', $package_id)->first();

        // Return 404 page if package not found
        $package_query == null ? abort(404) : '';

        for ($x = 1; $x <= $this->max_deep_links; $x++) {
            $arra['title'] = $request->get('deep_link_' . $x . '_title');
            $arra['url'] = $request->get('deep_link_' . $x . '_url');
            $arra['type'] = $request->get('deep_link_' . $x . '_type');
            $deep_links_array['link_' . $x] = $arra;
        }

        $deep_links = serialize($deep_links_array);

        $other_contacts = array(

            "facebook_url" => array(
                "url" => $request->get('facebook_url'),
            ),

            "twitter_url" => array(
                "url" => $request->get('twitter_url'),
            ),

            "instagram_url" => array(
                "url" => $request->get('instagram_url'),
            ),

            "linkedin_url" => array(
                "url" => $request->get('linkedin_url'),
            ),

            "whatsapp_number" => array(
                "number" => $request->get('whatsapp_number'),
            ),

            "telegram_url" => array(
                "url" => $request->get('telegram_url'),
            ),
        );

        $other_contacts = serialize($other_contacts);

        // List of categories
        $categories_list = Category::orderBy('title', 'ASC')->get();

        $category_slug = [];
        $category_name = [];

        foreach ($categories_list as $category) {
            $category_name[$category->id] = $category->title;
        }

        foreach ($this->packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        if ($package_query->price != '0.00' && request()->payment != '2') {
            $payment_verification_token = Str::random(60);
        } else {
            $payment_verification_token = '';
        }

        if (request()->payment == '2') {
            $bank_reference_code = 'W-' . mt_rand(1111111111, 9999999999);
        } else {
            $bank_reference_code = '';
        }

        // Run the validation rules on the inputs from the form
        $rulesList = [
            "title" => "required",
            "description" => "required|min:25|max:755",
            "url" => "required|url",
            'categories' => ['required', 'array', 'min:1'],
            'categories.*' => ['required', 'integer', 'exists:categories,id'],
            "deep_link_1_url" => "nullable|url",
            "deep_link_2_url" => "nullable|url",
            "deep_link_3_url" => "nullable|url",
        ];

        if ($package_query->reciprocal_link == 1) {
            $rulesList["reciprocal_link"] = "required|url";
        }

        if ($package_query->phone_number == 1) {
            $rulesList["phone_number"] = "nullable|max:30";
        }

        if ($this->optional_submission_icon == 1) {
            $rulesList["image"] = "nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:1024|dimensions:min_width=200,min_height=200";
        } else {
            $rulesList["image"] = "required|image|mimes:jpeg,png,jpg,gif,svg,webp|max:1024|dimensions:min_width=200,min_height=200";
        }

        if ($package_query->email == 1) {
            $rulesList["email"] = "nullable|email";
        }

        if ($package_query->price != '0.00') {
            $rulesList["payment"] = "required";
        }

        $validator = Validator::make($request->all(), $rulesList);

        if ($validator->fails()) {
            return back()->withInput()->withErrors($validator->getMessageBag()->toArray());
        }

        $image_file_name = null;

        // Upload Image
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '.' . $image->getClientOriginalExtension();
            $location = public_path('images/submissions/' . $filename);
            Image::make($image)->resize(200, 200)->save($location);
            $image_file_name = $filename;
        }

        $payment_method = $request->get('payment') ? request()->payment : 0;
        $bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $bypass_details = $request->get('bypass_details') ? 1 : 0;
        $bypass_type = $package_query->bypass_redirect_type;
        $bypass_details_type = $package_query->bypass_details_type;

        // Insert Submission Record
        $last_id = DB::table('submissions')->insertGetId(array(
            'title' => request()->title,
            'url' => request()->url,
            'reciprocal_link' => request()->reciprocal_link,
            'image' => $image_file_name,
            'description' => request()->description,
            'featured_category' => $package_query->featured_category,
            'featured_home' => $package_query->featured_home,
            'phone_number' => request()->phone_number,
            'email' => request()->email,
            'address' => request()->address,
            'visible_url' => request()->visible_url,
            'deep_links' => $deep_links,
            'other_contacts' => $other_contacts,
            'bypass_redirect' => $bypass_redirect,
            'bypass_type' => $bypass_type,
            'bypass_details' => $bypass_details,
            'bypass_details_type' => $bypass_details_type,
            'user' => $member->id,
            'ip' => $request->ip(),
            'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
        ));

        // Insert Category Records
        $categories = $request->input('categories');

        foreach ($categories as $value) {
            DB::table('category_submissions')->insert(
                [
                    'link_id' => $last_id,
                    'category_id' => $value,
                ]
            );
        }

        // Create Free Submission Order Record
        if ($package_query->price == '0.00') {
            $name = $member->name;
            $email = $member->email;
            $title = request()->title;
            $url = request()->url;
            $description = request()->description;
            $category = request()->category;
            $package = $package_id;
            $link_package = $package_name[$package];
            $ip_address = $request->ip();
            $admin_email = $this->admin_email;

            // Insert Notification Record
            insert_notification('2', $url);

            // Insert Order Record
            DB::table('orders')->insert(
                [
                    'submission_id' => $last_id,
                    'payment_method' => $payment_method,
                    'package' => $package_id,
                    'user' => $member->id,
                    'price' => $package_query->price,
                    'status' => '1',
                    'currency_symbol' => $this->payment_currency_symbol,
                    'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
                ]
            );

            // Notify site owner by email
            if ($this->submission_notification_mail == '1') {
                \Mail::to($admin_email)->send(new Submissionreceived($name, $email, $link_package, $this->mail_site_title, $url, $title, $description, $ip_address, $this->mail_from));
            }
        }

        // Create PayPal Payment - One-Time Payment
        if (request()->payment == '1') {
            $paypal_single_charge = paypal_single_charge(null, $package_query->price, $last_id, $package_id, $member->id, $this->language_prefix, $this->payment_currency, $this->payment_currency_symbol, $last_id, 0);

            if ($paypal_single_charge['0'] == 'success') {
                return redirect($paypal_single_charge[1]);
            } else {
                return back()->withErrors($paypal_single_charge[1])->withInput();
            }
        }

        // Create PayPal Payment - Subscription
        if (request()->payment == '5') {
            $paypal_subscription = paypal_subscription($package_query->price, $last_id, $package_id, $member->id, $this->language_prefix, $this->payment_currency, $this->payment_currency_symbol, $last_id, $package_query->duration, $package_query->title, $member->name, $member->email, $payment_method);

            if ($paypal_subscription['0'] == 'success') {
                return redirect($paypal_subscription[1]);
            } else {
                return back()->withErrors($paypal_subscription[1])->withInput();
            }
        }

        // Create Stripe Payment
        if (request()->payment == '3') {

            // Insert Order Record
            $last_order_id = DB::table('orders')->insertGetId(array(
                'submission_id' => $last_id,
                'payment_method' => $payment_method,
                'package' => $package_id,
                'user' => $member->id,
                'price' => $package_query->price,
                'currency_symbol' => $_ENV['CASHIER_CURRENCY_SYMBOL'],
                'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
            ));

            return Redirect::to($this->language_prefix . 'member-area/stripe-payment/' . $last_order_id);
        }

        // Create Coinbase Payment
        if (request()->payment == '4') {
            $coinbase_order = coinbase_order(null, $package_id, $this->mail_site_title, $package_query->price, $this->payment_currency, $last_id, $member->id, $this->payment_currency_symbol, 0);
            if ($coinbase_order['0'] == 'success') {
                return Redirect::to($coinbase_order['1']);
            }
        }

        if (request()->payment == null && $package_query->price == '0.00') {
            // Submission Durations
            $durations = submission_durations();

            foreach ($this->packages as $package) {
                $package_name[$package->id] = $package->title;
            }

            return back()->withSuccess(__('general.submission_thanks'));
        }

        // Create Bank Payment
        if (request()->payment == '2') {
            $create_bank_payment = create_bank_payment(null, $this->language_prefix, $last_id, $payment_method, $package_id, $member->id, $package_query->price, $bank_reference_code, $this->payment_currency_symbol, 0);

            if ($create_bank_payment['0'] == 'success' && $create_bank_payment['1'] == 'renewal') {
                return Redirect::to($create_bank_payment['2']);
            } else {
                return back()->with('bank_payment_details', $create_bank_payment[2])->withSuccess('bank_payment_details');
            }

        }

    }

    /** Renew Package Form */
    public function renew_package(Request $request)
    {
        $member = Auth::user();

        // Check if the link exists
        $link_query = DB::table('links')
            ->leftJoin('packages', 'links.package', '=', 'packages.id')
            ->select('packages.price AS price', 'links.*')
            ->where('links.id', request()->id)
            ->where('links.owner', $member->id)
            ->first();

        // Return 404 page if package not found
        $link_query == null ? abort(404) : '';

        // Meta tags
        meta_tags(__('general.payment_methods') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // Return view
        return view('' . $this->language_direction . 'frontend::member.renew')->with('member', $member)->with('link_query', $link_query);
    }

    /** Stripe Order Form */
    public function stripe_form(Request $request)
    {
        $member = Auth::user();

        // Check if the link exists
        $order_check = DB::table('orders')->where('id', request()->id)->where('payment_status', 0)->where('payment_method', 3)->where('user', $member->id)->first();

        // Return 404 page if submission not found
        $order_check == null ? abort(404) : '';

        // Meta tags
        meta_tags(__('general.payment_page') . ' › ' . $this->site_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), '600', '315', 'website');

        // Return view
        return view('' . $this->language_direction . 'frontend::member.stripe_form')->with('order_details', $order_check)->with('member', $member)->with('intent', $member->createSetupIntent());
    }

    /** Member Logout Function */
    public function dologout()
    {
        Auth::logout(); // log the user out of application
        return Redirect::to($this->language_prefix . 'member-area/login'); // redirect the user to the login screen
    }

}