<?php

use App\Mail\Bankpaymentnotification;
use App\Mail\Ordercomplete;
use App\Mail\Packagerenewed;
use App\Mail\Submissionreceived;
use App\Models\Bank;
use App\Models\Category;
use App\Models\Location;
use Carbon\Carbon;
use Spatie\SchemaOrg\Schema;
use Srmklive\PayPal\Services\PayPal as PayPalClient;

// Expire Time Calculation
function expire_time($duration, $time)
{
    $expire_time = array(
        '0' => null,
        '1' => date('Y-m-d H:i:s', strtotime("$time + 24 hours")),
        '2' => date('Y-m-d H:i:s', strtotime("$time + 7 days")),
        '3' => date('Y-m-d H:i:s', strtotime("$time + 1 month")),
        '4' => date('Y-m-d H:i:s', strtotime("$time + 3 months")),
        '5' => date('Y-m-d H:i:s', strtotime("$time + 6 months")),
        '6' => date('Y-m-d H:i:s', strtotime("$time + 9 months")),
        '7' => date('Y-m-d H:i:s', strtotime("$time + 1 year")),
    );
    return $expire_time[$duration];
}

// Submission Durations
function submission_durations()
{
    $durations = array(
        '0' => __('general.unlimited'),
        '1' => __('general.daily'),
        '2' => __('general.weekly'),
        '3' => __('general.monthly'),
        '4' => __('general.3_months'),
        '5' => __('general.6_months'),
        '6' => __('general.9_months'),
        '7' => __('general.yearly'),
    );

    return $durations;
}

/** Notifications Types */
function notification_types($type)
{
    $notification_types = array(
        '1' => __('admin.notification_new_member'),
        '2' => __('admin.notification_new_link'),
        '3' => __('admin.notification_payment'),
        '4' => __('admin.notification_package_renewed'),
        '5' => __('admin.notification_subscription_payment'),
    );

    return $notification_types[$type];
}

// Notifications
function notifications()
{
    return DB::table('notifications')->where('notification_read', '0')->orderBy('id', 'DESC')->get();
}

/** Stars Function */
function stars($votes, $rating_as_number, $type = '1', $direction = '2')
{
    if ($type == '1') {
        $editor_star = ' checked';
    } else {
        $editor_star = ' editor-star';
    }

    if ($rating_as_number == '1') {

        if (round($votes) == '0') {
            return '<span class="icon-star icon-small"></span>';
        } else {
            return '<span class="icon-star icon-small ' . $editor_star . '"></span>' . "<span class=\"number\">" . floatval($votes) . "</span>";
        }

    } else {

        $stars = null;

        if ( $direction == '1' ) {        
        for ($i = 5; $i >= 1; $i--) {
            if ($i <= round($votes)) {
                $stars .= '<span class="icon-star icon-small ' . $editor_star . '"></span>';
            } else {
                $stars .= '<span class="icon-star icon-small"></span>';
            }
        }   
        } else {
        for ($i = 1; $i <= 5; $i++) {
            if ($i <= round($votes)) {
                $stars .= '<span class="icon-star icon-small ' . $editor_star . '"></span>';
            } else {
                $stars .= '<span class="icon-star icon-small"></span>';
            }
        }            
        }

        return $stars;

    }

}

// Create Bank Payment
function create_bank_payment($id, $language_prefix, $last_id, $payment_method, $package_id, $member, $price, $bank_reference_code, $payment_currency_symbol, $is_renewal)
{
    if ($bank_reference_code == null) {
        $bank_reference_code = 'W-' . mt_rand(1111111111, 9999999999);
    }

    // Insert Order Record
    $last_order_id = DB::table('orders')->insertGetId(array(
        'link_id' => $id,
        'submission_id' => $last_id,
        'payment_method' => $payment_method,
        'package' => $package_id,
        'user' => $member,
        'is_renewal' => $is_renewal,
        'price' => $price,
        'bank_reference_code' => $bank_reference_code,
        'currency_symbol' => $payment_currency_symbol,
        'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
    ));

    if ($is_renewal == '1') {
        return ["success", "renewal", $language_prefix . 'member-area/payment-notification/' . $last_order_id];
    } else {
        // Insert Notification Record
        insert_notification('2', request()->url);
        return ["success", "new", $last_order_id];
    }

}

/** Stripe Package Renewal Function */
function stripe_renewal($id, $submission_id, $package, $user, $price)
{
    $last_order_id = DB::table('orders')->insertGetId(array(
        'link_id' => $id,
        'submission_id' => $submission_id,
        'payment_method' => 3,
        'is_renewal' => 1,
        'package' => $package,
        'user' => $user,
        'price' => $price,
        'currency_symbol' => $_ENV['CASHIER_CURRENCY_SYMBOL'],
        'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
    ));

    return ["success", $last_order_id];
}

/** Stripe Order Process Function */
function stripe_order($request, $payment_currency_symbol, $mail_site_title, $mail_from, $packages, $categories, $submission_notification_mail, $admin_email)
{
    $member = Auth::user();

    // Retrieve Order Details
    $order_query = DB::table('orders')
        ->leftJoin('users', 'orders.user', '=', 'users.id')
        ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
        ->leftJoin('packages', 'orders.package', '=', 'packages.id')
        ->select('users.name AS link_name', 'users.email AS link_email', 'orders.package AS package', 'orders.link_id AS link_id', 'orders.price AS price', 'orders.id AS order_id', 'packages.duration AS duration', 'orders.is_renewal AS is_renewal', 'submissions.*')
        ->where('orders.id', request()->id)->where('orders.payment_status', 0)->where('orders.payment_method', 3)->where('orders.user', $member->id)->first();

    // Return 404 page if order not found
    $order_query == null ? abort(404) : '';

    $amount = $order_query->price * 100;

    $user = auth()->user();
    $user->createOrGetStripeCustomer();
    $paymentMethod = $request->payment_method;

    $paymentMethod = $user->addPaymentMethod($paymentMethod);
    $charge = $user->charge($amount, $paymentMethod->id);

    $now = Carbon::now();

    // Package Durations
    $durations = submission_durations();

    if ($order_query->is_renewal == '1') {
        if ($request->payment_method != null) {
            $link_query = DB::table('links')->where('id', $order_query->link_id)->first();
            if ($now > $link_query->expire_at) {
                $new_expire_time = expire_time($order_query->duration, $now);
                DB::table('links')->where('id', $link_query->id)->update(['enabled' => '1', 'status' => '2']);
            } else {
                $new_expire_time = expire_time($order_query->duration, $link_query->expire_at);
            }
            DB::table('links')->where('id', $link_query->id)->update(['expire_at' => $new_expire_time]);
            DB::table('orders')->where('id', $order_query->order_id)->update(['payment_status' => '1']);
            DB::table('expire_notifications')->where('link_id', '=', $link_query->id)->delete();

            // Insert Notification Record
            insert_notification('3', $payment_currency_symbol . $order_query->price);
            insert_notification('4', $link_query->url);

            // Package Details
            $package = DB::table('packages')->where('id', $link_query->package)->first();

            // Notify link owner by email
            \Mail::to($order_query->link_email)->send(new Packagerenewed($durations[$order_query->duration], $payment_currency_symbol . $order_query->price, $package->title, $link_query->url, $mail_site_title, $mail_from));

            return ["success", "renewal", $order_query->package];

        }
    } else {

        foreach ($packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        $link_package = $package_name[$order_query->package];

        if ($request->payment_method != null) {
            DB::table('orders')->where('id', request()->id)->update(['payment_status' => '1', 'status' => '1', 'payment_id' => $charge->id]);

            // Insert Notification Record
            insert_notification('3', $payment_currency_symbol . $order_query->price);
            insert_notification('2', $order_query->url);

            // Package Details
            $package = DB::table('packages')->where('id', $order_query->package)->first();

            // Notify link owner by email
            \Mail::to($order_query->link_email)->send(new Ordercomplete($durations[$order_query->duration], $payment_currency_symbol . $order_query->price, $order_query->title, $order_query->url, $mail_site_title, $mail_from));

            // Notify site owner by email
            if ($submission_notification_mail == '1') {
                \Mail::to($admin_email)->send(new Submissionreceived($member->name, $member->email, $link_package, $mail_site_title, $order_query->url, $order_query->title, $order_query->description, $order_query->ip, $mail_from));
            }
            return ["success", "new", $order_query->package];

        } else {
            return ["error", $order_query->package];

        }
    }
}

// Create Coinbase Order
function coinbase_order($id, $package, $mail_site_title, $price, $payment_currency, $submission_id, $user, $payment_currency_symbol, $is_renewal)
{
    // Retrieve package details
    $package_query = DB::table('packages')->where('id', $package)->first();

    $charge = \Coinbase::createCharge([
        'name' => $package_query->title,
        'description' => $mail_site_title,
        'local_price' => [
            'amount' => $price,
            'currency' => $payment_currency,
        ],
        'pricing_type' => 'fixed_price',
    ]);

    // Insert Order Record
    $last_order_id = DB::table('orders')->insertGetId(array(
        'link_id' => $id,
        'submission_id' => $submission_id,
        'payment_id' => $charge['data']['code'],
        'payment_method' => 4,
        'is_renewal' => $is_renewal,
        'package' => $package,
        'user' => $user,
        'price' => $price,
        'currency_symbol' => $payment_currency_symbol,
        'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
    ));

    return ["success", $charge['data']['hosted_url']];
}

// Create PayPal Payment - Subscription
function paypal_subscription($price, $submission_id, $package_id, $user, $language_prefix, $payment_currency, $payment_currency_symbol, $last_id, $duration, $package_name, $member_name, $member_email, $payment_method)
{
    $payment_verification_token = Str::random(60);

    $provider = new PayPalClient;
    $provider = \PayPal::setProvider();
    $provider->getAccessToken();

    $success_url = asset($language_prefix . 'member-area/paypal-subscription?success_token=' . $payment_verification_token . '');
    $cancel_url = asset($language_prefix . 'member-area/my-orders');

    // Subscription Durations
    $subscription_duration = array
        (
        "1" => array("time_unit" => "DAY", "value" => "1"),
        "2" => array("time_unit" => "DAY", "value" => "7"),
        "3" => array("time_unit" => "MONTH", "value" => "1"),
        "4" => array("time_unit" => "MONTH", "value" => "3"),
        "5" => array("time_unit" => "MONTH", "value" => "6"),
        "6" => array("time_unit" => "MONTH", "value" => "9"),
        "7" => array("time_unit" => "YEAR", "value" => "1"),
    );

    $now = Carbon::now();
    $expire_time = expire_time($duration, $now);

    try {
        $order = $provider->addProduct($package_name, $package_name, 'SERVICE', 'SOFTWARE')
            ->addCustomPlan($package_name, $package_name, $price, $subscription_duration[$duration]['time_unit'], $subscription_duration[$duration]['value'])
            ->setReturnAndCancelUrl($success_url, $cancel_url)
            ->setupSubscription($member_name, $member_email, $expire_time);
    } catch (Throwable $e) {
        DB::table('submissions')->where('id', $last_id)->delete();
        return ["error", __('general.an_error_occurred')];
    }

    $ba_token = explode("ba_token=", $order['links']['0']['href']);

    if ($order['status'] == 'APPROVAL_PENDING') {
        // Insert Subscription Record
        DB::table('orders')->insert(
            [
                'submission_id' => $last_id,
                'subscription_id' => $order['id'],
                'subscription_token' => $ba_token[1],
                'payment_verification_token' => $payment_verification_token,
                'payment_method' => $payment_method,
                'package' => $package_id,
                'user' => $user,
                'price' => $price,
                'currency_symbol' => $payment_currency_symbol,
                'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
            ]
        );

        if ($_ENV['PAYPAL_MODE'] == 'live') {
            return ["success", "https://www.paypal.com/webapps/billing/subscriptions?ba_token=$ba_token[1]"];
        } else {
            return ["success", "https://www.sandbox.paypal.com/webapps/billing/subscriptions?ba_token=$ba_token[1]"];
        }

    }
}

/** PayPal Subscriptions Process Function */
function paypal_subscription_process($request, $language_prefix)
{
    $success_token = request()->success_token;
    $token = request()->subscription_id;

    $success_token == null || $token == null ? abort(404) : '';

    $order_query = DB::table('orders')
        ->leftJoin('users', 'orders.user', '=', 'users.id')
        ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
        ->select('users.name AS link_name', 'users.email AS link_email', 'orders.package AS package', 'orders.id AS order_id', 'submissions.*')
        ->where('orders.subscription_id', $token)->where('orders.payment_verification_token', $success_token)->where('orders.payment_status', 0)->first();

    // Return 404 page if order not found
    $order_query == null ? abort(404) : '';

    DB::table('orders')->where('submission_id', $order_query->id)->update(['payment_status' => '2']);

    return ["success", $language_prefix . 'member-area/package/' . $order_query->package . ''];
}

/** PayPal One Time Charge */
function paypal_single_charge($id, $price, $submission_id, $package, $user, $language_prefix, $payment_currency, $payment_currency_symbol, $last_id, $is_renewal)
{
    $payment_verification_token = Str::random(60);

    $provider = new PayPalClient;
    $provider = \PayPal::setProvider();
    $provider->getAccessToken();

    $data = json_decode('{
             "intent": "CAPTURE",
              "application_context": {
             "return_url": "' . asset($language_prefix . 'member-area/paypal-order?success_token=' . $payment_verification_token . '') . '",
              "cancel_url": "' . asset($language_prefix . 'member-area/my-orders') . '"
              },
              "purchase_units": [
              {
              "amount": {
              "currency_code": "' . $payment_currency . '",
               "value": "' . $price . '"
              }
              }
              ]
             }', true);

    $order = $provider->createOrder($data);

    if (isset($order['error'])) {
        if (isset($last_id)) {
            DB::table('submissions')->where('id', $last_id)->delete();
        }
        return ["error", $order['error']['message']];
    } else {
        if ($order['status'] == 'CREATED') {

            // Insert Order Record
            DB::table('orders')->insert(
                [
                    'link_id' => $id,
                    'submission_id' => $submission_id,
                    'payment_id' => $order['id'],
                    'payment_verification_token' => $payment_verification_token,
                    'payment_method' => 1,
                    'is_renewal' => $is_renewal,
                    'package' => $package,
                    'user' => $user,
                    'price' => $price,
                    'currency_symbol' => $payment_currency_symbol,
                    'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
                ]
            );

            if ($_ENV['PAYPAL_MODE'] == 'live') {
                return ["success", "https://www.paypal.com/checkoutnow?token=$order[id]"];
            } else {
                return ["success", "https://www.sandbox.paypal.com/checkoutnow?token=$order[id]"];
            }
        }
    }
}

// Schema.org Breadcrumbs
function schema_generator($site_title, $language_prefix, $data_name_1, $data_url_1, $data_name_2 = null, $data_url_2 = null, $data_name_3 = null, $data_url_3 = null)
{
    if ($data_name_2 == null && $data_name_3 == null) {
        $breadcrumb_schema_data = Schema::BreadcrumbList()
            ->itemListElement([
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(1)
                    ->name($site_title)
                    ->item(rtrim(asset($language_prefix), "/")),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(2)
                    ->name($data_name_1)
                    ->item($data_url_1),
            ]);

    }
    if ($data_name_2 != null) {
        $breadcrumb_schema_data = Schema::BreadcrumbList()
            ->itemListElement([
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(1)
                    ->name($site_title)
                    ->item(rtrim(asset($language_prefix), "/")),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(2)
                    ->name($data_name_1)
                    ->item(asset($language_prefix . $data_url_1)),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(3)
                    ->name($data_name_2)
                    ->item($data_url_2),
            ]);
    }

    if ($data_name_3 != null) {
        $breadcrumb_schema_data = Schema::BreadcrumbList()
            ->itemListElement([
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(1)
                    ->name($site_title)
                    ->item(rtrim(asset($language_prefix), "/")),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(2)
                    ->name($data_name_1)
                    ->item(asset($language_prefix . $data_url_1)),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(3)
                    ->name($data_name_2)
                    ->item(asset($language_prefix . $data_url_2)),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(4)
                    ->name($data_name_3)
                    ->item($data_url_3),
            ]);
    }

    return $breadcrumb_schema_data;
}

/** PayPal One Time Order Process Function */
function paypal_order($request, $language_prefix, $payment_currency_symbol, $admin_email, $packages, $categories, $mail_site_title, $mail_from, $submission_notification_mail)
{
    $success_token = request()->success_token;
    $token = request()->token;

    $success_token == null || $token == null ? abort(404) : '';

    $provider = new PayPalClient;
    $provider = \PayPal::setProvider();
    $provider->getAccessToken();
    $order = $provider->capturePaymentOrder($token);

    // Retrieve Order Details
    $order_query = DB::table('orders')
        ->leftJoin('users', 'orders.user', '=', 'users.id')
        ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
        ->leftJoin('packages', 'orders.package', '=', 'packages.id')
        ->select('users.name AS link_name', 'users.email AS link_email', 'orders.package AS package', 'orders.link_id AS link_id', 'orders.id AS order_id', 'packages.duration AS duration', 'orders.price AS price', 'orders.is_renewal AS is_renewal', 'submissions.*')
        ->where('orders.payment_id', $token)->where('orders.payment_verification_token', $success_token)->where('orders.payment_status', 0)->first();

    // Return 404 page if order not found
    $order_query == null ? abort(404) : '';

    $now = Carbon::now();

    // Package Durations
    $durations = submission_durations();

    if ($order_query->is_renewal == '1') {
        $link_query = DB::table('links')->where('id', $order_query->link_id)->first();
        if ($now > $link_query->expire_at) {
            $new_expire_time = expire_time($order_query->duration, $now);
            DB::table('links')->where('id', $link_query->id)->update(['enabled' => '1', 'status' => '2']);
        } else {
            $new_expire_time = expire_time($order_query->duration, $link_query->expire_at);
        }
        DB::table('links')->where('id', $link_query->id)->update(['expire_at' => $new_expire_time]);
        DB::table('orders')->where('payment_id', $token)->update(['payment_status' => '1']);
        DB::table('expire_notifications')->where('link_id', '=', $link_query->id)->delete();

        // Insert Notification Record
        insert_notification('3', $payment_currency_symbol . $order_query->price);
        insert_notification('4', $link_query->url);

        // Package Details
        $package = DB::table('packages')->where('id', $link_query->package)->first();

        // Notify link owner by email
        \Mail::to($order_query->link_email)->send(new Packagerenewed($durations[$order_query->duration], $payment_currency_symbol . $order_query->price, $package->title, $link_query->url, $mail_site_title, $mail_from));

        return ["success", $language_prefix . 'member-area/package/' . $order_query->package, __('general.package_extended')];

    } else {

        foreach ($packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        $name = $order_query->link_name;
        $email = $order_query->link_email;
        $title = $order_query->title;
        $url = $order_query->url;
        $description = $order_query->description;
        $package = $order_query->package;
        $link_package = $package_name[$package];
        $ip_address = $request->ip();

        // Insert Notification Record
        insert_notification('3', $payment_currency_symbol . $order_query->price);
        insert_notification('2', $order_query->url);

        // Package Details
        $package = DB::table('packages')->where('id', $order_query->package)->first();

        // Notify link owner by email
        \Mail::to($order_query->link_email)->send(new Ordercomplete($durations[$order_query->duration], $payment_currency_symbol . $order_query->price, $order_query->title, $order_query->url, $mail_site_title, $mail_from));

        // Notify site owner by email
        if ($submission_notification_mail == '1') {
            \Mail::to($admin_email)->send(new Submissionreceived($name, $email, $link_package, $mail_site_title, $url, $title, $description, $ip_address, $mail_from));
        }

        DB::table('orders')->where('payment_id', $token)->update(['payment_status' => 1, 'status' => 1]);

        return ["success", $language_prefix . 'member-area/package/' . $order_query->package, __('general.submission_thanks')];

    }

}

/** Send Payment Notification */
function send_payment_notification($request, $payment_currency_symbol, $admin_email, $packages, $categories, $mail_site_title, $mail_from, $submission_notification_mail)
{
    // Validate the info, create rules for the inputs
    $rules = array(
        'sender_name' => 'required|max:255',
        'bank_account' => 'required|integer',
        'details' => 'nullable|max:2000',
    );

    // Run the validation rules on the inputs from the form
    $validator = Validator::make($request->all(), $rules);

    if ($validator->fails()) {
        return back()->withInput()->withErrors($validator->getMessageBag()->toArray());
    }

    // Check if the order exists
    $order_query = DB::table('orders')
        ->leftJoin('users', 'orders.user', '=', 'users.id')
        ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
        ->leftJoin('packages', 'orders.package', '=', 'packages.id')
        ->select('users.name AS link_name', 'users.email AS link_email', 'orders.package AS package', 'orders.link_id AS link_id', 'orders.price AS price', 'orders.id AS order_id', 'packages.duration AS duration', 'orders.is_renewal AS is_renewal', 'orders.bank_reference_code AS bank_reference_code', 'submissions.*')
        ->where('orders.id', request()->id)
        ->where('orders.payment_method', 2)
        ->where('orders.payment_status', 0)
        ->first();

    // Return 404 page if order not found
    $order_query == null ? abort(404) : '';

    DB::table('orders')->where('id', request()->id)->update(['status' => 1, 'payment_status' => 1]);

    // Insert Payment Notification Record
    $last_id = DB::table('payment_notifications')->insertGetId(array(
        'order_id' => request()->id,
        'link_id' => $order_query->link_id,
        'sender_name' => request()->sender_name,
        'bank_account' => request()->bank_account,
        'details' => request()->additional_details,
        'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
    ));

    // List of bank accounts
    $banks = Bank::orderBy('sort', 'ASC')->get();

    foreach ($banks as $bank) {
        $bank_name[$bank->id] = $bank->bank_name;
    }

    $sender_name = request()->sender_name;
    $bank_account = $bank_name[request()->bank_account];
    $details = request()->additional_details;
    $amount = "$payment_currency_symbol $order_query->price";
    $reference_code = $order_query->bank_reference_code;
    if (isset($order_query->link_id)) {
        $url = asset(env('ADMIN_URL') . '/payment_notification/' . $last_id . '');
        // Notify site owner by email
        \Mail::to($admin_email)->send(new Bankpaymentnotification($sender_name, $bank_account, $details, $amount, $mail_site_title, $mail_from, $url, $reference_code));
    } else {
        foreach ($packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        $link_package = $package_name[$order_query->package];

        // Notify site owner by email
        if ($submission_notification_mail == '1') {
            \Mail::to($admin_email)->send(new Submissionreceived($order_query->link_name, $order_query->link_email, $link_package, $mail_site_title, $order_query->url, $order_query->title, $order_query->description, $order_query->ip, $mail_from));
        }
    }

    return true;
}

// Insert Notification Function
function insert_notification($action, $details)
{
    DB::table('notifications')->insert(
        [
            'action' => $action,
            'details' => $details,
            'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
        ]
    );
}

// Shortcode Function
function add_shortcode($ad_code, $ad_id)
{
    if ($_ENV['APP_PURIFIER'] == '1') {
        $result = Purifier::clean($ad_code, 'titles');
    } else {
        $result = $ad_code;
    }
    $result = preg_replace('/\[adsense ad-slot="([0-9]+)"]/', '
          <script async src="https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js?client=ca-pub-' . $ad_id . '"
               crossorigin="anonymous"></script>
          <ins class="adsbygoogle"
               style="display:block"
               data-ad-client="ca-pub-' . $ad_id . '"
               data-ad-slot="\1"
               data-ad-format="auto"
               data-full-width-responsive="true"></ins>
          <script>
               (adsbygoogle = window.adsbygoogle || []).push({});
          </script>
          ', $result);
    return $result;
}

// Google reCAPTCHA Function
function recaptcha_check($g_recaptcha_response, $google_recaptcha_secret_key)
{
    if (!isset($g_recaptcha_response) || empty($g_recaptcha_response)) {
        return "error";
    }

    // Google reCAPTCHA validation
    $secret = $google_recaptcha_secret_key;

    $client = new \GuzzleHttp\Client();
    $response = $client->request('POST', 'https://www.google.com/recaptcha/api/siteverify', [
        'form_params' => [
            'secret' => $secret,
            'response' => $g_recaptcha_response,
        ],
    ]);

    $response = $response->getBody();

    $responseData = json_decode($response, true);

    if ($responseData['success'] == false) {

        // If Google reCAPTCHA validation fails, return error response
        return "error";
    }
}

// MetaTag Function
function meta_tags($title, $description, $twitter_account, $image, $width, $height, $og_type)
{
    return MetaTag::setTags([
        'title' => $title,
        'description' => $description,
        'twitter_site' => $twitter_account,
        'twitter_title' => $title,
        'twitter_card' => 'summary',
        'twitter_url' => url()->current(),
        'twitter_description' => $description,
        'twitter_image' => $image,
        'og_title' => $title,
        'og_description' => $description,
        'og_url' => url()->current(),
        'og_image' => $image, 'og_image_width' => $width, 'og_image_height' => $height,
        'og_type' => $og_type,
    ]);
}

// Custom Title Function
function title_format($site_title, $title_format, $link_title = null, $category_title = null, $page_title = null, $article_title = null)
{
    $date = Carbon::now();

    $before = array(
        '%sep%', // Seperator
        '%site_name%', // Site Name
        '%category_title%', // Category Title
        '%link_title%', // Link Title
        '%page_title%', // Custom Page Title
        '%article_title%', // Custom Article Title
        '%year%', // Year
        '%month%', // Month
        '%day%', // Day
        '%month_text%', // Month (Full Textual Representation)
        '%day_text%', // Day (Full Textual Representation)
    );

    $after = array(
        "›",
        $site_title,
        $category_title,
        $link_title,
        $page_title,
        $article_title,
        date('Y'),
        date('m'),
        date('d'),
        \Carbon\Carbon::parse($date)->translatedFormat('F'),
        \Carbon\Carbon::parse($date)->translatedFormat('l'),
    );

    return str_replace($before, $after, $title_format);
}

// Link Types
function link_types()
{
    $link_types = array(
        '' => '', // Regular Link
        '0' => '', // Regular Link
        '1' => ' rel="sponsored"', //Sponsored
        '2' => ' rel="ugc"', // User Generated Content
        '3' => ' rel="nofollow"', // Nofollow
    );

    return $link_types;
}

// Link Type Values
function link_type_values()
{
    $link_type_values = array(
        '' => __('admin.regular_link'),
        '1' => __('admin.paid_sponsored'),
        '2' => __('admin.ugc'),
        '3' => __('admin.nofollow'),
    );

    return $link_type_values;
}

// Update number of links in categories
function count_categories()
{
    $main_array = [];

    // List of categories
    $categories = Category::get();

    foreach ($categories as $category) {

        array_push($main_array, $category->id);
        $sub_categories = Category::where('parent_id', $category->id)->get();

        foreach ($sub_categories as $sub_category) {
            $sum = DB::table('category_link')->where('category_id', $sub_category->id)->count(DB::raw('DISTINCT link_id'));
            DB::table('categories')->where('id', $sub_category->id)->update(['item_count' => $sum]);
            array_push($main_array, $sub_category->id);
        }

        if ( $category->parent_id == 0) {
        $sum = DB::table('category_link')->whereIn('category_id', $main_array)->count(DB::raw('DISTINCT link_id'));
       DB::table('categories')->where('id', $category->id)->update(['item_count' => $sum]);
        }

        $main_array = [];

    }

}

// Update number of links in locations
function count_locations()
{
    $main_array = [];

    // List of locations
    $locations = Location::get();

    foreach ($locations as $location) {

        array_push($main_array, $location->id);
        $sub_locations = Location::where('parent_id', $location->id)->get();

        foreach ($sub_locations as $sub_location) {
            $sum = DB::table('link_location')->where('location_id', $sub_location->id)->count(DB::raw('DISTINCT link_id'));
            DB::table('locations')->where('id', $sub_location->id)->update(['item_count' => $sum]);
            array_push($main_array, $sub_location->id);
        }

        if ( $location->parent_id == 0) {
        $sum = DB::table('link_location')->whereIn('location_id', $main_array)->count(DB::raw('DISTINCT link_id'));
       DB::table('locations')->where('id', $location->id)->update(['item_count' => $sum]);
        }

        $main_array = [];

    }

}
