<?php
namespace App\Http\Controllers;

use App;
use App\Models\Category;
use App\Models\Link;
use App\Models\Translation;
use Carbon\Carbon;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Intervention\Image\ImageManagerStatic as Image;
use Redirect;
use \Cviebrock\EloquentSluggable\Services\SlugService;

class CategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            } else {
                $settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        // List of categories
        $this->categories = Category::where('parent_id', '=', 0)->orderBy('language', 'ASC')->get();

        // List of languages
        $translations = Translation::orderBy('sort', 'ASC')->get();

        foreach ($translations as $language) {
            $language_id[$language->code] = $language->id;
            $language_code[$language->id] = $language->icon;
            if ($settings['root_language'] == '1' && $language->code == $settings['site_language']) {
                $language_prefix[$language->id] = '/';
            } else {
                $language_prefix[$language->id] = $language->code . '/';
            }
        }

        $this->site_language_id = $language_id[$settings['site_language']];

        // Array Data
        $array_data = array(
            'language_code' => $language_code,
            'language_prefix' => $language_prefix,
            'site_language_id' => $this->site_language_id ?? '',
        );

        // Pass data to views
        View::share(['settings' => $settings, 'categories' => $this->categories, 'translations' => $translations, 'array_data' => $array_data]);
    }

    /** Index */
    public function index(Request $request)
    {
        if ($request->has('sort')) {
            // List of pages
            $posts = Category::orderBy('sort', 'ASC')->get();

            $id = $request->input('id');
            $sorting = $request->input('sort');

            // Update sort order
            foreach ($posts as $item) {
                Category::where('id', '=', $id)->update(array(
                    'sort' => $sorting,
                ));
            }
            return \Response::json('success', 200);
        }

        // Return view
        return view('adminlte::categories.index')->with('main_category', '2');
    }

    /** Main Categories */
    public function main_categories(Request $request)
    {
        $id = request()->id;

        // List of categories
        $categories = Category::where('parent_id', 0)->where('language', $id)->orderBy('sort', 'ASC')->get();

        // Return view
        return view('adminlte::categories.index')->with('main_category', '1')->with('categories', $categories);
    }

    /** Subcategories */
    public function sub_categories()
    {
        $id = request()->id;

        foreach ($this->categories as $category) {
            $parent_name[$category->id] = $category->title;
            $parent_slug[$category->id] = $category->slug;
            $category_language[$category->id] = $category->language;
        }

        $data_to_pass = array(
            'parent_category_name' => $parent_name[$id],
            'parent_category_slug' => $parent_slug[$id],
            'category_language' => $category_language[$id],
        );

        // List of subcategories
        $categories = Category::where('parent_id', $id)->orderBy('sort', 'ASC')->get();

        // Return view
        return view('adminlte::categories.index')->with('categories', $categories)->with('data_to_pass', $data_to_pass)->with('main_category', '0');
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::categories.create');
    }

    /** Store */
    public function store(Request $request)
    {
        // Check if slug exists
        $slug_check = Category::where('slug', $request->get('slug'))->first();

        // Return error message if slug is in use
        if ($slug_check != null) {
            return Redirect::back()->withInput()->withErrors(__('admin.slug_in_use'));
        }

        $this->validate($request, [
            'title' => 'required|max:255',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
        ]);

        $category = new Category;

        // Check if the picture has been uploaded
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/categories/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location, $this->image_quality);
            $category->image = $file_name;
        }

        $category->slug = $request->get('slug');
        $category->title = $request->get('title');
        $category->custom_title = $request->get('custom_title');
        $category->custom_description = $request->get('custom_description');
        $category->custom_content = $request->get('custom_content');
        if ($category->custom_content == '<br>') {
            $category->custom_content = null;
        }
        $category->custom_content = str_replace('<p><br></p>', "", $category->custom_content);
        $category->custom_content = str_replace('<p></p>', "", $category->custom_content);
        $category->home_page = $request->get('home_page');
        $category->footer = $request->get('footer');
        $category->parent_id = $request->get('parent_id');
        $category->language = $request->get('language');
        $category->home_page = $request->get('home_page') ? 1 : 0;
        $category->footer = $request->get('footer') ? 1 : 0;

        if ($category->parent_id == 0) {
            $category->language = $request->get('language');
        } else {
            $category_language_id = DB::table('categories')->where('id', $request->get('parent_id'))->first();
            $category->language = $category_language_id->language;
        }

        // Retrieve the last item in sort order and add +1
        $category->sort = Category::max('sort') + 1;

        $category->save();

        if ($request->get('slug') == null) {
            $category->slug = null;
            $category->update([
                'title' => $category->title,
            ]);
        }

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/categories');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Redirect to category edit page
        return redirect()->route('categories.edit', $category->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve category details
        $category = Category::find($id);

        // Return 404 page if category not found
        if ($category == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::categories.edit', compact('category', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        // Check if slug exists
        $slug_check = Category::where('slug', $request->get('slug'))->where('id', '!=', $id)->first();

        // Return error message if slug is in use
        if ($slug_check != null) {
            return Redirect::back()->withErrors(__('admin.slug_in_use'));
        }

        $this->validate($request, [
            'title' => 'required|max:255',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
        ]);

        // Retrieve category details
        $category = Category::find($id);

        $category->slug = $request->get('slug');
        $category->title = $request->get('title');
        $category->custom_title = $request->get('custom_title');
        $category->custom_description = $request->get('custom_description');
        $category->custom_content = $request->get('custom_content');
        if ($category->custom_content == '<br>') {
            $category->custom_content = null;
        }
        $category->custom_content = str_replace('<p><br></p>', "", $category->custom_content);
        $category->custom_content = str_replace('<p></p>', "", $category->custom_content);
        $category->home_page = $request->get('home_page');
        $category->footer = $request->get('footer');
        $category->parent_id = $request->get('parent_id');
        $category->home_page = $request->get('home_page') ? 1 : 0;
        $category->footer = $request->get('footer') ? 1 : 0;

        // Ping Google
        if ($category->isDirty() && $this->ping_google == '1') {
            $sitemap_url = asset('sitemap/categories');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Check if the parent_id has been changed
        if ($category->isDirty('parent_id')) {

            // Get New Category Language ID
            $category_language_id = DB::table('categories')->where('id', $request->get('parent_id'))->first();

            // Check if the language of the new category is different
            if ($category_language_id->language != $category->language) {

                $category->language = $category_language_id->language;
                DB::table('links')->where('category', $id)->update(['language' => $category_language_id->language]);
            }
        }

        // Check if language has changed
        if ($request->has('language') && $request->get('parent_id') != '0') {
            // Get New Category Language ID
            $category_language_id = DB::table('categories')->where('id', $request->get('parent_id'))->first();

            DB::table('categories')->where('id', $id)->update(['language' => $category_language_id->language]);
            DB::table('links')->where('category', $id)->update(['language' => $category_language_id->language]);
        }

        // Check if the main category is assigned as a subcategory
        if ($request->has('language') && $request->get('parent_id') == '0') {

            $category->language = $request->get('language');

            if ($category->isDirty('language')) {

                $category_array = DB::table('categories')->where('parent_id', $id)->get();

                $sub_cats = [];

                foreach ($category_array as $row) {
                    $sub_cats[] = $row->id; // Push category id to array
                }

                DB::table('categories')->wherein('id', $sub_cats)->update(['language' => $request->get('language')]);
                DB::table('links')->wherein('category', $sub_cats)->update(['language' => $request->get('language')]);
                DB::table('links')->where('category', $id)->update(['language' => $request->get('language')]);

            }
        }

        // Check if the picture has been changed
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/categories/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location, $this->image_quality);
            $old_image = public_path('images/categories/' . $category->image);
            $category->image = $file_name;
            File::delete($old_image);
        }

        if ($request->get('slug') == null) {
            $category->slug = null;
            $category->update(['title' => $category->title]);
        }

        $category->save();

        // Update category link counts
        count_categories();

        // Redirect to category edit page
        return redirect()->route('categories.edit', $category->id)->with('success', __('admin.data_updated'));
    }

    /** Create Categories in Bulk */
    public function bulk(Request $request)
    {
        $method = $request->method();
        $language = request()->language;

        if ($request->isMethod('get')) {

            // Return view
            return view('adminlte::categories.create_bulk');

        } else {

            $explode_lines = preg_split('/\r\n|\r|\n/', $request->get('categories'));

            $i = 0;

            foreach ($explode_lines as $line) {

                $explode_line = explode("|", $line);

                if (($explode_line[1] ?? null) === '1') {
                    $home_page = 1;
                } else {
                    $home_page = 0;
                }

                if (($explode_line[2] ?? null) === '1') {
                    $footer = 1;
                } else {
                    $footer = 0;
                }

                $slug = SlugService::createSlug(Category::class, 'slug', $explode_line[0]);

                if ($request->has('sub_category')) {

                    $sub_category = request()->sub_category;

                    if ($explode_line[0] != null) {

                        $sort = Category::max('sort') + 1;

                        Category::insert([
                            'title' => $explode_line[0],
                            'slug' => $slug,
                            'parent_id' => $sub_category,
                            'item_count' => 0,
                            'sort' => $sort,
                            'home_page' => $home_page,
                            'footer' => $footer,
                            'language' => $language,
                            'created_at' => Carbon::now(),
                        ]);

                        $i++;
                    }
                } else {

                    if ($line != null) {

                        $sort = Category::max('sort') + 1;

                        Category::insert([
                            'title' => $explode_line[0],
                            'slug' => $slug,
                            'parent_id' => 0,
                            'item_count' => 0,
                            'sort' => $sort,
                            'home_page' => $home_page,
                            'footer' => $footer,
                            'language' => $language,
                            'created_at' => Carbon::now(),
                        ]);
                        $i++;
                    }
                }

            }

            if ($i >= '1') {
                return redirect()->back()->with('success', __('admin.bulk_categories_added', ['count' => $i]));
            } else {
                return Redirect::back()->withErrors(__('admin.no_category_to_add'));
            }

        }
    }

    /** Destroy */
    public function destroy($id)
    {
        // Retrieve category details
        $category = Category::find($id);

        // Check if links exist under the category
        $link_check = DB::table('category_link')->where('category_id', $id)->first();

        if ($link_check >= '1') {
            return Redirect::back()->withErrors(__('admin.links_exists_under_category'));
        }

        // Check if subcategories exist under the category
        $check_subcategories = DB::table('categories')->where('parent_id', $id)->count();

        if ($check_subcategories >= '1') {
            return Redirect::back()->withErrors(__('admin.subcategories_exists_under_category'));
        }

        $category->delete();

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/categories');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Redirect to list of categories
        return redirect()->back()->with('success', __('admin.data_deleted'));
    }

}
