<?php

namespace App\Http\Controllers;

use App;
use App\Models\Category;
use App\Models\Location;
use App\Models\Link;
use App\Models\Package;
use App\Models\Translation;
use App\Models\User;
use Brick\PhoneNumber\PhoneNumber;
use Brick\PhoneNumber\PhoneNumberParseException;
use Carbon\Carbon;
use DOMDocument;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Intervention\Image\ImageManagerStatic as Image;
use Redirect;
use \Cviebrock\EloquentSluggable\Services\SlugService;

class LinkController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // List of categories
        $categories = Category::where('parent_id', '=', 0)->orderBy('language', 'ASC')->get();

        foreach ($categories as $category) {
            $category_title[$category->id] = $category->title;
            foreach ($category->children as $submenu) {
                $category_title[$submenu->id] = $submenu->title;
            }
        }
        
        // List of locations
        $locations = Location::where('parent_id', '=', 0)->orderBy('language', 'ASC')->get();

        foreach ($locations as $location) {
            $location_title[$location->id] = $location->title;
            foreach ($location->children as $submenu) {
                $location_title[$submenu->id] = $submenu->title;
            }
        }

        // Total number of links
        $total_links = DB::table('links')->count('id');

        // Total number of members
        $total_members = DB::table('users')->where('user_type', '2')->count('id');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $this->settings[$setting->name] = $setting->value;
            } else {
                $this->settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        // Count of Pending Comments
        $pending_comments = DB::table('comments')->where('approval', '=', '0')->count('id');

        // Pending Submissions
        $pending_submissions = DB::table('orders')->where('status', 1)->where('is_renewal', 0)->count('id');

        // List of users
        $users = User::orderBy('name', 'ASC')->get();
        foreach ($users as $user) {
            $username[$user->id] = $user->name;
        }

        // List of languages
        $translations = Translation::orderBy('sort', 'ASC')->get();

        foreach ($translations as $language) {
            $language_id[$language->code] = $language->id;
            $language_code[$language->id] = strtoupper($language->code);
            $language_icon[$language->id] = $language->icon;
            if ($this->root_language == '1' && $language->code == $this->site_language) {
                $language_prefix[$language->id] = '/';
            } else {
                $language_prefix[$language->id] = $language->code . '/';
            }
        }

        // List of packages
        $packages = Package::orderBy('sort', 'ASC')->get();

        $package_name[0] = "-";
        foreach ($packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        // Array Data
        $array_data = array(
            'total_links' => $total_links,
            'total_members' => $total_members,
            'pending_comments' => $pending_comments,
            'pending_submissions' => $pending_submissions,
            'language_code' => $language_code,
            'language_icon' => $language_icon,
            'language_prefix' => $language_prefix,
            'category_title' => $category_title ?? '',
            'location_title' => $location_title ?? '',
            'username' => $username,
            'package_name' => $package_name,
        );

        // Pass data to views
        View::share(['categories' => $categories, 'locations' => $locations, 'array_data' => $array_data, 'settings' => $this->settings, 'link_types' => link_type_values(), 'users' => $users, 'packages' => $packages, 'translations' => $translations]);
    }

    /** Search */
    public function search(Request $request)
    {
        $searchquery = request()->post('term');
        $userquery = request()->post('user');
        $packagequery = request()->post('package');

        if ($userquery != null) {
            $searchquery = $userquery;
        }
        if ($packagequery != null) {
            $searchquery = $packagequery;
        }

        // Return 403 page if query is empty
        if ($searchquery == null) {
            abort(403);
        }

        // Search Results
        if ($userquery != null) {
            $links = Link::orderBy('id', 'desc')->where('owner', $userquery)->paginate(15);
        } elseif ($packagequery != null) {
            $links = Link::orderBy('id', 'desc')->where('package', $packagequery)->paginate(15);
        } else {
            $links = Link::orderBy('id', 'desc')->where('title', 'like', "%{$searchquery}%")->paginate(15);
        }

        // Return view
        return View::make('adminlte::links.index')->with('links', $links)->with('searchquery', $searchquery);
    }

    /** Index */
    public function index(Request $request)
    {
        // Retrieve target URL title and description
        if ($request->has('url')) {
            function curl_retrieve($url)
            {
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_HEADER, 0);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
                $agent = 'facebookexternalhit/1.1 (+http://www.facebook.com/externalhit_uatext.php)';
                curl_setopt($ch, CURLOPT_USERAGENT, $agent);
                $data = curl_exec($ch);
                curl_close($ch);
                return $data;
            }

            $url = $request->get('url');
            $html = curl_retrieve($url);

            $doc = new DOMDocument();
            @$doc->loadHTML($html);
            $nodes = $doc->getElementsByTagName('title');

            if (isset($nodes->item(0)->nodeValue)) {

                $title = $nodes->item(0)->nodeValue;

                $metas = $doc->getElementsByTagName('meta');

                for ($i = 0; $i < $metas->length; $i++) {
                    $meta = $metas->item($i);
                    if ($meta->getAttribute('name') == 'description') {
                        $description = $meta->getAttribute('content');
                    }
                }
            }

            $array_data = [];
            if (isset($title) || isset($description)) {
                if (isset($title)) {
                    array_push($array_data, array('title' => $title));
                }
                if (isset($description)) {
                    array_push($array_data, array('description' => $description));
                } else {
                    array_push($array_data, array('description' => null));
                }
                return json_encode($array_data);

            } else {
                return response()->json(['message' => 'error'], 500);
            }

        }

        // Generate Screenshot
        if ($request->has('screenshot')) {
            $id = $request->get('id');
            $as_icon = $request->get('as_icon') == 1 ? 1 : 0;

            // Retrieve link details
            $link = Link::find($id);

            // Return 404 page if link not found
            if ($link == null) {
                abort(404);
            }

            // Retrieve queue details
            $queue = DB::table('screenshot_queue')->where('link_id', $id)->where('as_icon', $as_icon)->first();

            // Return error message if link already in queue
            if ($queue != null) {
                return redirect()->route('links.edit', $id)->withErrors(__('admin.already_in_queue'));
            }

            $image_url = "https://image.thum.io/get/auth/$this->thum_io_id-$this->thum_io_url_key/$link->url";
            $ch = curl_init($image_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);

            // Insert Queue Record
            DB::table('screenshot_queue')->insert(
                [
                    'link_id' => $id,
                    'created_at' => date("Y-m-d H:i:s"),
                    'as_icon' => $as_icon,
                ]
            );

            return redirect()->route('links.edit', $id)->with('success', __('admin.added_to_queue'));
        }

        // List of latest links
        $links = Link::leftJoin('submissions', 'links.submission_id', '=', 'submissions.id')
            ->select('submissions.reciprocal_link AS reciprocal_link', 'links.*')
            ->orderBy('id', 'DESC')
            ->paginate(15);

        // Return view
        return view('adminlte::links.index', compact('links'));
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::links.create');
    }

    /** Store */
    public function store(Request $request)
    {
        // Check if slug exists
        $slug_check = Link::where('slug', $request->get('slug'))->first();

        // Return error message if slug is in use
        if ($slug_check != null) {
            return back()->withInput()->withErrors(__('admin.slug_in_use'));
        }

        // Check if phone number is valid
        if ($request->get('phone_number') != null) {
            try {
                $number = PhoneNumber::parse($request->get('phone_number'));
            } catch (PhoneNumberParseException $e) {
                return back()->withInput()->withErrors($e->getMessage());
            }
        }

        $this->validate($request, [
            'title' => 'required|max:100',
            'description' => 'required|max:755',
            'page_views' => 'required|numeric',
            'clicks' => 'required|numeric',
            'url' => 'required|url',
            'categories' => ['required', 'array', 'min:1'],
            'categories.*' => ['required', 'integer', 'exists:categories,id'],            
            'email' => 'nullable|email',
            'visible_url' => 'nullable',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
            'slug' => 'nullable|max:100',
            'votes' => 'required|numeric',
            'total_votes' => 'required|numeric',
        ]);

        $link = new Link;

        // Check if the site icon has been uploaded
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/links/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location, $this->image_quality);
            $link->image = $file_name;
        }

        // Check if the screenshot has been uploaded
        if ($request->hasFile('screenshot')) {
            $screenshot = $request->file('screenshot');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $screenshot->getClientOriginalExtension();
            }
            $location = public_path('images/screenshots/' . $file_name);
            Image::make($screenshot)->resize(600, 600)->save($location, $this->image_quality);
            $link->screenshot = $file_name;
        }

        $category_data = $request->input('categories');

        // Get Category Language ID
        $category_language_id = DB::table('categories')->where('id', $category_data[0])->first();

        $link->slug = $request->get('slug');
        $link->title = $request->get('title');
        $link->description = $request->get('description');
        $link->page_views = $request->get('page_views');
        $link->clicks = $request->get('clicks');
        $link->url = $request->get('url');
        $link->visible_url = $request->get('visible_url');
        $link->bypass_type = $request->get('bypass_type');
        $link->bypass_details_type = $request->get('bypass_details_type');
        $link->custom_title = $request->get('custom_title');
        $link->custom_description = $request->get('custom_description');
        $link->phone_number = $request->get('phone_number');
        $link->email = $request->get('email');
        $link->address = $request->get('address');
        $link->owner = $request->get('owner');
        $link->package = $request->get('package');
        $link->featured_category = $request->get('featured_category') ? 1 : 0;
        $link->featured_home = $request->get('featured_home') ? 1 : 0;
        $link->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $link->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $link->badge = $request->get('badge') ? 1 : 0;
        $link->enabled = $request->get('enabled') ? 1 : 0;
        $link->language = $category_language_id->language;
        $link->expire_at = $request->get('expire_at');
        $link->votes = $request->get('votes');
        $link->total_votes = $request->get('total_votes');
        $link->map_lon = $request->get('map_lon');
        $link->map_lat = $request->get('map_lat');
        $link->map_zoom = $request->get('map_zoom');
        $link->status = '2';
        $link->gallery = '';

        $link->hidden_content = $request->get('hidden_content');

        preg_match_all('/src="(data:image\/[^;]+;base64[^"]+)"/', $link->hidden_content, $result, PREG_PATTERN_ORDER);

        $total_images = count($result[0]);

        for ($x = 0; $x <= $total_images - 1; $x++) {
            $data = substr($result[0][$x], strpos($result[0][$x], ',') + 1);
            $data = base64_decode($data);
            if ($this->save_as_webp == '1') {
                $file_name = time() . rand(11111, 99999) . '.webp';
            } else {
                $file_name = time() . rand(11111, 99999) . '.png';
            }
            \File::put(public_path() . '/images/uploads/' . $file_name, $data);
            $imagefile = asset('images/uploads') . '/' . $file_name;
            $link->hidden_content = str_replace($result[0][$x], "src=\"$imagefile\" class=\"img-fluid\"", $link->hidden_content);
            $link->hidden_content = preg_replace('/(\<img[^>]+)(style\=\"[^\"]+\")([^>]+)(>)/', '${1}${3}${4}', $link->hidden_content);
        }

        if ($link->hidden_content == '<br>') {
            $link->hidden_content = null;
        }
        $link->hidden_content = str_replace('<p><br></p>', "", $link->hidden_content);
        $link->hidden_content = str_replace('<p></p>', "", $link->hidden_content);

        for ($x = 1; $x <= $this->max_deep_links; $x++) {
            $arra['title'] = $request->get('deep_link_' . $x . '_title');
            $arra['url'] = $request->get('deep_link_' . $x . '_url');
            $arra['type'] = $request->get('deep_link_' . $x . '_type');

            $deep_links['link_' . $x] = $arra;
        }

        $link->deep_links = serialize($deep_links);

        $other_contacts = array(

            "facebook_url" => array(
                "url" => $request->get('facebook_url'),
            ),

            "twitter_url" => array(
                "url" => $request->get('twitter_url'),
            ),

            "instagram_url" => array(
                "url" => $request->get('instagram_url'),
            ),

            "linkedin_url" => array(
                "url" => $request->get('linkedin_url'),
            ),

            "whatsapp_number" => array(
                "number" => $request->get('whatsapp_number'),
            ),

            "telegram_url" => array(
                "url" => $request->get('telegram_url'),
            ),
        );

        $link->other_contacts = serialize($other_contacts);

        $link->save();

        $link->categories()->sync((array) $request->input('categories'));
        $link->locations()->sync((array) $request->input('locations'));

        $tags = explode(",", $request->get('tags'));
        $link->tag($tags);

        if ($request->get('slug') == null) {
            $link->slug = null;
            $link->update(['title' => $link->title]);
        }

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/links');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Update category link counts
        count_categories();
        
        // Update location link counts
        count_locations();
        
        if ($this->save_screenshot_as_icon == '1') {
            $image_url = "https://image.thum.io/get/auth/$this->thum_io_id-$this->thum_io_url_key/$link->url";
            $ch = curl_init($image_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);

            // Insert Queue Record
            DB::table('screenshot_queue')->insert(
                [
                    'link_id' => $link->id,
                    'as_icon' => 1,
                    'created_at' => date("Y-m-d H:i:s"),
                ]
            );
        }

        // Redirect to link edit page
        return redirect()->route('links.edit', $link->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve link details
        $link = Link::find($id);

        // Return 404 page if link not found
        if ($link == null) {
            abort(404);
        }

          // Tags
          $item_count = count($link->tags);
          $i = 0;
          $tags = '';
          foreach ($link->tags as $key => $value) {
              if (++$i === $item_count) {
                  $tags .= $value->name;
              } else {
                  $tags .= "$value->name, ";
              }
          }

        // Check pending icon queue
        $icon_queue = DB::table('screenshot_queue')->where('link_id', $link->id)->where('as_icon', '1')->first();

        // Check pending screenshot queue
        $screenshot_queue = DB::table('screenshot_queue')->where('link_id', $link->id)->where('as_icon', '0')->first();

        $deep_link_data = unserialize($link->deep_links);

        if (!$deep_link_data) {
            for ($x = 1; $x <= $this->max_deep_links; $x++) {
                $arra['title'] = null;
                $arra['url'] = null;
                $arra['type'] = null;

                $deep_link_data['link_' . $x] = $arra;
            }
        }

        $other_contacts = unserialize($link->other_contacts);

        if (!$other_contacts) {
            $other_contacts = array(
                "facebook_url" => array("url" => null),
                "twitter_url" => array("url" => null),
                "instagram_url" => array("url" => null),
                "linkedin_url" => array("url" => null),
                "whatsapp_number" => array("number" => null),
                "telegram_url" => array("url" => null),
            );
        }

        // Return view
        return view('adminlte::links.edit')->with('deep_link_data', $deep_link_data)->with('other_contacts', $other_contacts)->with('link', $link)->with('icon_queue', $icon_queue)->with('screenshot_queue', $screenshot_queue)->with('tags', $tags);
    }

    /** Update */
    public function update(Request $request, $id)
    {
        // Check if slug exists
        $slug_check = Link::where('slug', $request->get('slug'))->where('id', '!=', $id)->first();

        // Return error message if slug is in use
        if ($slug_check != null) {
            return back()->withInput()->withErrors(__('admin.slug_in_use'));
        }

        // Check if phone number is valid
        if ($request->get('phone_number') != null) {
            try {
                $number = PhoneNumber::parse($request->get('phone_number'));
            } catch (PhoneNumberParseException $e) {
                return back()->withInput()->withErrors($e->getMessage());
            }
        }

        $this->validate($request, [
            'title' => 'required|max:100',
            'description' => 'required|max:755',
            'page_views' => 'required|numeric',
            'clicks' => 'required|numeric',
            'url' => 'required|url',
            'categories' => ['required', 'array', 'min:1'],
            'categories.*' => ['required', 'integer', 'exists:categories,id'],            
            'email' => 'nullable|email',
            'visible_url' => 'nullable',
            'custom_title' => 'nullable|max:255',
            'custom_description' => 'nullable|max:255',
            'slug' => 'nullable|max:100',
            'votes' => 'required|numeric|max:100',
            'total_votes' => 'required|numeric',
        ]);

        $category_data = $request->input('categories');

        // Retrieve link details
        $link = Link::find($id);

        $link_slug = $link->slug;

        $category_data = $request->input('categories');

        // Get Category Language ID
        $category_language_id = DB::table('categories')->where('id', $category_data[0])->first();

        $link->slug = $request->get('slug');
        $link->title = $request->get('title');
        $link->description = $request->get('description');
        $link->page_views = $request->get('page_views');
        $link->clicks = $request->get('clicks');
        $link->url = $request->get('url');
        $link->visible_url = $request->get('visible_url');
        $link->bypass_type = $request->get('bypass_type');
        $link->bypass_details_type = $request->get('bypass_details_type');
        $link->custom_title = $request->get('custom_title');
        $link->custom_description = $request->get('custom_description');
        $link->phone_number = $request->get('phone_number');
        $link->email = $request->get('email');
        $link->address = $request->get('address');
        $link->owner = $request->get('owner');
        $link->package = $request->get('package');
        $link->featured_category = $request->get('featured_category') ? 1 : 0;
        $link->featured_home = $request->get('featured_home') ? 1 : 0;
        $link->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $link->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $link->badge = $request->get('badge') ? 1 : 0;
        $link->enabled = $request->get('enabled') ? 1 : 0;
        $link->language = $category_language_id->language;
        $link->expire_at = $request->get('expire_at');
        $link->votes = $request->get('votes');
        $link->total_votes = $request->get('total_votes');
        $link->map_lon = $request->get('map_lon');
        $link->map_lat = $request->get('map_lat');
        $link->map_zoom = $request->get('map_zoom');
        $link->untag();
        $tags = explode(",", $request->get('tags'));
        $link->tag($tags);

        $link->hidden_content = $request->get('hidden_content');

        preg_match_all('/src="(data:image\/[^;]+;base64[^"]+)"/', $link->hidden_content, $result, PREG_PATTERN_ORDER);

        $total_images = count($result[0]);

        for ($x = 0; $x <= $total_images - 1; $x++) {
            $data = substr($result[0][$x], strpos($result[0][$x], ',') + 1);
            $data = base64_decode($data);
            if ($this->save_as_webp == '1') {
                $file_name = time() . rand(11111, 99999) . '.webp';
            } else {
                $file_name = time() . rand(11111, 99999) . '.png';
            }
            \File::put(public_path() . '/images/uploads/' . $file_name, $data);
            $imagefile = asset('images/uploads') . '/' . $file_name;
            $link->hidden_content = str_replace($result[0][$x], "src=\"$imagefile\" class=\"img-fluid\"", $link->hidden_content);
            $link->hidden_content = preg_replace('/(\<img[^>]+)(style\=\"[^\"]+\")([^>]+)(>)/', '${1}${3}${4}', $link->hidden_content);
        }

        if ($link->hidden_content == '<br>') {
            $link->hidden_content = null;
        }
        $link->hidden_content = str_replace('<p><br></p>', "", $link->hidden_content);
        $link->hidden_content = str_replace('<p></p>', "", $link->hidden_content);

        for ($x = 1; $x <= $this->max_deep_links; $x++) {
            $arra['title'] = $request->get('deep_link_' . $x . '_title');
            $arra['url'] = $request->get('deep_link_' . $x . '_url');
            $arra['type'] = $request->get('deep_link_' . $x . '_type');

            $deep_links['link_' . $x] = $arra;
        }

        $link->deep_links = serialize($deep_links);

        $other_contacts = array(

            "facebook_url" => array(
                "url" => $request->get('facebook_url'),
            ),

            "twitter_url" => array(
                "url" => $request->get('twitter_url'),
            ),

            "instagram_url" => array(
                "url" => $request->get('instagram_url'),
            ),

            "linkedin_url" => array(
                "url" => $request->get('linkedin_url'),
            ),

            "whatsapp_number" => array(
                "number" => $request->get('whatsapp_number'),
            ),

            "telegram_url" => array(
                "url" => $request->get('telegram_url'),
            ),
        );

        $link->other_contacts = serialize($other_contacts);

        // Check if the picture has been changed
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $image->getClientOriginalExtension();
            }
            $location = public_path('images/links/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location, $this->image_quality);
            $old_image = public_path('images/links/' . $link->image);
            $link->image = $file_name;
            File::delete($old_image);
        }

        // Check if the screenshot has been changed
        if ($request->hasFile('screenshot')) {
            $screenshot = $request->file('screenshot');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $screenshot->getClientOriginalExtension();
            }
            $location = public_path('images/screenshots/' . $file_name);
            Image::make($screenshot)->resize(600, 600)->save($location, $this->image_quality);
            $old_screenshot = public_path('images/screenshots/' . $link->screenshot);
            $link->screenshot = $file_name;
            if (!empty($link->screenshot)) {
                File::delete($old_screenshot);
            }
        }

        // Ping Google
        if ($link->isDirty() && $this->ping_google == '1') {
            $sitemap_url = asset('sitemap/links');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        if ($request->get('slug') == null) {
            $link->slug = null;
            $link->update(['title' => $link->title]);
        }

        if ($link->isDirty('slug')) {
            if (file_exists(public_path() . '/images/qr_codes/' . $link_slug . '.png')) {
                unlink(public_path() . '/images/qr_codes/' . $link_slug . '.png');
            }
        }

        if ($link->isDirty('enabled') && $link->enabled == '1') {
            DB::table('links')->where('id', $id)->update(['status' => '2']);
        }

        $link->save();
        
        $link->categories()->sync((array) $request->input('categories'));
        $link->locations()->sync($request->input('locations'));
        
        // Update category link counts
        count_categories();

        // Update location link counts
        count_locations();
        
        // Redirect to link edit page
        return redirect()->route('links.edit', $link->id)->with('success', __('admin.data_updated'));
    }

    /** Create Links in Bulk */
    public function bulk(Request $request)
    {
        $method = $request->method();
        $language = request()->language;

        if ($request->isMethod('get')) {

            // Return view
            return view('adminlte::links.create_bulk');

        } else {

            $failed_links = [];

            $explode_lines = preg_split('/\r\n|\r|\n/', $request->get('links'));

            $s = 0;
            $f = 0;

            foreach ($explode_lines as $line) {

                $title = null;
                $url = null;
                $category = null;
                $description = null;

                $explode_line = explode("|", $line);

                if (isset($explode_line[0])) {
                    $title = $explode_line[0];
                }

                if (isset($explode_line[1])) {
                    $url = $explode_line[1];
                }

                if (isset($explode_line[2])) {
                    $category = $explode_line[2];
                }

                if (isset($explode_line[3])) {
                    $description = $explode_line[3];
                }

                $slug = SlugService::createSlug(Link::class, 'slug', $explode_line[0]);
                
                $cat_language = Category::where('id', $explode_line[2])->first();

                if ($title != null && $url != null && $category != null && $description != null) {

                        $last_id = Link::insertGetId(array(
                        'title' => $title,
                        'description' => $description,
                        'slug' => $slug,
                        'url' => $url,
                        'bypass_redirect' => 0,
                        'bypass_details' => 0,
                        'page_views' => 0,
                        'clicks' => 0,
                        'featured_category' => 0,
                        'featured_home' => 0,
                        'votes' => '0.00',
                        'total_votes' => 0,
                        'badge' => 0,
                        'owner' => 1,
                        'package' => 0,
                        'gallery' => '',
                        'language' => $cat_language->language,
                        'enabled' => 1,
                        'status' => 2,
                        'is_subscription' => 0,
                        'created_at' => Carbon::now(),
                        ));
                        
                        // Insert Category Record
            DB::table('category_link')->insert(
                [
                    'link_id' => $last_id,
                    'category_id' => $explode_line[2],
                ]
            );

                    $s++;
                } else {

                    array_push($failed_links, $line);

                    $f++;

                }

            }

            $failed = '';

            foreach ($failed_links as $link) {
                if (next($failed_links)) {
                    $failed .= $link . "\n";
                } else {
                    $failed .= $link;
                }
            }

            if ($s >= '1') {

            // Update category link counts
            count_categories();

            // Update location link counts
            count_locations();
            
            }

            return view('adminlte::links.create_bulk')->with('success_count', $s)->with('fail_count', $f)->with('failed_links', $failed);

        }
    }

    /** Destroy */
    public function destroy($id)
    {
        // Retrieve link details
        $link = Link::find($id);

        if (!empty($link->image)) {
            if (file_exists(public_path() . '/images/links/' . $link->image)) {
                unlink(public_path() . '/images/links/' . $link->image);
            }
        }

        if (file_exists(public_path() . '/images/qr_codes/' . $link->slug . '.png')) {
            unlink(public_path() . '/images/qr_codes/' . $link->slug . '.png');
        }

        $link->delete();

        // Update category link counts
        count_categories();
        
        // Update location link counts
        count_locations();
        
        // Delete category records
        DB::table('category_link')->where('link_id', $id)->delete();
        
        // Delete location records
        DB::table('link_location')->where('link_id', $id)->delete();

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/links');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        // Redirect to list of links
        return redirect()->route('links.index')->with('success', __('admin.data_deleted'));
    }

}
