<?php
namespace App\Http\Controllers;

use App;
use App\Models\Link;
use App\Models\Member;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\View;
use App\Mail\Sendemail;
use Redirect;
use Mail;
use Purifier;

class MemberController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if($setting->language == null) {
            $settings[$setting->name] = $setting->value;
            }
        }
        
        // Get main language ID
        $site_language = DB::table('translations')->where('code', $this->site_language)->first();
        $language_id = $site_language->id;

         // Append main language variables
        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            if($setting->language == $language_id) {
            $this->$setting_name = $setting->value;
            $settings[$setting->name] = $setting->value;
            }
        }

        // Pass data to views
        View::share(['settings' => $settings]);
    }

    /** Index */
    public function index()
    {
        // List of members
        $members = Member::orderBy('id', 'DESC')->where('user_type', 2)->paginate(15);

        // Return view
        return view('adminlte::members.index', compact('members'));
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::members.create');
    }

    /** Store */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|max:255',
            'email' => 'unique:users,email|required|email',
            'password' => 'required|min:6|confirmed',
            'password_confirmation' => 'required|min:6',
        ]);

        $member = new Member;
        $member->name = $request->get('name');
        $member->email = $request->get('email');
        $member->password = Hash::make($request->password);
        $member->ip = $request->ip();
        $member->last_ip = $request->ip();
        $member->activated = $request->get('activated') ? 1 : 0;
        $member->disabled = $request->get('disabled') ? 1 : 0;

        $member->save();

        // Redirect to member edit page
        return redirect()->route('members.edit', $member->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve member details
        $member = Member::where('id', $id)->where('user_type', 2)->first();

        // Return 404 page if member not found
        if ($member == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::members.edit', compact('member', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'name' => 'required|max:255',
            'email' => "unique:users,email,$id|required|email",
            'new_password' => 'nullable|required_with:current_password,new_password_confirmation|min:6',
            'new_password_confirmation' => 'same:new_password',
        ]);

        // Retrieve member details
        $member = Member::find($id);
        $member->name = $request->get('name');
        $member->email = $request->get('email');
        $member->activated = $request->get('activated') ? 1 : 0;
        $member->disabled = $request->get('disabled') ? 1 : 0;

        $member->save();

        $new_password = $request->input('new_password');
        $hashed_password = Hash::make($new_password);

        if (!empty($new_password)) {
            DB::update("update users set password = '$hashed_password' WHERE id = $id");
        }

        // Redirect to member edit page
        return redirect()->route('members.edit', $member->id)->with('success', __('admin.data_updated'));
    }
    
    /** Send Email to Member */
    public function send_email(Request $request, $id)
    {
        // Retrieve member details
        $member = Member::where('id', $id)->where('user_type', 2)->first();

        // Return 404 page if member not found
        if ($member == null) {
            abort(404);
        }
        
        // If method is Post
        if ($request->isMethod('post')) {
            
             $this->validate($request, [
            'subject' => 'required|max:100',
            'message' => 'required',
        ]);

        $email_subject = $request->get('subject');
        $email_body = nl2br($request->get('message'));
        $email_body = Purifier::clean($email_body, 'titles');
        
        $site_title = $this->site_title;
        $mail_from = $_ENV['MAIL_FROM_ADDRESS'];
            
        $email = $member->email;
            
        Mail::to($email)->send(new Sendemail($email_subject, $email_body, $site_title, $mail_from));
            
        return Redirect::back()->with('success', __('admin.email_sent'));
        }

        // Return view
        return view('adminlte::members.send_email');
    }

    /** Destroy */
    public function destroy($id)
    {
        // Check if links added by this user
        $link_check = Link::where('owner', $id)->first();

        if ($link_check >= '1') {
            return Redirect::back()->withErrors(__('admin.links_exists_by_user'));
        }
        
        DB::table('submissions')->where('user', $id)->delete();

        // Retrieve member details
        $member = Member::where('id', $id)->where('user_type', 2)->first();

        $member->delete();

        // Redirect to list of member
        return redirect()->route('members.index')->with('success', __('admin.data_deleted'));
    }

}