<?php
namespace App\Http\Controllers;

use App;
use App\Models\Link;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Redirect;

class PackageController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            $settings[$setting->name] = $setting->value;
        }

        // Link Types
        $link_types = array(
            '0' => __('admin.regular_link'),
            '1' => __('admin.paid_sponsored'),
            '2' => __('admin.ugc'),
            '3' => __('admin.nofollow'),
        );

        // Pass data to views
        View::share(['settings' => $settings, 'link_types' => $link_types]);
    }

    /** Index */
    public function index(Request $request)
    {
        if ($request->has('sort')) {
            // List of pages
            $posts = Package::orderBy('sort', 'ASC')->get();

            $id = $request->input('id');
            $sorting = $request->input('sort');

            // Update sort order
            foreach ($posts as $item) {
                Package::where('id', '=', $id)->update(array(
                    'sort' => $sorting,
                ));
            }
            return \Response::json('success', 200);
        }
        // List of packages
        $packages = Package::orderBy('sort', 'ASC')->get();

        // Return view
        return view('adminlte::packages.index', compact('packages'));
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::packages.create');
    }

    /** Store */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'price' => 'required|numeric',
            'real_price' => 'nullable|numeric',
            'description' => 'nullable|max:755',
            'deep_links_count' => 'nullable|numeric|min:0|max:' . $this->max_deep_links . '',
        ]);

        $package = new Package;
        $package->title = $request->get('title');
        $package->description = $request->get('description');
        $package->price = $request->get('price');
        $package->real_price = $request->get('real_price');
        $package->deep_links_count = $request->get('deep_links_count');
        $package->deep_links_type = $request->get('deep_links_type');
        $package->bypass_redirect_type = $request->get('bypass_type');
        $package->bypass_details_type = $request->get('bypass_details_type');
        $package->available = $request->get('available') ? 1 : 0;
        $package->featured_category = $request->get('featured_category') ? 1 : 0;
        $package->featured_home = $request->get('featured_home') ? 1 : 0;
        $package->deep_links = $request->get('deep_links') ? 1 : 0;
        $package->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $package->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $package->visible_url = $request->get('visible_url') ? 1 : 0;
        $package->reciprocal_link = $request->get('reciprocal_link') ? 1 : 0;
        $package->phone_number = $request->get('phone_number') ? 1 : 0;
        $package->email = $request->get('email') ? 1 : 0;
        $package->address = $request->get('address') ? 1 : 0;
        $package->other_contacts = $request->get('other_contacts') ? 1 : 0;
        $package->duration = $request->get('duration');

        // Retrieve last item in sort order and add +1
        $package->sort = Package::max('sort') + 1;

        $package->save();

        // Redirect to package edit page
        return redirect()->route('packages.edit', $package->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve package details
        $package = Package::find($id);

        // Return 404 page if package not found
        if ($package == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::packages.edit', compact('package', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'price' => 'required|numeric',
            'real_price' => 'nullable|numeric',
            'description' => 'nullable|max:755',
            'deep_links_count' => 'nullable|numeric|min:0|max:' . $this->max_deep_links . '',
        ]);

        // Retrieve package details
        $package = Package::find($id);
        $package->title = $request->get('title');
        $package->description = $request->get('description');
        $package->price = $request->get('price');
        $package->real_price = $request->get('real_price');
        $package->deep_links_count = $request->get('deep_links_count');
        $package->deep_links_type = $request->get('deep_links_type');
        $package->bypass_redirect_type = $request->get('bypass_type');
        $package->bypass_details_type = $request->get('bypass_details_type');
        $package->available = $request->get('available') ? 1 : 0;
        $package->featured_category = $request->get('featured_category') ? 1 : 0;
        $package->featured_home = $request->get('featured_home') ? 1 : 0;
        $package->deep_links = $request->get('deep_links') ? 1 : 0;
        $package->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $package->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $package->visible_url = $request->get('visible_url') ? 1 : 0;
        $package->reciprocal_link = $request->get('reciprocal_link') ? 1 : 0;
        $package->phone_number = $request->get('phone_number') ? 1 : 0;
        $package->email = $request->get('email') ? 1 : 0;
        $package->address = $request->get('address') ? 1 : 0;
        $package->other_contacts = $request->get('other_contacts') ? 1 : 0;
        $package->duration = $request->get('duration');

        $package->save();

        // Redirect to package edit page
        return redirect()->route('packages.edit', $package->id)->with('success', __('admin.data_updated'));
    }

    /** Destroy */
    public function destroy($id)
    {
        // Check if the links are affiliated with the package
        $package_link_check = Link::where('package', $id)->count();

        // Check if the orders are affiliated with the package
        $package_submission_check = DB::table('orders')->where('package', $id)->count();

        $package_usage = $package_link_check + $package_submission_check;

        // Display error message if the package is affiliated with link or submissions
        if ($package_usage >= '1') {
            return Redirect::back()->withErrors(__('admin.package_in_use'));
        }

        // Retrieve package details
        $package = Package::find($id);

        $package->delete();

        // Redirect to list of packages
        return redirect()->route('packages.index')->with('success', __('admin.data_deleted'));
    }

}
