<?php
namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Package;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;
use Srmklive\PayPal\Services\PayPal as PayPalClient;

class PaymentController extends Controller
{
    public function __construct()
    {
        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            } else {
                $settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        // List of languages
        $languages = DB::table('translations')->OrderBy('sort', 'ASC')->get();

        $lang_codes = [];
        foreach ($languages as $language) {
            if ($language->code == $settings['site_language']) {
                $site_language_id = $language->id;
            }
            array_push($lang_codes, $language->code);
            $language_id[$language->code] = $language->id;
            $language_icon[$language->code] = $language->icon;
            if ($settings['root_language'] == '1' && $language->code == $settings['site_language']) {
                $menu_language_prefix[$language->id] = '/';
            } else {
                $menu_language_prefix[$language->id] = $language->code . '/';
            }
            $locale_tags[$language->id] = $language->locale_code;
        }

        if ($settings['root_language'] == '1') {
            if (in_array(request()->segment(1), $lang_codes) && request()->segment(1) != $settings['site_language']) {
                $language_code = request()->segment(1);
                $language_icon_code = $language_icon[request()->segment(1)];
            } else {
                $language_code = null;
                $language_icon_code = $language_icon[$settings['site_language']];
            }
        } else {
            $language_code = request()->segment(1);
            $language_icon_code = $language_icon[$language_code];
        }

        $this->language_prefix = $language_code . '/';

        // List of categories
        $this->categories = Category::get();

        // List of packages
        $this->packages = Package::OrderBy('sort', 'ASC')->where('available', 1)->get();

        $this->mail_site_title = $settings['site_title' . '-' . $site_language_id];
        $this->mail_from = $_ENV['MAIL_FROM_ADDRESS'];
        $this->admin_email = $settings['admin_email'];
    }

    /** Stripe Order Process Function */
    public function stripe_order(Request $request)
    {
        $stripe_order = stripe_order($request, $this->payment_currency_symbol, $this->mail_site_title, $this->mail_from, $this->packages, $this->categories, $this->submission_notification_mail, $this->admin_email);

        if ($stripe_order['0'] == 'success' && $stripe_order['1'] == 'renewal') {
            return Redirect::to($this->language_prefix . 'member-area/package/' . $stripe_order[2])->withSuccess(__('general.package_extended'));
        } elseif ($stripe_order['0'] == 'success' && $stripe_order['1'] == 'new') {
            return Redirect::to($this->language_prefix . 'member-area/package/' . $stripe_order[2])->withSuccess(__('general.submission_thanks'));
        } else {
            return Redirect::to($this->language_prefix . 'member-area/package/' . $stripe_order[2])->withErrors(__('general.an_error_occurred'));
        }
    }

    // PayPal Webhook
    public function paypal_webhook(Request $request)
    {
        // Return 404 page if webhook code is not correct
        $this->paypal_webhook_code != request()->slug ? abort(404) : '';

        $provider = new PayPalClient;
        $provider = \PayPal::setProvider();
        $provider->getAccessToken();
        $headers = $request->headers->all();
        $headers = array_change_key_case($headers, CASE_UPPER);

        if (!isset($headers['PAYPAL-AUTH-ALGO'][0]) || !isset($headers['PAYPAL-TRANSMISSION-ID'][0]) || !isset($headers['PAYPAL-CERT-URL'][0]) || !isset($headers['PAYPAL-TRANSMISSION-SIG'][0]) || !isset($headers['PAYPAL-TRANSMISSION-TIME'][0])
        ) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid headers',
            ]);
        }

        $responseBody = $request->all();

        $verify_data = [
            'auth_algo' => $headers['PAYPAL-AUTH-ALGO'][0],
            'cert_url' => $headers['PAYPAL-CERT-URL'][0],
            'transmission_id' => $headers['PAYPAL-TRANSMISSION-ID'][0],
            'transmission_sig' => $headers['PAYPAL-TRANSMISSION-SIG'][0],
            'transmission_time' => $headers['PAYPAL-TRANSMISSION-TIME'][0],
            'webhook_id' => env('PAYPAL_WEBHOOK_ID'),
            'webhook_event' => $responseBody,
        ];

        $response = $provider->verifyWebHook($verify_data);
        $save_data = json_encode($responseBody);

        // Insert Webhook Record
        DB::table('paypal_webhook_calls')->insert(
            [
                'payload' => $save_data,
                'created_at' => \Carbon\Carbon::now()->toDateTimeString(),
            ]
        );
    }

    /** Renew/Extend Packages */
    public function renewal(Request $request)
    {
        $id = request()->id;
        $payment_method = request()->method;

        $renewal_query = DB::table('links')
            ->leftJoin('packages', 'links.package', '=', 'packages.id')
            ->select('packages.price AS price', 'links.*', 'links.owner as user')
            ->where('links.id', $id)->first();

        // Return 404 page if order not found
        $renewal_query == null ? abort(404) : '';

        // Create PayPal Payment - One-Time Payment
        if ($payment_method == '1') {
            $paypal_single_charge = paypal_single_charge($id, $renewal_query->price, $renewal_query->submission_id, $renewal_query->package, $renewal_query->user, $this->language_prefix, $this->payment_currency, $this->payment_currency_symbol, null, 1);

            if ($paypal_single_charge['0'] == 'success') {
                return redirect($paypal_single_charge[1]);
            } else {
                return back()->withErrors($paypal_single_charge[1])->withInput();
            }
        }

        // Create Bank Payment
        if ($payment_method == '2') {
            $create_bank_payment = create_bank_payment($id, $this->language_prefix, $renewal_query->submission_id, 2, $renewal_query->package, $renewal_query->user, $renewal_query->price, null, $this->payment_currency_symbol, 1);

            if ($create_bank_payment['0'] == 'success' && $create_bank_payment['1'] == 'renewal') {
                return Redirect::to($create_bank_payment['2']);
            } else {
                return back()->with('bank_payment_details', $create_bank_payment[2])->withSuccess('bank_payment_details');
            }

        }

        // Create Coinbase Payment
        if ($payment_method == '4') {
            $coinbase_order = coinbase_order($id, $renewal_query->package, $this->mail_site_title, $renewal_query->price, $this->payment_currency, $renewal_query->submission_id, $renewal_query->user, $this->payment_currency_symbol, 1);
            if ($coinbase_order['0'] == 'success') {
                return Redirect::to($coinbase_order['1']);
            }
        }

        // Create Stripe Payment
        if ($payment_method == '3') {
            $stripe_renewal = stripe_renewal($id, $renewal_query->submission_id, $renewal_query->package, $renewal_query->user, $renewal_query->price);

            if ($stripe_renewal['0'] == 'success') {
                return Redirect::to($this->language_prefix . 'member-area/stripe-payment/' . $stripe_renewal[1]);
            }

        }
    }

    /** Paypal One Time Order Process Function */
    public function paypal_order(Request $request)
    {
        $paypal_order = paypal_order($request, $this->language_prefix, $this->payment_currency_symbol, $this->admin_email, $this->packages, $this->categories, $this->mail_site_title, $this->mail_from, $this->submission_notification_mail);

        if ($paypal_order['0'] == 'success') {
            return Redirect::to($paypal_order[1])->withSuccess($paypal_order[2]);
        }
    }

    /** PayPal Subscriptions Process Function */
    public function paypal_subscription_process(Request $request)
    {
        $paypal_subscription_process = paypal_subscription_process($request, $this->language_prefix);

        if ($paypal_subscription_process['0'] == 'success') {
            return Redirect::to($paypal_subscription_process[1])->withSuccess(__('general.submission_thanks'));
        }
    }

    /** Send Payment Notification */
    public function send_payment_notification(Request $request)
    {
        $send_payment_notification = send_payment_notification($request, $this->payment_currency_symbol, $this->admin_email, $this->packages, $this->categories, $this->mail_site_title, $this->mail_from, $this->submission_notification_mail);

        if ($send_payment_notification) {
            return back()->with('success', __('general.payment_notification_received'));
        }
    }

}
