<?php

namespace App\Http\Controllers;

use App\Models\Article;
use App\Models\Category;
use App\Models\Link;
use App\Models\Location;
use App\Models\Page;
use App\Models\Partner;
use Auth;
use Brick\PhoneNumber\PhoneNumber;
use Brick\PhoneNumber\PhoneNumberFormat;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use MetaTag;
use Redirect;
use Session;
use Spatie\SchemaOrg\Schema;

class SiteController extends Controller
{
    public function __construct()
    {
        // Site Settings
        $this->site_settings = DB::table('settings')->get();

        foreach ($this->site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            }
        }

        // Languages
        $this->languages = DB::table('translations')->OrderBy('sort', 'ASC')->get();

        foreach ($this->languages as $language) {
            $lang_codes[] = $language->code;
            $language_id[$language->code] = $language->id;
            $language_icon[$language->code] = $language->icon;
            $language_direction[$language->id] = $language->text_direction;
            if ($settings['root_language'] == '1' && $language->code == $settings['site_language']) {
                $menu_language_prefix[$language->id] = '/';
            } else {
                $menu_language_prefix[$language->id] = $language->code . '/';
            }
            $locale_tags[$language->id] = $language->locale_code;
        }

        // Check if site main language is running in home directory
        if ($settings['root_language'] == '1') {
            if (request()->segment(1) == $settings['site_language'] && request()->segment(2) == null) {
                Redirect::to(asset('/'), 301)->send();
            }
            if (request()->segment(1) == $settings['site_language']) {
                abort(404);
            }
            if (in_array(request()->segment(1), $lang_codes) && request()->segment(1) != $settings['site_language']) {
                $this->language_id = $language_id[request()->segment(1)];
                $language_code = request()->segment(1);
                $language_icon_code = $language_icon[request()->segment(1)];
            } else {
                $this->language_id = $language_id[$settings['site_language']];
                $language_code = null;
                $language_icon_code = $language_icon[$settings['site_language']];
            }
        } else {
            if (!in_array(request()->segment(1), $lang_codes) && request()->segment(1) == null) {
                Redirect::to($settings['site_language'], 301)->send();
            }
            if (!in_array(request()->segment(1), $lang_codes)) {
                abort(404);
            }
            $language_code = request()->segment(1);
            $this->language_id = $language_id[$language_code];
            $language_icon_code = $language_icon[$language_code];
        }

        $this->language_prefix = $language_code . '/';

        // List of Language Variables
        foreach ($this->site_settings as $setting) {
            $setting_name = $setting->name;
            if ($setting->language == $this->language_id) {
                $this->$setting_name = $setting->value;
                $settings[$setting->name] = $setting->value;
            }
        }

        // Ad Places
        $ad_places = DB::table('ads')->get();
        foreach ($ad_places as $ads) {
            $ad[$ads->id] = add_shortcode($ads->code, $settings['ga_publisher_id']);
        }

        // List of Pages
        $pages = Page::OrderBy('sort', 'ASC')->where('language', $this->language_id)->get();

        // List of Categories
        $this->categories = Category::where('language', $this->language_id)->orderBy('title', 'ASC')->get();

        // List of Locations
        $this->locations = Location::where('language', $this->language_id)->orderBy('title', 'ASC')->get();

        // List of Partners
        $partners = Partner::where('language', $this->language_id)->orderBy('sort', 'ASC')->get();

        // List of Latest Articles
        $articles = Article::orderBy('id', 'desc')->where('language', $this->language_id)->limit(4)->get();

        $category_slug = [];
        $category_name = [];

        foreach ($this->categories as $category) {
            $category_slug[$category->id] = $category->slug;
            $category_name[$category->id] = $category->title;
        }

        $location_slug = [];
        $location_name = [];

        foreach ($this->locations as $location) {
            $location_slug[$location->id] = $location->slug;
            $location_name[$location->id] = $location->title;
        }

        if ($language_direction[$this->language_id] == '1') {
            $this->language_direction = null;
        } else {
            $this->language_direction = 'rtl-';
        }

        $twitter_url = $this->twitter_account ? 'https://www.twitter.com/' . $this->twitter_account : '';
        $social_media = array($twitter_url, $this->facebook_page, $this->telegram_page);

        $localBusiness = Schema::Organization()
            ->name($this->site_title)
            ->email($this->admin_email)
            ->url(asset('/'))
            ->sameAs($social_media)
            ->logo(Schema::ImageObject()->url(asset('/'))->id(asset('/images/logo.png')));

        // Pass data to views
        View::share(['ad' => $ad, 'settings' => $settings, 'pages' => $pages, 'articles' => $articles, 'categories' => $this->categories, 'partners' => $partners, 'locale_tag' => $locale_tags[$this->language_id], 'category_slug' => $category_slug, 'category_name' => $category_name, 'location_slug' => $location_slug, 'location_name' => $location_name, 'link_types' => link_types(), 'languages' => $this->languages, 'language_prefix' => $this->language_prefix, 'language_icon_code' => $language_icon_code, 'menu_language_prefix' => $menu_language_prefix, 'localBusiness' => $localBusiness]);
    }

    /** Index */
    public function index()
    {
        // Default Home Page Title Format
        if ($this->home_page_title_format == null) {
            $this->home_page_title_format = '%site_name%';
        }

        $home_page_title = title_format($this->site_title, $this->home_page_title_format);

        // Meta tags
        meta_tags($home_page_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // List of featured links on home page
        if ($this->show_featured_links_randomly == '1') {
            $featured_links = DB::table('links')->where('featured_home', 1)->where('enabled', '1')->where('language', $this->language_id)->inRandomOrder()->limit(1)->get();
        } else {
            $featured_links = DB::table('links')->where('featured_home', 1)->where('enabled', '1')->where('language', $this->language_id)->orderBy('id', 'desc')->get();
        }

        // List of categories on the home page
        $categories_home = Category::where('parent_id', '=', 0)->where('home_page', '1')->where('language', $this->language_id)->orderBy('sort', 'ASC')->get();

        $latest_links = DB::table('links')->orderBy('id', 'desc')->where('enabled', '1')->where('language', $this->language_id)->limit(5)->get();

        $this->popular_links_sort_by == '1' ? $sort_by = 'page_views' : '';
        $this->popular_links_sort_by == '2' ? $sort_by = 'clicks' : '';
        $this->popular_links_sort_by == '3' ? $sort_by = 'votes' : '';

        $popular_links = DB::table('links')->orderBy($sort_by, 'desc')->where('enabled', '1')->where('language', $this->language_id)->limit(5)->get();

        // Latest reviews
        $latest_reviews = DB::table('comments')->leftJoin('links', 'comments.content_id', '=', 'links.id')
            ->select('comments.*', 'links.title as link_title', 'links.slug as slug', 'links.image as image')
            ->where('comments.type', '1')->where('comments.approval', 1)
            ->where('links.language', $this->language_id)
            ->orderBy('id', 'desc')
            ->limit(3)
            ->get();

        // Return view
        return view('' . $this->language_direction . 'frontend::main')->with('categories_home', $categories_home)->with('featured_links', $featured_links)->with('latest_links', $latest_links)->with('popular_links', $popular_links)->with('latest_reviews', $latest_reviews);
    }

    /** Categories */
    public function category(Request $request)
    {
        $slug = request()->slug;
        $parent_slug = request()->parent_slug;
        $page = request()->has('page') ? request()->get('page') : 1;

        foreach ($this->categories as $category) {
            $category_name[$category->id] = $category->title;
            $category_slug[$category->id] = $category->slug;
            $category_slug_id[$category->slug] = $category->id;
        }

        if (!array_key_exists($slug, $category_slug_id)) {
            abort(404);
        }

        if ($parent_slug != null) {
            if (!array_key_exists($parent_slug, $category_slug_id)) {
                abort(404);
            }
        }

        if ($parent_slug != null) {
            $parent_category_id = $category_slug_id[$parent_slug];
            $category_query = DB::table('categories')->where('parent_id', $parent_category_id)->where('language', $this->language_id)->where('slug', $slug)->first();
        } else {
            $category_query = DB::table('categories')->where('language', $this->language_id)->where('slug', $slug)->first();
        }

        if ($parent_slug == null && $category_query->parent_id != '0') {
            abort(404);
        }

        // Return 404 page if query is empty
        $category_query == null ? abort(404) : '';

        // List of subcategories of the category
        $sub_categories = Category::where('parent_id', $category_query->id)->where('language', $this->language_id)->orderBy('sort', 'ASC')->get();

        $sub_cats[] = $category_query->id;

        if ($sub_categories->isNotEmpty()) {
            foreach ($sub_categories as $row) {
                $sub_cats[] = $row->id; // Push category id to array
            }
        }

        // Allowed Query String Parameters
        $sort_array = array('Ascending', 'Descending', 'NewestFirst', 'OldestFirst', 'MostViews', 'LeastViews', 'MostClicks', 'LeastClicks', 'MostVotes', 'LeastVotes');

        if ($request->filled('sort')) {
            $filter_name = $request->sort;
        } else {
            $filter_name = $this->category_sort_by;
        }

        // Sort Filter
        if (!in_array($filter_name, $sort_array)) {
            abort(404);
        }

        switch ($filter_name) {
            case "NewestFirst":
                $order_type = 'links.id';
                $order_direction = 'desc';
                $current_filter = __('general.newest_first');
                break;
            case "OldestFirst":
                $order_type = 'links.id';
                $order_direction = 'asc';
                $current_filter = __('general.oldest_first');
                break;
            case "Ascending":
                $order_type = 'links.title';
                $order_direction = 'asc';
                $current_filter = __('general.title_ascending');
                break;
            case "Descending":
                $order_type = 'links.title';
                $order_direction = 'desc';
                $current_filter = __('general.title_descending');
                break;
            case "MostViews":
                $order_type = 'links.page_views';
                $order_direction = 'desc';
                $current_filter = __('general.most_page_views');
                break;
            case "LeastViews":
                $order_type = 'links.page_views';
                $order_direction = 'asc';
                $current_filter = __('general.least_page_views');
                break;
            case "MostClicks":
                $order_type = 'links.clicks';
                $order_direction = 'desc';
                $current_filter = __('general.most_clicks');
                break;
            case "LeastClicks":
                $order_type = 'links.clicks';
                $order_direction = 'asc';
                $current_filter = __('general.least_clicks');
                break;
            case "MostVotes":
                $order_type = 'links.votes';
                $order_direction = 'desc';
                $current_filter = __('general.most_votes');
                break;
            case "LeastVotes":
                $order_type = 'links.votes';
                $order_direction = 'asc';
                $current_filter = __('general.least_votes');
                break;
            default:
        }

        // List of links on category page
        if ($this->show_links_under_subcategories == '1') {
            $links = Link::whereHas('categories', function ($query) use ($sub_cats) {
                $query->whereIn('category_link.category_id', $sub_cats);
            })->where('language', $this->language_id)->orderBy('featured_category', 'desc')->orderBy($order_type, $order_direction)->paginate($this->links_per_page);
        } else {
            $links = Link::whereHas('categories', function ($query) use ($sub_cats, $category_query) {
                $query->where('category_link.category_id', $category_query->id);
            })->where('language', $this->language_id)->orderBy('featured_category', 'desc')->orderBy($order_type, $order_direction)->paginate($this->links_per_page);
        }

        /*
        Return 404 page if query is empty
        The system will not return 404 page on the first page, it will display a warning message
         */
        $links->isEmpty() && $page > '1' ? abort(404) : '';

        if ($category_query->parent_id == '0') {
            $data_to_pass = array(
                'category_name' => $category_name[$category_query->id],
                'category_slug' => $category_slug[$category_query->id],
            );
        } else {
            $data_to_pass = array(
                'category_name' => $category_name[$category_query->id],
                'category_slug' => $category_slug[$category_query->id],
                'parent_category_name' => $category_name[$category_query->parent_id],
                'parent_category_slug' => $category_slug[$category_query->parent_id],
            );
        }

        // Default Category Title Format
        if ($this->category_title_format == null) {
            $this->category_title_format = '%category_title% %sep% %site_name%';
        }

        $category_title = title_format($this->site_title, $this->category_title_format, null, $category_name[$category_query->id]);

        if ($category_query->custom_description != null) {
            $this->site_description = title_format($this->site_title, $category_query->custom_description, null, $category_name[$category_query->id]);
        }

        if ($category_query->custom_title != null) {
            $category_title = title_format($this->site_title, $category_query->custom_title, null, $category_name[$category_query->id]);
        }

        $share_image = $category_query->image;

        // Use default image if image is not uploaded
        empty($category_query->image) ? $share_image = 'no_image.png' : '';

        // Meta tags
        meta_tags($category_title, $this->site_description, $this->twitter_account, asset('images/categories/' . $share_image), 200, 200, 'website');

        // Schema.org Breadcrumbs
        if ($category_query->parent_id != 0) {
            $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, $data_to_pass['parent_category_name'], $this->category_base . '/' . $data_to_pass['parent_category_slug'], $category_query->title, url()->current());
        } else {
            $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, $category_query->title, url()->current());
        }

        // Return View
        return view('' . $this->language_direction . 'frontend::category')->with('links', $links)->with('category_query', $category_query)->with('sub_categories', $sub_categories)->with('data_to_pass', $data_to_pass)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('current_filter', $current_filter)->with('filter_name', $filter_name);
    }

    /** Locations */
    public function location(Request $request)
    {
        $slug = request()->slug;
        $parent_slug = request()->parent_slug;
        $page = request()->has('page') ? request()->get('page') : 1;

        foreach ($this->locations as $category) {
            $category_name[$category->id] = $category->title;
            $category_slug[$category->id] = $category->slug;
            $category_slug_id[$category->slug] = $category->id;
        }

        if (!array_key_exists($slug, $category_slug_id)) {
            abort(404);
        }

        if ($parent_slug != null) {
            if (!array_key_exists($parent_slug, $category_slug_id)) {
                abort(404);
            }
        }

        if ($parent_slug != null) {
            $parent_category_id = $category_slug_id[$parent_slug];
            $category_query = DB::table('locations')->where('parent_id', $parent_category_id)->where('language', $this->language_id)->where('slug', $slug)->first();
        } else {
            $category_query = DB::table('locations')->where('language', $this->language_id)->where('slug', $slug)->first();
        }

        if ($parent_slug == null && $category_query->parent_id != '0') {
            abort(404);
        }

        // Return 404 page if query is empty
        $category_query == null ? abort(404) : '';

        // List of subcategories of the category
        $sub_categories = Location::where('parent_id', $category_query->id)->where('language', $this->language_id)->orderBy('title', 'ASC')->get();

        $sub_cats[] = $category_query->id;

        if ($sub_categories->isNotEmpty()) {
            foreach ($sub_categories as $row) {
                $sub_cats[] = $row->id; // Push category id to array
            }
        }

        // List of links on location page
        if ($this->show_links_under_subcategories == '1') {
            $links = Link::whereHas('locations', function ($query) use ($sub_cats) {
                $query->whereIn('link_location.location_id', $sub_cats);
            })->where('language', $this->language_id)->orderBy('id', 'desc')->paginate($this->links_per_page);
        } else {
            $links = Link::whereHas('locations', function ($query) use ($sub_cats, $category_query) {
                $query->where('link_location.location_id', $category_query->id);
            })->where('language', $this->language_id)->orderBy('id', 'desc')->paginate($this->links_per_page);
        }

        /*
        Return 404 page if query is empty
        The system will not return 404 page on the first page, it will display a warning message
         */
        $links->isEmpty() && $page > '1' ? abort(404) : '';

        if ($category_query->parent_id == '0') {
            $data_to_pass = array(
                'category_name' => $category_name[$category_query->id],
                'category_slug' => $category_slug[$category_query->id],
            );
        } else {
            $data_to_pass = array(
                'category_name' => $category_name[$category_query->id],
                'category_slug' => $category_slug[$category_query->id],
                'parent_category_name' => $category_name[$category_query->parent_id],
                'parent_category_slug' => $category_slug[$category_query->parent_id],
            );
        }

        // Default Category Title Format
        if ($this->category_title_format == null) {
            $this->category_title_format = '%category_title% %sep% %site_name%';
        }

        $category_title = title_format($this->site_title, $this->category_title_format, null, $category_name[$category_query->id]);

        if ($category_query->custom_description != null) {
            $this->site_description = title_format($this->site_title, $category_query->custom_description, null, $category_name[$category_query->id]);
        }

        if ($category_query->custom_title != null) {
            $category_title = title_format($this->site_title, $category_query->custom_title, null, $category_name[$category_query->id]);
        }

        // Meta tags
        meta_tags($category_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // Schema.org Breadcrumbs
        if ($category_query->parent_id != 0) {
            $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, $data_to_pass['parent_category_name'], $this->location_base . '/' . $data_to_pass['parent_category_slug'], $category_query->title, url()->current());
        } else {
            $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, $category_query->title, url()->current());
        }

        // Return View
        return view('' . $this->language_direction . 'frontend::location')->with('links', $links)->with('category_query', $category_query)->with('sub_categories', $sub_categories)->with('data_to_pass', $data_to_pass)->with('breadcrumb_schema_data', $breadcrumb_schema_data);
    }

    /** Articles */
    public function article()
    {
        $slug = request()->slug;

        // Article query
        $article_query = DB::table('articles')->where('slug', $slug)->where('language', $this->language_id)->first();

        // Return 404 page if article not found
        $article_query == null ? abort(404) : '';

        // Custom Page Title Format
        if ($this->custom_article_title_format == null) {
            $this->custom_article_title_format = '%article_title% %sep% %site_name%';
        }

        $custom_page_title = title_format($this->site_title, $this->custom_article_title_format, null, null, null, $article_query->title);

        if ($article_query->custom_description != null) {
            $this->site_description = title_format($this->site_title, $article_query->custom_description, null, null, null, $article_query->title);
        }

        if ($article_query->custom_title != null) {
            $custom_page_title = title_format($this->site_title, $article_query->custom_title, null, null, null, $article_query->title);
        }

        // Meta tags
        meta_tags($custom_page_title, $this->site_description, $this->twitter_account, asset('images/articles/' . $article_query->image), 770, 450, 'article');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.articles'), $this->all_articles_slug, $article_query->title, url()->current());

        // If Enabled, Count Each Page View Only Once
        if ($this->page_view_once == '1') {
            $viewed = Session::get('viewed_articles', []);
            if (!in_array($article_query->id, $viewed)) {
                Session::push('viewed_articles', $article_query->id);
                DB::table('articles')->where('id', $article_query->id)->increment('page_views');
            }
        } else {
            DB::table('articles')->where('id', $article_query->id)->increment('page_views');
        }

        // List of comments
        $news_comments = DB::table('comments')->where('content_id', $article_query->id)->where('type', 2)->where('approval', 1)->orderBy('id', 'desc')->get();

        // Return view
        return view('' . $this->language_direction . 'frontend::article')->with('article_query', $article_query)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('news_comments', $news_comments);
    }

    /** All Articles */
    public function all_articles()
    {
        $page = request()->has('page') ? request()->get('page') : 1;

        // List of articles
        $all_articles = DB::table('articles')->orderBy('id', 'desc')->where('language', $this->language_id)->paginate($this->articles_per_page);

        /*
        Return 404 page if query is empty
        The system will not return 404 page on the first page, it will display a warning message
         */
        $all_articles->isEmpty() and $page > '1' ? abort(404) : '';

        // Default Articles Page Title Format
        if ($this->articles_page_title_format == null) {
            $this->articles_page_title_format = __('general.articles') . ' › %site_name%';
        }

        $articles_page_title_format = title_format($this->site_title, $this->articles_page_title_format);

        if ($this->articles_page_description != null) {
            $this->site_description = title_format($this->site_title, $this->articles_page_description);
        }

        // Meta tags
        meta_tags($articles_page_title_format, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.articles'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::articles')->with('all_articles', $all_articles)->with('breadcrumb_schema_data', $breadcrumb_schema_data);
    }

    /** Search */
    public function search()
    {
        $search_query = request()->post('term');

        $char_count = strlen($search_query);

        // Meta tags
        MetaTag::setTags([
            'title' => __('general.search') . " › $this->site_title",
            'robots' => 'noindex',
        ]);

        $links = collect(new Link);
        $category_search = collect(new Category);

        // Return a warning message if the search text is less than or equal to 2 characters
        if ($char_count <= 2) {
            return view('' . $this->language_direction . 'frontend::search')->withErrors(['msg', 'The Message'])->with('links', $links)->with('category_search', $category_search)->with('search_query', $search_query);
        }

        // Category search query
        $category_search = Category::where('title', 'like', "%{$search_query}%")->where('language', $this->language_id)->get();

        // Search query
        $links = DB::table('links')->orderBy('featured_category', 'desc')->orderBy('id', 'desc')->where('title', 'like', "%{$search_query}%")->where('enabled', '1')->where('language', $this->language_id)->limit($this->search_page_links_per_page)->get();

        // Return view
        return view('' . $this->language_direction . 'frontend::search')->with('links', $links)->with('category_search', $category_search)->with('search_query', $search_query);
    }

    /** Json Search */
    public function json_search(Request $request)
    {
        $request_type = $request->get('request');
        $list_id = $request->get('list_id');

        if ($request_type == '1') {
            $search_term = $request->get('search');

            $links = DB::table('links')->where('title', 'like', "%{$search_term}%")->where('enabled', '1')->where('language', $this->language_id)->orderBy('id', 'desc')->limit(5)->get();

            if ($links->isEmpty()) {
                $response = array();
            } else {
                foreach ($links as $row) {
                    if (empty($row->image)) {
                        $image_url = asset('/images/links/no_image.png');
                    } else {
                        $image_url = asset('/images/links/' . $row->image);
                    }

                    if ($row->bypass_details == '0') {
                        $site_url = asset($this->language_prefix . $this->detail_base) . '/' . $row->slug;
                    } else {
                        $site_url = $row->url;
                    }

                    if (!empty($row->visible_url)) {
                        $row->url = $row->visible_url;
                    }

                    $votes = stars($row->votes, 2, 1);

                    $response[] = array("value" => $row->id, "title" => $row->title, "site_url" => $site_url, "url" => $row->url, "image" => $image_url, "votes" => $votes);
                }
            }
            echo json_encode($response);
        }
    }

    /** Custom Pages */
    public function page()
    {
        $slug = request()->slug;

        // Page query
        $page_query = DB::table('pages')->where('slug', $slug)->where('language', $this->language_id)->first();

        // Return 404 page if page not found
        $page_query == null ? abort(404) : '';

        // Custom Page Title Format
        if ($this->custom_page_title_format == null) {
            $this->custom_page_title_format = '%page_title% %sep% %site_name%';
        }

        $custom_page_title = title_format($this->site_title, $this->custom_page_title_format, null, null, $page_query->title);

        if ($page_query->custom_description != null) {
            $this->site_description = title_format($this->site_title, $page_query->custom_description, null, null, $page_query->title);
        }

        if ($page_query->custom_title != null) {
            $custom_page_title = title_format($this->site_title, $page_query->custom_title, null, null, $page_query->title);
        }

        // Meta tags
        meta_tags($custom_page_title, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'article');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, $page_query->title, url()->current());

        // If Enabled, Count Each Page View Only Once
        if ($this->page_view_once == '1') {
            $viewed = Session::get('viewed_pages', []);
            if (!in_array($page_query->id, $viewed)) {
                Session::push('viewed_pages', $page_query->id);
                DB::table('pages')->where('id', $page_query->id)->increment('page_views');
            }
        } else {
            DB::table('pages')->where('id', $page_query->id)->increment('page_views');
        }

        // Return view
        return view('' . $this->language_direction . 'frontend::custom_page')->with('page_query', $page_query)->with('breadcrumb_schema_data', $breadcrumb_schema_data);
    }

    /** Latest Links */
    public function latest_links()
    {
        // Default Latest Links Page Title Format
        if ($this->latest_links_page_title_format == null) {
            $this->latest_links_page_title_format = __('general.latest_links') . ' › %site_name%';
        }

        $latest_links_page_title_format = title_format($this->site_title, $this->latest_links_page_title_format);

        if ($this->latest_links_page_description != null) {
            $this->site_description = title_format($this->site_title, $this->latest_links_page_description);
        }

        // List of latest links
        $links = DB::table('links')->orderBy('id', 'desc')->where('enabled', '1')->where('language', $this->language_id)->limit($this->links_per_page)->get();

        // Meta tags
        meta_tags($latest_links_page_title_format, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.latest_links'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::more_pages')->with('links', $links)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('page_type', '1');
    }

    /** Popular Links */
    public function popular_links()
    {
        // Default Popular Links Page Title Format
        if ($this->popular_links_page_title_format == null) {
            $this->popular_links_page_title_format = __('general.popular_links') . ' › %site_name%';
        }

        $popular_links_page_title_format = title_format($this->site_title, $this->popular_links_page_title_format);

        if ($this->popular_links_page_description != null) {
            $this->site_description = title_format($this->site_title, $this->popular_links_page_description);
        }

        $this->popular_links_sort_by == '1' ? $sort_by = 'page_views' : '';
        $this->popular_links_sort_by == '2' ? $sort_by = 'clicks' : '';
        $this->popular_links_sort_by == '3' ? $sort_by = 'votes' : '';

        // List of popular links
        $links = DB::table('links')->orderBy($sort_by, 'desc')->where('enabled', '1')->where('language', $this->language_id)->limit($this->links_per_page)->get();

        // Meta tags
        meta_tags($popular_links_page_title_format, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.popular_links'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::more_pages')->with('links', $links)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('page_type', '2');
    }

    /** Link Detail Page */
    public function detail(Request $request)
    {
        $slug = request()->slug;

        // Check if the link exists
        $link_query = Link::where('slug', $slug)->where('bypass_details', '0')->where('enabled', '1')->where('language', $this->language_id)->first();

        // Return 404 page if link not found
        $link_query == null ? abort(404) : '';

        // List of comments
        $link_reviews = DB::table('comments')->where('content_id', $link_query->id)->where('approval', 1)->orderBy('id', 'desc')->get();

        // Count comment
        $comment_data = DB::table('comments')->select('rating')->where('content_id', $link_query->id)->where('approval', 1)->selectRaw('count(rating) AS total_rating')->groupBy('rating')->orderBy('rating', 'desc')->get();

        // Format ratings
        $comment_order = array();
        foreach ($comment_data as $comment_c) {
            $comment_order[$comment_c->rating] = $comment_c->total_rating;
        }

        for ($x = 1; $x <= 5; $x++) {
            if (!array_key_exists("$x", $comment_order)) {
                $comment_order[$x] = 0;
            }
        }
        krsort($comment_order);

        // Total reviews

        $data_to_pass = [];
        $data_to_pass['total_reviews'] = array_sum($comment_order);

        $category_name = [];
        $category_slug = [];
        foreach ($this->categories as $category) {
            $category_name[$category->id] = $category->title;
            $category_slug[$category->id] = $category->slug;
        }

        // Category List for Custom Meta Tags
        $meta_categories = '';
        foreach ($link_query->categories as $key => $row) {
            if ($key != 0) {$meta_categories .= ', ';}
            $meta_categories .= $row->title;
        }

        // Default Link Title Format
        if ($this->link_title_format == null) {
            $this->link_title_format = '%link_title% %sep% %site_name%';
        }

        $link_title = title_format($this->site_title, $this->link_title_format, $link_query->title, $meta_categories);
        $link_description = $link_query->description;

        if ($link_query->custom_description != null) {
            $link_description = title_format($this->site_title, $link_query->custom_description, null, $meta_categories);
        }

        if ($link_query->custom_title != null) {
            $link_title = title_format($this->site_title, $link_query->custom_title, null, $meta_categories);
        }

        $share_image = $link_query->image;

        // Use default image if image is not uploaded
        empty($link_query->image) ? $share_image = 'no_image.png' : '';

        // Meta tags
        meta_tags($link_title, $link_description, $this->twitter_account, asset('images/links/' . $share_image), 200, 200, 'article');

        // If Enabled, Count Each Page View Only Once
        if ($this->page_view_once == '1') {
            $viewed = Session::get('viewed_links', []);
            if (!in_array($link_query->id, $viewed)) {
                Session::push('viewed_links', $link_query->id);
                DB::table('links')->where('id', $link_query->id)->increment('page_views');
            }
        } else {
            DB::table('links')->where('id', $link_query->id)->increment('page_views');
            $link_query->page_views = $link_query->page_views + 1;
        }

        $deep_link_data = unserialize($link_query->deep_links);

        if (!$deep_link_data) {
            $deep_link_data = array(
                "link_1" => array("title" => null, "url" => null, "type" => null),
                "link_2" => array("title" => null, "url" => null, "type" => null),
                "link_3" => array("title" => null, "url" => null, "type" => null),
            );
        }

        $other_contacts = unserialize($link_query->other_contacts);

        if (!$other_contacts) {
            $other_contacts = array(
                "facebook_url" => array("url" => null),
                "twitter_url" => array("url" => null),
                "instagram_url" => array("url" => null),
                "linkedin_url" => array("url" => null),
                "whatsapp_number" => array("number" => null),
                "telegram_url" => array("url" => null),
            );
        }

        // List of Report Reasons
        $report_reasons = array(
            __('general.link_not_working'),
            __('general.domain_expired'),
            __('general.redirecting_another_site'),
        );

        $category_list = $link_query->categories;

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = Schema::BreadcrumbList()
            ->itemListElement([
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(1)
                    ->name($this->site_title)
                    ->item(asset($this->language_prefix)),
                array_map(function ($category_list) {
                    return Schema::ListItem()
                        ->position(2)
                        ->name($category_list->title)
                        ->item(url($this->language_prefix . $this->category_base . '/' . $category_list->slug));
                }, iterator_to_array($category_list)),
                \Spatie\SchemaOrg\Schema::ListItem()
                    ->position(3)
                    ->name($link_query->title)
                    ->item(url()->current()),
            ]);

        // Schema.org Comments
        $schema_comments = Schema::Organization()
            ->name($link_query->title)
            ->review(
                array_map(function ($link_reviews) {
                    return Schema::Review()
                        ->author(['name' => $link_reviews->name, 'type' => 'Person'])
                        ->reviewRating(['ratingValue' => $link_reviews->rating, 'type' => 'Rating'])
                        ->reviewBody($link_reviews->comment)
                        ->datePublished(\Carbon\Carbon::parse($link_reviews->created_at)->translatedFormat('F d, Y'));
                }, iterator_to_array($link_reviews))
            );

        if ($link_query->total_votes != '0') {
            $schema_comments->aggregateRating(Schema::aggregateRating()
                    ->ratingValue($link_query->votes)
                    ->ratingCount($link_query->total_votes)
            );
        }

        // Format Phone Number
        if ($link_query->phone_number != null) {
            $phone_number = PhoneNumber::parse($link_query->phone_number);
            $data_to_pass['phone_country_code'] = $phone_number->getRegionCode();
            $data_to_pass['phone_number_callable'] = $phone_number->format(PhoneNumberFormat::INTERNATIONAL);
            $data_to_pass['phone_number_visible'] = $link_query->phone_number;
        }

        // User History
        if ($this->allow_user_history == 1) {
            if (Auth::check()) {
                $uh_check = DB::table('user_histories')->where('link_id', $link_query->id)->count();
                if ($uh_check == '0') {
                    DB::table('user_histories')->insert(
                        [
                            'user_id' => Auth::user()->id,
                            'link_id' => $link_query->id,
                        ]
                    );
                } else {
                    DB::table('user_histories')->where('link_id', $link_query->id)->where('user_id', Auth::user()->id)->delete();
                    DB::table('user_histories')->insert(
                        [
                            'user_id' => Auth::user()->id,
                            'link_id' => $link_query->id,
                        ]
                    );

                }
            }
        }

        $favorite_check = '';

        // User History
        if ($this->allow_user_favorites == 1) {

            if ($request->has('add')) {
                if (!Auth::check()) {
                    return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.login_to_perform_action'));
                }

                DB::table('user_favorites')->insert(
                    [
                        'user_id' => Auth::user()->id,
                        'link_id' => $link_query->id,
                    ]
                );
                Session::flash('message', __('general.add_message'));
                return redirect()->back();
            }

            if ($request->has('remove')) {
                if (!Auth::check()) {
                    return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.login_to_perform_action'));
                }

                DB::table('user_favorites')->where('link_id', $link_query->id)->where('user_id', Auth::user()->id)->delete();
                Session::flash('message', __('general.remove_message'));
                return redirect()->back();
            }

            if (Auth::check()) {
                $favorite_check = DB::table('user_favorites')->where('user_id', Auth::user()->id)->where('link_id', $link_query->id)->count();
            }
        }

        $mysplit = explode(',', $link_query->gallery);
        $gallery_data = array_reverse($mysplit);

        // Return View
        return view('' . $this->language_direction . 'frontend::detail')->with('link_query', $link_query)->with('data_to_pass', $data_to_pass)->with('deep_link_data', $deep_link_data)->with('other_contacts', $other_contacts)->with('report_reasons', $report_reasons)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('comment_order', $comment_order)->with('category_name', $category_name)->with('category_slug', $category_slug)->with('link_reviews', $link_reviews)->with('favorite_check', $favorite_check)->with('schema_comments', $schema_comments)->with('gallery_data', $gallery_data);
    }

    /** Redirection Page */
    public function redirect()
    {
        $slug = request()->slug;

        // Check if the link exists
        $link_query = Link::where('slug', $slug)->first();

        // Return 404 page if link not found
        $link_query == null ? abort(404) : '';

        // Return 404 page if redirection bypass is enabled
        $link_query->bypass_redirect == '1' ? abort(404) : '';

        // Default Link Title Format
        if ($this->redirection_title_format == null) {
            $this->redirection_title_format = '%link_title% %sep% %site_name%';
        }

        // Category List for Custom Meta Tags
        $meta_categories = '';
        foreach ($link_query->categories as $key => $row) {
            if ($key != 0) {$meta_categories .= ', ';}
            $meta_categories .= $row->title;
        }

        $link_title = title_format($this->site_title, $this->redirection_title_format, $link_query->title, $meta_categories);

        // Meta tags
        MetaTag::setTags([
            'title' => $link_title,
            'canonical' => asset($this->detail_base . '/' . $link_query->slug),
        ]);

        if ($this->redirection_page_noindex == '1') {
            MetaTag::setTags([
                'robots' => 'noindex',
            ]);
        }

        // Return View
        return view('' . $this->language_direction . 'frontend::redirect')->with('link_query', $link_query);
    }

    /** Link Redirection Process */
    public function link_redirect()
    {
        // Retrieve link details
        $redirection_query = DB::table('links')->where('id', request()->id)->first();

        // Return 404 page if link not found
        $redirection_query == null ? abort(404) : '';

        // Increase clicks count
        DB::table('links')->where('id', request()->id)->increment('clicks');

        // Redirect to URL
        return redirect($redirection_query->url, 301);
    }

    /** HTML Widget */
    public function widget()
    {
        // Return 404 page if widget embled is not allowed
        $this->allow_widget_embed == '0' ? abort(404) : '';

        // Retrieve link details
        $embed_query = DB::table('links')->where('bypass_details', '0')->where('enabled', '1')->where('id', request()->id)->first();

        // Return 404 page if link not found
        $embed_query == null ? abort(404) : '';

        $lang_slugs = array('detail_base');

        foreach ($this->site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $settings[$setting->name] = $setting->value;
            } else {
                $settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        foreach ($this->languages as $language) {
            if ($settings['root_language'] == '1' && $language->code == $settings['site_language']) {
                $language_prefix[$language->id] = '/';
            } else {
                $language_prefix[$language->id] = $language->code . '/';
            }
        }

        // Return View
        return view('' . $this->language_direction . 'frontend::widget')->with('embed_query', $embed_query)->with('settings', $settings)->with('language_prefix', $language_prefix);
    }

    /** Tags */
    public function tag()
    {
        $slug = request()->slug;

        $tag_title = DB::table('tagging_tags')->where('slug', $slug)->first();

        // Return 404 page if tag not found
        if ($tag_title == null) {
            abort(404);
        }

        // Search query
        $links = Link::orderBy('featured_category', 'desc')->orderBy('id', 'desc')->withAnyTag([$slug])->where('enabled', '1')->limit($this->search_page_links_per_page)->get();
        // $links = Link::orderBy('featured_category', 'desc')->orderBy('id', 'desc')->withAnyTag([$slug])->where('enabled', '1')->where('language', $this->language_id)->limit($this->search_page_links_per_page)->get();

        // Meta tags
        MetaTag::setTags([
            'title' => __('general.tagged_with', ['keyword' => $tag_title->name]),
            'description' => $this->site_description,
            'twitter_site' => $this->twitter_account,
            'twitter_title' => __('general.tagged_with', ['keyword' => $tag_title->name]),
            'twitter_card' => 'summary',
            'twitter_url' => url()->current(),
            'twitter_description' => $this->site_description,
            'twitter_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png',
            'og_title' => __('general.tagged_with', ['keyword' => $tag_title->name]),
            'og_description' => $this->site_description,
            'og_url' => url()->current(),
            'og_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png', 'og_image_width' => 600, 'og_image_height' => 315,
        ]);

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.tagged_with', ['keyword' => $tag_title->name]), url()->current());

        return view('' . $this->language_direction . 'frontend::tags')->with('links', $links)->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('tag_title', $tag_title->name);
    }

    /** Random Link */
    public function random()
    {
        // Grab a random link
        $random_link = Link::orderBy(\DB::raw('RAND()'))->where('enabled', '1')->where('bypass_details', '0')->where('language', $this->language_id)->first();

        // Redirect to home page if there are no links on the site
        if ($random_link == null) {
            return redirect($this->language_prefix);
        }

        // Redirect to link
        return redirect("$this->language_prefix" . "$this->detail_base/$random_link[slug]");
    }

    /** Contact Page */
    public function contact()
    {
        // Default Contact Page Title Format
        if ($this->contact_page_title_format == null) {
            $this->contact_page_title_format = __('general.contact') . ' › %site_name%';
        }

        $contact_page_title_format = title_format($this->site_title, $this->contact_page_title_format);

        if ($this->contact_page_description != null) {
            $this->site_description = title_format($this->site_title, $this->contact_page_description);
        }

        // Meta tags
        meta_tags($contact_page_title_format, $this->site_description, $this->twitter_account, asset('images/default_share_image.png'), 600, 315, 'website');

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.contact'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::contact')->with('breadcrumb_schema_data', $breadcrumb_schema_data);
    }

    /** Browse History */
    public function browse_history(Request $request)
    {
        if ($this->allow_user_history == 0) {
            abort(404);
        }

        if (!Auth::check()) {
            return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.login_to_view_page'));
        }

        if ($request->has('clear')) {
            DB::table('user_histories')->where('user_id', Auth::user()->id)->delete();
            Session::flash('message', __('general.history_cleared'));
            return redirect()->back();
        }

        // Meta tags
        MetaTag::setTags([
            'title' => __('general.history') . ' › ' . $this->site_title,
            'description' => $this->site_description,
            'twitter_site' => $this->twitter_account,
            'twitter_title' => __('general.history') . ' › ' . $this->site_title,
            'twitter_card' => 'summary',
            'twitter_url' => url()->current(),
            'twitter_description' => $this->site_description,
            'twitter_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png',
            'og_title' => __('general.history') . ' › ' . $this->site_title,
            'og_description' => $this->site_description,
            'og_url' => url()->current(),
            'og_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png', 'og_image_width' => 600, 'og_image_height' => 315,
        ]);

        $browse_history = DB::table('user_histories')->orderBy('id', 'desc')->where('user_id', Auth::user()->id)->get()->pluck('link_id')->implode(', ');

        if ($browse_history == null) {
            $links = collect();
        } else {
            $links = DB::table('links')->where('enabled', '1')->where('language', $this->language_id)->whereIn('id', explode(',', $browse_history))->orderByRaw(DB::raw("FIELD(id, $browse_history)"))->get();
        }

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.history'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::history')->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('links', $links);
    }

    /** Favorites */
    public function favorites(Request $request)
    {
        if ($this->allow_user_favorites == 0) {
            abort(404);
        }

        if (!Auth::check()) {
            return Redirect::to($this->language_prefix . 'member-area/login')->withErrors(__('general.login_to_view_page'));
        }

        if ($request->has('clear')) {
            DB::table('user_favorites')->where('user_id', Auth::user()->id)->delete();
            Session::flash('message', __('general.favorites_cleared'));
            return redirect()->back();
        }

        // Meta tags
        MetaTag::setTags([
            'title' => __('general.favorites') . ' › ' . $this->site_title,
            'description' => $this->site_description,
            'twitter_site' => $this->twitter_account,
            'twitter_title' => __('general.favorites') . ' › ' . $this->site_title,
            'twitter_card' => 'summary',
            'twitter_url' => url()->current(),
            'twitter_description' => $this->site_description,
            'twitter_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png',
            'og_title' => __('general.favorites') . ' › ' . $this->site_title,
            'og_description' => $this->site_description,
            'og_url' => url()->current(),
            'og_image' => request()->getSchemeAndHttpHost() . '/images/default_share_image.png', 'og_image_width' => 600, 'og_image_height' => 315,
        ]);

        $favorites = DB::table('user_favorites')->orderBy('id', 'desc')->where('user_id', Auth::user()->id)->get()->pluck('link_id')->implode(', ');

        if ($favorites == null) {
            $links = collect();
        } else {
            $links = DB::table('links')->where('enabled', '1')->where('language', $this->language_id)->whereIn('id', explode(',', $favorites))->orderByRaw(DB::raw("FIELD(id, $favorites)"))->get();
        }

        // Schema.org Breadcrumbs
        $breadcrumb_schema_data = schema_generator($this->site_title, $this->language_prefix, __('general.favorites'), url()->current());

        // Return view
        return view('' . $this->language_direction . 'frontend::favorites')->with('breadcrumb_schema_data', $breadcrumb_schema_data)->with('links', $links);
    }

}