<?php

namespace App\Http\Controllers;

use App;
use App\Models\Translation;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Redirect;

class TranslationController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $this->site_settings = DB::table('settings')->get();

        foreach ($this->site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
        }

        // List of translations
        $translations = Translation::orderBy('sort', 'ASC')->get();

        // List of countries
        $countries = json_decode(file_get_contents(public_path() . "/vendor/countries.json"), true);

        // Pass data to views
        View::share(['countries' => $countries, 'translations' => $translations]);
    }

    /** Index */
    public function index(Request $request)
    {
        if ($request->has('sort')) {
            // List of translations
            $posts = Translation::orderBy('sort', 'ASC')->get();

            $id = $request->input('id');
            $sorting = $request->input('sort');

            // Update sort order
            foreach ($posts as $item) {
                Translation::where('id', '=', $id)->update(array(
                    'sort' => $sorting,
                ));
            }
            return \Response::json('success', 200);
        }

        // Return view
        return view('adminlte::translations.index');
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::translations.create');
    }

    /** Store */
    public function store(Request $request)
    {
        $this->validate($request, [
            'language' => 'required',
            'code' => 'required|unique:translations,code|max:2',
            'locale_code' => 'required',
            'icon' => 'required',
            'text_direction' => 'required',
        ]);

        $translate = new Translation;
        $translate->language = $request->get('language');
        $translate->code = $request->get('code');
        $translate->locale_code = $request->get('locale_code');
        $translate->icon = $request->get('icon');
        $translate->text_direction = $request->get('text_direction');
        $translate->hidden = $request->get('hidden') ? 1 : 0;

        // Retrieve last item in sort order and add +1
        $translate->sort = Translation::max('sort') + 1;

        $translation_folder = app()['path.lang'] . '/' . $translate->code;

        File::makeDirectory($translation_folder);
        File::copy(app()['path.lang'] . '/en/general.php', app()['path.lang'] . '/' . $translate->code . '/general.php');
        File::copy(app()['path.lang'] . '/en/admin.php', app()['path.lang'] . '/' . $translate->code . '/admin.php');

        $translate->save();

        // Insert slug records
        $data = [
            ['name' => 'detail_base', 'value' => 'detail', 'language' => $translate->id],
            ['name' => 'category_base', 'value' => 'category', 'language' => $translate->id],
            ['name' => 'page_base', 'value' => 'page', 'language' => $translate->id],
            ['name' => 'redirect_base', 'value' => 'redirect', 'language' => $translate->id],
            ['name' => 'article_base', 'value' => 'article', 'language' => $translate->id],
            ['name' => 'tag_base', 'value' => 'tag', 'language' => $translate->id],
            ['name' => 'location_base', 'value' => 'location', 'language' => $translate->id],
            ['name' => 'contact_slug', 'value' => 'contact-us', 'language' => $translate->id],
            ['name' => 'latest_links_slug', 'value' => 'latest-links', 'language' => $translate->id],
            ['name' => 'popular_links_slug', 'value' => 'popular-links', 'language' => $translate->id],
            ['name' => 'all_articles_slug', 'value' => 'all-articles', 'language' => $translate->id],
            ['name' => 'favorites_slug', 'value' => 'favorites', 'language' => $translate->id],
            ['name' => 'history_slug', 'value' => 'history', 'language' => $translate->id],
            ['name' => 'home_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'link_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'redirection_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'category_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'custom_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'custom_article_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'articles_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'contact_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'latest_links_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'popular_links_page_title_format', 'value' => null, 'language' => $translate->id],
            ['name' => 'articles_page_description', 'value' => null, 'language' => $translate->id],
            ['name' => 'contact_page_description', 'value' => null, 'language' => $translate->id],
            ['name' => 'latest_links_page_description', 'value' => null, 'language' => $translate->id],
            ['name' => 'popular_links_page_description', 'value' => null, 'language' => $translate->id],
            ['name' => 'site_title', 'value' => 'Link Directory Pro', 'language' => $translate->id],
            ['name' => 'site_description', 'value' => 'Site Description', 'language' => $translate->id],
        ];

        DB::table('settings')->insert($data); // Query Builder approach

        // Redirect to translation edit page
        return redirect()->route('translations.edit', $translate->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve translation details
        $translation = Translation::find($id);

        // Return 404 page if translation not found
        if ($translation == null) {
            abort(404);
        }

        $frontend_location = app()['path.lang'] . '/en/general.php';
        $admin_location = app()['path.lang'] . '/en/admin.php';
        $translation_admin_org = include $admin_location;
        $translation_frontend_org = include $frontend_location;

        $frontend_location_target = app()['path.lang'] . '/' . $translation->code . '/general.php';
        $admin_location_target = app()['path.lang'] . '/' . $translation->code . '/admin.php';
        $translation_frontend_target = include $frontend_location_target;
        $translation_admin_target = include $admin_location_target;

        // Return view
        return view('adminlte::translations.edit', compact('translation', 'id', 'translation_admin_org', 'translation_frontend_org', 'translation_frontend_target', 'translation_admin_target'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        // Retrieve translation details
        $translate = Translation::find($id);

        $translate_code = $translate->code;

        if ($request->get('translation_type') == 1) {
            $this->validate($request, [
                'language' => 'required',
                'code' => 'required|max:2|unique:translations,code,' . $id,
                'locale_code' => 'required',
                'icon' => 'required',
                'text_direction' => 'required',
            ]);

            $translation_folder = $translate->code;
            $translate->language = $request->get('language');
            $translate->code = $request->get('code');
            $translate->locale_code = $request->get('locale_code');
            $translate->icon = $request->get('icon');
            $translate->text_direction = $request->get('text_direction');
            $translate->hidden = $request->get('hidden') ? 1 : 0;

            if ($translate->isDirty('code')) {
                File::moveDirectory(app()['path.lang'] . '/' . $translation_folder, app()['path.lang'] . '/' . $translate->code);
            }

            // Update settings if main site language code is changed
            if ($translate_code == $this->site_language && $translate->isDirty('code')) {
                DB::update("update settings set value = '$translate->code' WHERE name = 'site_language'");
            }

            $translate->save();
        }

        if ($request->get('translation_type') == 2 or $request->get('translation_type') == 3) {

            function varexport($expression, $return = false)
            {
                $export = var_export($expression, true);
                $export = preg_replace("/^([ ]*)(.*)/m", '$1$1$2', $export);
                $array = preg_split("/\r\n|\n|\r/", $export);
                $array = preg_replace(["/\s*array\s\($/", "/\)(,)?$/", "/\s=>\s$/"], [null, ']$1', ' => ['], $array);
                $export = join(PHP_EOL, array_filter(["["] + $array));
                $export = "<?php\nreturn " . $export . ";";
                if ((bool) $return) {
                    return $export;
                } else {
                    echo $export;
                }

            }

            $a = array();

            foreach ($request->except(array('_token', '_method')) as $key => $value) {
                $a[$key] = $value;
            }

            if ($request->get('translation_type') == 2) {
                $target_lang = app()['path.lang'] . '/' . $translate->code . '/general.php';
            }
            if ($request->get('translation_type') == 3) {
                $target_lang = app()['path.lang'] . '/' . $translate->code . '/admin.php';
            }

            File::put($target_lang, varexport($a, true));

        }

        // Redirect to translation edit page
        return redirect()->route('translations.edit', $translate->id)->with('success', __('admin.data_updated'));
    }

    /** Destroy */
    public function destroy($id)
    {
        // Retrieve translation details
        $translate = Translation::find($id);

        // Check if the links are affiliated with the language
        $link_count = DB::table('links')->where('language', $id)->count();

        // Check if the categories are affiliated with the language
        $category_count = DB::table('categories')->where('language', $id)->count();

        // Check if the articles are affiliated with the language
        $article_count = DB::table('articles')->where('language', $id)->count();

        // Check if the pages are affiliated with the language
        $page_count = DB::table('pages')->where('language', $id)->count();

        $item_count = $link_count + $category_count + $article_count + $page_count;

        // Display error message if the package is affiliated with link or submissions
        if ($item_count >= '1') {
            return Redirect::back()->withErrors(__('admin.content_associated_with_language'));
        }

        // Check if user is trying to delete main language (English)
        if ($id == '1') {

            // Redirect to list of translations
            return redirect()->route('translations.index')->with('error', __('admin.error'));

        } else {

            $translate->delete();

            // Delete language records in settings table
            DB::table('settings')->where('language', $id)->delete();

            File::deleteDirectory(app()['path.lang'] . '/' . $translate->code);

            // Redirect to list of translations
            return redirect()->route('translations.index')->with('success', __('admin.data_deleted'));

        }
    }

}
