<?php
namespace App\Http\Controllers;

use App;
use App\Models\Bank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Redirect;

class BankController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            $settings[$setting->name] = $setting->value;
        }

        // Pass data to views
        View::share(['settings' => $settings]);
    }

    /** Index */
    public function index(Request $request)
    {

        if ($request->has('sort')) {
            // List of banks
            $banks = Bank::orderBy('sort', 'ASC')->get();

            $id = $request->input('id');
            $sorting = $request->input('sort');

            // Update sort order
            foreach ($banks as $item) {
                Bank::where('id', '=', $id)->update(array(
                    'sort' => $sorting,
                ));
            }
            
            return \Response::json('success', 200);

        }
        
        // List of banks
        $banks = Bank::orderBy('sort', 'ASC')->get();

        // Return view
        return view('adminlte::bank_accounts.index', compact('banks'));
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::bank_accounts.create');
    }

    /** Store */
    public function store(Request $request)
    {
        $this->validate($request, [
            'bank_name' => 'required|max:255',
            'details' => 'required',
        ]);

        $bank = new Bank;
        $bank->bank_name = $request->get('bank_name');
        $bank->details = $request->get('details');
        $bank->enabled = $request->get('enabled') ? 1 : 0;
        
        // Retrieve last item in sort order and add +1
        $bank->sort = Bank::max('sort') + 1;

        $bank->save();

        // Redirect to bank edit page
        return redirect()->route('bank_accounts.edit', $bank->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve bank details
        $bank = Bank::find($id);

        // Return 404 page if bank not found
        if ($bank == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::bank_accounts.edit', compact('bank', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'bank_name' => 'required|max:255',
            'details' => 'required',
        ]);

        // Retrieve bank details
        $bank = Bank::find($id);
        $bank->bank_name = $request->get('bank_name');
        $bank->details = $request->get('details');
        $bank->enabled = $request->get('enabled') ? 1 : 0;

        $bank->save();

        // Redirect to bank edit page
        return redirect()->route('bank_accounts.edit', $bank->id)->with('success', __('admin.data_updated'));
    }

    /** Destroy */
    public function destroy($id)
    {
        // Check if the orders are affiliated with the bank account
        $payment_notification_check = DB::table('payment_notifications')->where('bank_account', $id)->count();

        // Display error message if the bank account is affiliated with orders
        if ($payment_notification_check >= '1') {
            return Redirect::back()->withErrors(__('admin.bank_account_in_use'));
        }

        // Retrieve bank details
        $bank = Bank::find($id);

        $bank->delete();

        // Redirect to list of banks
        return redirect()->route('bank_accounts.index')->with('success', __('admin.data_deleted'));
    }

}
