<?php
namespace App\Http\Controllers;

use App;
use App\Models\Partner;
use App\Models\Translation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;

class PartnerController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $settings[$setting->name] = $setting->value;
        }

        // Link Types
        $link_types = array(
            '' => __('admin.regular_link'),
            '1' => __('admin.paid_sponsored'),
            '2' => __('admin.ugc'),
            '3' => __('admin.nofollow'),
        );

        // List of languages
        $translations = Translation::orderBy('id', 'ASC')->get();

        foreach ($translations as $language) {
            $language_id[$language->code] = $language->id;
            $language_code[$language->id] = $language->icon;
        }

        // Array Data
        $array_data = array(
            'language_code' => $language_code,
            'site_language_id' => $language_id[$settings['site_language']],
        );

        // Pass data to views
        View::share(['settings' => $settings, 'link_types' => $link_types, 'translations' => $translations, 'array_data' => $array_data]);
    }

    /** Index */
    public function index(Request $request)
    {
        if ($request->has('sort')) {
            // List of partners
            $partners = Partner::orderBy('sort', 'ASC')->get();

            $id = $request->input('id');
            $sorting = $request->input('sort');

            // Update sort order
            foreach ($partners as $item) {
                Partner::where('id', '=', $id)->update(array(
                    'sort' => $sorting,
                ));
            }
            return \Response::json('success', 200);
        }

        // Return view
        return view('adminlte::partners.index')->with('language_menu', '1');
    }

    /** Partner List */
    public function partner_list(Request $request)
    {
        $id = request()->id;

        // List of partners
        $partners = Partner::orderBy('sort', 'ASC')->where('language', $id)->get();

        // Return view
        return view('adminlte::partners.index')->with('partners', $partners)->with('language_menu', '0');
    }

    /** Create */
    public function create()
    {
        // Return view
        return view('adminlte::partners.create');
    }

    /** Store */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'url' => 'required|url',
        ]);

        $partner = new Partner;
        $partner->title = $request->get('title');
        $partner->url = $request->get('url');
        $partner->link_type = $request->get('link_type');
        $partner->navbar = $request->get('navbar') ? 1 : 0;
        $partner->footer = $request->get('footer') ? 1 : 0;
        $partner->language = $request->get('language');

        // Retrieve last item in sort order and add +1
        $partner->sort = Partner::max('sort') + 1;

        $partner->save();

        // Redirect to partner edit page
        return redirect()->route('partners.edit', $partner->id)->with('success', __('admin.data_added'));
    }

    /** Edit */
    public function edit($id)
    {
        // Retrieve partner details
        $partner = Partner::find($id);

        // Return 404 page if partner not found
        if ($partner == null) {
            abort(404);
        }

        // Return view
        return view('adminlte::partners.edit', compact('partner', 'id'));
    }

    /** Update */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'title' => 'required|max:255',
            'url' => 'required|url',
        ]);

        // Retrieve partner details
        $partner = Partner::find($id);
        $partner->title = $request->get('title');
        $partner->url = $request->get('url');
        $partner->link_type = $request->get('link_type');
        $partner->navbar = $request->get('navbar') ? 1 : 0;
        $partner->footer = $request->get('footer') ? 1 : 0;
        $partner->language = $request->get('language');

        $partner->save();

        // Redirect to partner edit page
        return redirect()->route('partners.edit', $partner->id)->with('success', __('admin.data_updated'));
    }

    /** Destroy */
    public function destroy($id)
    {
        // Retrieve partner details
        $partner = Partner::find($id);

        $partner->delete();

        // Redirect to list of partners
        return redirect()->route('partners.index')->with('success', __('admin.data_deleted'));
    }

}
