<?php

namespace App\Http\Controllers;

use App;
use App\Mail\Sendemail;
use App\Mail\Submissionapproved;
use App\Mail\Submissionrejected;
use App\Models\Bank;
use App\Models\Category;
use App\Models\Location;
use App\Models\Link;
use App\Models\Package;
use App\Models\Submission;
use App\Models\Translation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\View;
use Intervention\Image\ImageManagerStatic as Image;
use Mail;
use Redirect;
use Brick\PhoneNumber\PhoneNumber;
use Brick\PhoneNumber\PhoneNumberParseException;

class SubmissionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');

        // Site Settings
        $site_settings = DB::table('settings')->get();

        foreach ($site_settings as $setting) {
            $setting_name = $setting->name;
            $this->$setting_name = $setting->value;
            if ($setting->language == null) {
                $this->settings[$setting->name] = $setting->value;
            } else {
                $this->settings[$setting->name . '-' . $setting->language] = $setting->value;
            }
        }

        // List of categories
        $this->categories = Category::where('parent_id', '=', 0)->orderBy('sort', 'ASC')->get();

        // List of packages
        $this->packages = Package::OrderBy('sort', 'ASC')->get();

        foreach ($this->packages as $package) {
            $package_name[$package->id] = $package->title;
        }

        // List of payment methods
        $this->payment_methods = DB::table('payment_methods')->orderBy('title', 'ASC')->get();

        foreach ($this->payment_methods as $payment_method) {
            $payment_method_name[$payment_method->id] = __('admin.' . $payment_method->title . '');
        }

        // List of languages
        $translations = Translation::orderBy('sort', 'ASC')->get();

        // Language prefixes
        foreach ($translations as $language) {
             $language_code[$language->id] = strtoupper($language->code);
           if ($language->code == $this->settings['site_language']) {
                $this->site_language_id = $language->id;
            }
            if ($this->settings['root_language'] == '1' && $language->code == $this->settings['site_language']) {
                $this->language_prefix[$language->id] = '/';
            } else {
                $this->language_prefix[$language->id] = $language->code . '/';
            }
        }

        $this->mail_site_title = $this->settings['site_title' . '-' . $this->site_language_id];
        $this->mail_from = $_ENV['MAIL_FROM_ADDRESS'];

        // Array Data
        $array_data = array(
            'package_name' => $package_name,
            'payment_method_name' => $payment_method_name,
            'language_code' => $language_code,
        );
        
        // List of locations
        $locations = Location::where('parent_id', '=', 0)->orderBy('language', 'ASC')->get();

        // Pass data to views
        View::share(['categories' => $this->categories, 'link_types' => link_type_values(), 'locations' => $locations, 'packages' => $this->packages, 'payment_methods' => $this->payment_methods, 'array_data' => $array_data]);
    }

    /** Index */
    public function index()
    {
        // List of latest submissions
        $submissions = DB::table('orders')
            ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
            ->leftJoin('packages', 'orders.package', '=', 'packages.id')
            ->select('orders.package AS package',
                'orders.payment_method AS payment_method',
                'packages.price AS price',
                'submissions.*')
            ->where('orders.status', '1')
            ->where('orders.is_renewal', '0')
            ->orderBy('orders.id', 'ASC')
            ->paginate(15);

        // Return view
        return view('adminlte::submissions.index', compact('submissions'));
    }

    /** Show */
    public function show(Request $request, $id)
    {
        // Retrieve order details
        $order = DB::table('orders')
            ->leftJoin('users', 'orders.user', '=', 'users.id')
            ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
            ->leftJoin('packages', 'orders.package', '=', 'packages.id')
            ->select('orders.package AS package',
                'orders.payment_method AS payment_method',
                'orders.subscription_id AS subscription_id',
                'orders.payment_status AS paid',
                'orders.payment_id AS payment_id',
                'orders.id AS order_id',
                'orders.created_at AS order_time',
                'orders.status AS status',
                'orders.price AS price',
                'orders.currency_symbol AS currency_symbol',
                'packages.duration AS duration',
                'users.name AS user_name',
                'users.email AS user_email',
                'submissions.*')
            ->where('orders.status', '1')
            ->where('submissions.id', $id)
            ->first();

        // Return 404 page if submission is not found
        if ($order == null) {
            abort(404);
        }

        // Payment not received notification
        if ($request->has('payment_not_received')) {
            Mail::to($order->user_email)->send(new Sendemail(__('admin.about_your_payment'), __('admin.payment_not_received_mail'), $this->mail_site_title, $this->mail_from));

            DB::table('orders')->where('id', $order->order_id)->update(['payment_status' => '0', 'status' => '0']);

            return redirect()->route('submissions.index')->with('success', __('admin.payment_not_received_notification'));
        }

        $data_to_pass = [];

        $deep_link_data = unserialize($order->deep_links);

        if (!$deep_link_data) {
            $deep_link_data = array(
                "link_1" => array("title" => null, "url" => null, "type" => null),
                "link_2" => array("title" => null, "url" => null, "type" => null),
                "link_3" => array("title" => null, "url" => null, "type" => null),
            );
        }

        $other_contacts = unserialize($order->other_contacts);

        if (!$other_contacts) {
            $other_contacts = array(
                "facebook_url" => array("url" => null),
                "twitter_url" => array("url" => null),
                "instagram_url" => array("url" => null),
                "linkedin_url" => array("url" => null),
                "whatsapp_number" => array("number" => null),
                "telegram_url" => array("url" => null),
            );
        }

        // Subscription Payments
        $subscription_payments = DB::table('orders')->where('submission_id', $id)->orderBy('id', 'DESC')->get();

        $now = Carbon::now();

        // Submission Durations
        $expire_time = expire_time($order->duration, $order->created_at);

        // Number of payments made
        $total_payments = count($subscription_payments);

        if ($order->duration == '0') {
            $new_expire_date = null;
            $expire_date = null;
            $additional_time = null;
        } else {
            $expire_date = $expire_time;
            $prev = Carbon::create($order->order_time);
            $additional_time = $prev->longAbsoluteDiffForHumans($now, 3);

            $new_expire_date = date('Y-m-d H:i:s', strtotime($expire_time) + (strtotime($now) - strtotime($order->order_time)));

            if (isset($order->subscription_id) && $total_payments >= '1') {
                for ($x = 1; $x <= $total_payments - 1; $x++) {
                    $new_expire_date = date('Y-m-d H:i:s', strtotime("$new_expire_date + 24 hours"));
                }
            }

        }

        // List of categories
        $categories_list = Category::orderBy('sort', 'ASC')->get();

        // Payment notification
        $payment_notification = DB::table('payment_notifications')->where('order_id', $order->order_id)->first();

        // Crypto payment details
        $crypto_payment = DB::table('crypto_payments')->where('order_id', $order->order_id)->first();

        if ($order->payment_method == '2') {
            // Bank account details
            $bank = Bank::where('id', $payment_notification->bank_account)->first();
            $data_to_pass['bank_name'] = $bank->bank_name;
        }

        // Append package name
        foreach ($this->packages as $package) {
            if ($package->id == $order->package) {
                $data_to_pass['package_name'] = $package->title;
            }
        }

        // Append payment method
        foreach ($this->payment_methods as $payment_method) {
            if ($payment_method->id == $order->payment_method) {
                $data_to_pass['payment_method'] = __('admin.' . $payment_method->title . '');
            }
        }

        $order_categories = DB::table('category_submissions')->where('link_id', $order->id)->get()->pluck('category_id')->toArray();

        // Return view
        return view('adminlte::submissions.create')->with('order', $order)->with('deep_link_data', $deep_link_data)->with('categories_list', $categories_list)->with('other_contacts', $other_contacts)->with('expire_date', $expire_date)->with('additional_time', $additional_time)->with('new_expire_date', $new_expire_date)->with('subscription_payments', $subscription_payments)->with('payment_notification', $payment_notification)->with('crypto_payment', $crypto_payment)->with('data_to_pass', $data_to_pass)->with('order_categories', $order_categories);
    }

    /** Store */
    public function store(Request $request)
    {
        // Check if slug exists
        $slug_check = Link::where('slug', $request->get('slug'))->first();

        if ($slug_check != null) {
            return back()->withInput()->withErrors(__('admin.slug_in_use'));
        }
        
        // Check if phone number is valid
        if ($request->get('phone_number') != null) {
            try {
                $number = PhoneNumber::parse($request->get('phone_number'));
            } catch (PhoneNumberParseException $e) {
                return back()->withInput()->withErrors($e->getMessage());
            }
        }

        $submission_id = $request->get('submission_id');

        // Retrieve submission details
        $submission = DB::table('orders')
            ->leftJoin('users', 'orders.user', '=', 'users.id')
            ->leftJoin('submissions', 'orders.submission_id', '=', 'submissions.id')
            ->leftJoin('payment_methods', 'orders.payment_method', '=', 'payment_methods.id')
            ->select('orders.id AS order_id', 'payment_methods.subscription AS subscription', 'orders.package', 'submissions.image', 'users.email', 'users.id AS owner_id')
            ->where('orders.submission_id', $submission_id)
            ->first();

        $site_owner_email = $submission->email;

        $link = new Link;

        $this->validate($request, [
            'title' => 'required|max:255',
            'description' => 'required|max:755',
            'categories' => ['required', 'array', 'min:1'],
            'categories.*' => ['required', 'integer', 'exists:categories,id'],
            'page_views' => 'required|numeric',
            'clicks' => 'required|numeric',
            'url' => 'required|url',
            'visible_url' => 'nullable',
            'votes' => 'required|numeric|max:5',
            'total_votes' => 'required|numeric',
        ]);

        $link->featured_category = $request->get('featured_category') ? 1 : 0;
        $link->featured_home = $request->get('featured_home') ? 1 : 0;
        $link->bypass_redirect = $request->get('bypass_redirect') ? 1 : 0;
        $link->bypass_details = $request->get('bypass_details') ? 1 : 0;
        $link->badge = $request->get('badge') ? 1 : 0;
        $link->submission_id = $submission_id;
        $link->status = '2';

        if ($submission->subscription == '1') {
            $link->is_subscription = '1';
        }

        for ($x = 1; $x <= $this->max_deep_links; $x++) {
            $arra['title'] = $request->get('deep_link_' . $x . '_title');
            $arra['url'] = $request->get('deep_link_' . $x . '_url');
            $arra['type'] = $request->get('deep_link_' . $x . '_type');

            $deep_links['link_' . $x] = $arra;
        }

        $link->deep_links = serialize($deep_links);

        $other_contacts = array(

            "facebook_url" => array(
                "url" => $request->get('facebook_url'),
            ),

            "twitter_url" => array(
                "url" => $request->get('twitter_url'),
            ),

            "instagram_url" => array(
                "url" => $request->get('instagram_url'),
            ),

            "linkedin_url" => array(
                "url" => $request->get('linkedin_url'),
            ),

            "whatsapp_number" => array(
                "number" => $request->get('whatsapp_number'),
            ),

            "telegram_url" => array(
                "url" => $request->get('telegram_url'),
            ),
        );

        $link->other_contacts = serialize($other_contacts);
        
        $category_data = $request->input('categories');

        // Get Category Language ID
        $category_language_id = DB::table('categories')->where('id', $category_data[0])->first();

        $has_image = 0;

        // Check if the site icon is uploaded
        if ($request->hasFile('different_image')) {
            $image = $request->file('different_image');
            $file_name = time() . '.' . $image->getClientOriginalExtension();
            $location = public_path('images/links/' . $file_name);
            Image::make($image)->resize(200, 200)->save($location);
            $link->image = $file_name;
            $has_image = 1;
        } else {
            $image = $request->get('image');
            if (!empty($submission->image)) {
                $path = $image;
                $file_name = time() . '.png';

                // Move image from submission folder
                $temp_image = public_path('images/submissions/' . $image);
                $location = public_path('images/links/' . $file_name);

                // Resize image and save
                Image::make($temp_image)->resize(200, 200)->save($location);

                $link->image = $file_name;

                unlink($temp_image);
                $has_image = 1;
            }
        }

        // Check if the screenshot has been uploaded
        if ($request->hasFile('screenshot')) {
            $screenshot = $request->file('screenshot');
            if ($this->save_as_webp == '1') {
                $file_name = time() . '.webp';
            } else {
                $file_name = time() . '.' . $screenshot->getClientOriginalExtension();
            }
            $location = public_path('images/screenshots/' . $file_name);
            Image::make($screenshot)->resize(600, 600)->save($location, $this->image_quality);
            $link->screenshot = $file_name;
        }

        $link->slug = $request->get('slug');
        $link->title = $request->get('title');
        $link->description = $request->get('description');
        $link->page_views = $request->get('page_views');
        $link->clicks = $request->get('clicks');
        $link->url = $request->get('url');
        $link->visible_url = $request->get('visible_url');
        $link->bypass_type = $request->get('bypass_type');
        $link->bypass_details_type = $request->get('bypass_details_type');
        $link->phone_number = $request->get('phone_number');
        $link->email = $request->get('email');
        $link->address = $request->get('address');
        $link->visible_url = $request->get('visible_url');
        $link->custom_title = $request->get('custom_title');
        $link->custom_description = $request->get('custom_description');
        $link->package = $submission->package;
        $link->owner = $submission->owner_id;
        $link->expire_at = $request->get('expire_at');
        $link->enabled = $request->get('enabled') ? 1 : 0;
        $link->votes = $request->get('votes');
        $link->total_votes = $request->get('total_votes');
        $link->language = $category_language_id->language;
        $link->map_lon = $request->get('map_lon');
        $link->map_lat = $request->get('map_lat');
        $link->map_zoom = $request->get('map_zoom');
        $link->gallery = '';

        $link->hidden_content = $request->get('hidden_content');

        preg_match_all('/src="(data:image\/[^;]+;base64[^"]+)"/', $link->hidden_content, $result, PREG_PATTERN_ORDER);

        $total_images = count($result[0]);

        for ($x = 0; $x <= $total_images - 1; $x++) {
            $data = substr($result[0][$x], strpos($result[0][$x], ',') + 1);
            $data = base64_decode($data);
            if ($this->save_as_webp == '1') {
                $file_name = time() . rand(11111, 99999) . '.webp';
            } else {
                $file_name = time() . rand(11111, 99999) . '.png';
            }
            \File::put(public_path() . '/images/uploads/' . $file_name, $data);
            $imagefile = asset('images/uploads') . '/' . $file_name;
            $link->hidden_content = str_replace($result[0][$x], "src=\"$imagefile\" class=\"img-fluid\"", $link->hidden_content);
            $link->hidden_content = preg_replace('/(\<img[^>]+)(style\=\"[^\"]+\")([^>]+)(>)/', '${1}${3}${4}', $link->hidden_content);
        }

        if ($link->hidden_content == '<p><br></p>') {
            $link->hidden_content = null;
        }

        $link->save();
        
        $link->categories()->sync((array) $request->input('categories'));
        $link->locations()->sync((array) $request->input('locations'));

        if ($request->get('slug') == null) {
            $link->slug = null;
            $link->update(['title' => $link->title]);
        }

        DB::table('orders')->where('id', $submission->order_id)->update(['status' => 2]);

        $url = asset($this->language_prefix[$category_language_id->language] . $this->settings['detail_base' . '-' . $category_language_id->language] . '/' . $link->slug);

        // Notify the submitter by email if submission is approved
        Mail::to($site_owner_email)->send(new Submissionapproved($url, $this->mail_site_title, $this->mail_from));

        // Update category link counts
        count_categories();

        // Update Order Record and Insert Link ID
        DB::table('orders')->where('submission_id', $submission_id)->update(['link_id' => $link->id]);

        // Ping Google
        if ($this->ping_google == '1') {
            $sitemap_url = asset('sitemap/links');
            $ch = curl_init("https://www.google.com/webmasters/tools/ping?sitemap=$sitemap_url");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, false);
            $result = curl_exec($ch);
            curl_close($ch);
        }

        if ($this->save_screenshot_as_icon == '1' && $has_image == '0') {
            $image_url = "https://image.thum.io/get/auth/$this->thum_io_id-$this->thum_io_url_key/$link->url";
            $ch = curl_init($image_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $result = curl_exec($ch);
            curl_close($ch);

            // Insert Queue Record
            DB::table('screenshot_queue')->insert(
                [
                    'link_id' => $link->id,
                    'as_icon' => 1,
                    'created_at' => date("Y-m-d H:i:s"),
                ]
            );
        }
        
        DB::table('category_submissions')->where('link_id', $submission->order_id)->delete();

        // Redirect to link edit page
        return redirect()->route('links.edit', $link->id)->with('success', __('admin.data_added'));
    }

    /** Destory */
    public function destroy(Request $request, $id)
    {
        // Retrieve submission details
        $submission = Submission::leftJoin('users', 'submissions.user', '=', 'users.id')
            ->select('users.email')
            ->where('submissions.id', $id)
            ->first();

        $site_owner_email = $submission->email;

        if (!empty($submission->image)) {
            if (file_exists(public_path() . '/images/submissions/' . $submission->image)) {
                unlink(public_path() . '/images/submissions/' . $submission->image);
            }
        }

        DB::table('orders')->where('submission_id', $id)->update(['status' => 3]);

        $reason = $request->reason;

        // Notify the submitter by email if submission is rejected
        Mail::to($site_owner_email)->send(new Submissionrejected($reason, $this->mail_site_title, $this->mail_from));

        // Redirect to list of submissions
        return redirect()->route('submissions.index')->with('success', __('admin.data_deleted'));
    }

}
