<?php
/*
Plugin Name: WP OpenAI Support Chatbot
Description: AI chat support with RAG, language detection, tone, login-only option, and enhanced indexing controls. SHORTCODE [openai_support_chatbot]
Version: 1.2
Author: Your Name
*/

if(!defined('ABSPATH')) exit;

class WP_OpenAI_Support_Chatbot {

    const OPT_KEY = 'wpoasc_api_key';
    const OPT_PROMPT = 'wpoasc_system_prompt';
    const OPT_TONE = 'wpoasc_tone';
    const OPT_SOURCES = 'wpoasc_sources';
    const OPT_MODEL = 'wpoasc_model';
    const OPT_LANG = 'wpoasc_default_lang';
    const OPT_LOGIN_ONLY = 'wpoasc_login_only';

    public function __construct(){
        register_activation_hook(__FILE__, [$this,'activate']);
        register_deactivation_hook(__FILE__, [$this,'deactivate']);

        add_action('admin_menu', [$this,'settings_page']);
        add_action('admin_init', [$this,'register_settings']);

        add_action('rest_api_init', function(){
            register_rest_route('openai-chat/v1','message',[
                'methods'=>'POST',
                'callback'=>[$this,'handle_message'],
                'permission_callback'=> '__return_true'
            ]);
        });

        add_action('wp_enqueue_scripts', [$this,'enqueue_assets']);
        add_shortcode('openai_support_chatbot', [$this,'shortcode']);
    }

    public function activate(){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            url TEXT NOT NULL,
            chunk LONGTEXT NOT NULL,
            embedding LONGTEXT NOT NULL,
            PRIMARY KEY (id)
        ) $charset;";
        require_once(ABSPATH.'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public function deactivate(){
        // opcionalno: briši tablicu
    }

    // -------------------------------
    // Settings page
    // -------------------------------
    public function settings_page(){
        add_options_page('OpenAI Chatbot','OpenAI Chatbot','manage_options','wpoasc_settings',[$this,'render_settings']);
    }

    public function register_settings(){
        register_setting('wpoasc_settings_group', self::OPT_KEY);
        register_setting('wpoasc_settings_group', self::OPT_PROMPT);
        register_setting('wpoasc_settings_group', self::OPT_TONE);
        register_setting('wpoasc_settings_group', self::OPT_SOURCES);
        register_setting('wpoasc_settings_group', self::OPT_MODEL);
        register_setting('wpoasc_settings_group', self::OPT_LANG);
        register_setting('wpoasc_settings_group', self::OPT_LOGIN_ONLY);

        add_settings_section('wpoasc_main','Main Settings', null, 'wpoasc_settings');

        add_settings_field(self::OPT_KEY,'OpenAI API Key', function(){
            $val = get_option(self::OPT_KEY,'');
            echo '<input type="text" name="'.self::OPT_KEY.'" value="'.esc_attr($val).'" size="50" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_PROMPT,'System Prompt', function(){
            $val = get_option(self::OPT_PROMPT,'');
            echo '<textarea name="'.self::OPT_PROMPT.'" rows="4" cols="50">'.esc_textarea($val).'</textarea>';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_TONE,'Tone / Style', function(){
            $val = get_option(self::OPT_TONE,'');
            echo '<input type="text" name="'.self::OPT_TONE.'" value="'.esc_attr($val).'" size="50" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_SOURCES,'Source URLs (one per line)', function(){
            $val = get_option(self::OPT_SOURCES,'');
            echo '<textarea name="'.self::OPT_SOURCES.'" rows="6" cols="50">'.esc_textarea($val).'</textarea>';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_MODEL,'Model', function(){
            $val = get_option(self::OPT_MODEL,'gpt-4o-mini');
            echo '<input type="text" name="'.self::OPT_MODEL.'" value="'.esc_attr($val).'" size="20" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_LANG,'Default Language', function(){
            $val = get_option(self::OPT_LANG,'en');
            echo '<input type="text" name="'.self::OPT_LANG.'" value="'.esc_attr($val).'" size="5" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_LOGIN_ONLY,'Login Only Chat', function(){
            $val = get_option(self::OPT_LOGIN_ONLY,0);
            echo '<input type="checkbox" name="'.self::OPT_LOGIN_ONLY.'" value="1" '.checked(1,$val,false).' /> Allow chat only for logged-in users';
        },'wpoasc_settings','wpoasc_main');
    }


    // -------------------------------
    // Render settings i action buttons
    // -------------------------------
    public function render_settings(){
        // Handle actions
        if(isset($_POST['wpoasc_delete_all'])){
            global $wpdb;
            $table = $wpdb->prefix.'wpoasc_embeddings';
            $wpdb->query("TRUNCATE TABLE $table");
            echo '<div class="notice notice-success is-dismissible"><p>All indexed pages deleted.</p></div>';
        }

        if(isset($_POST['wpoasc_update_existing'])){
            $result = $this->update_existing_index();
            echo '<div class="notice notice-success is-dismissible"><p>'.$result['message'].'</p></div>';
        }

        if(isset($_POST['wpoasc_add_new'])){
            $result = $this->add_new_pages_index();
            echo '<div class="notice notice-success is-dismissible"><p>'.$result['message'].'</p></div>';
        }

        // Standard settings form
        echo '<form method="post" action="options.php">';
        settings_fields('wpoasc_settings_group');
        do_settings_sections('wpoasc_settings');
        submit_button();
        echo '</form>';

        // Indexing action buttons
        echo '<hr>';
        echo '<form method="post">';
        echo '<input type="submit" name="wpoasc_delete_all" class="button button-secondary" value="Delete All Indexed Pages">';
        echo ' ';
        echo '<input type="submit" name="wpoasc_update_existing" class="button button-primary" value="Update Existing Indexed Pages">';
        echo ' ';
        echo '<input type="submit" name="wpoasc_add_new" class="button button-primary" value="Add New Pages to Index">';
        echo '</form>';
    }

    // -------------------------------
    // Indexiranje URL-ova
    // -------------------------------
    public function index_urls(){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $sources = array_filter(array_map('trim', explode("\n", get_option(self::OPT_SOURCES))));
        $api_key = get_option(self::OPT_KEY);
        if(!$api_key) return ['success'=>false,'message'=>'API key not set'];

        foreach($sources as $url){
            $res = wp_remote_get($url,['timeout'=>15]);
            if(is_wp_error($res)) continue;
            $text = wp_strip_all_tags(wp_remote_retrieve_body($res));
            $chunks = str_split($text,800);
            foreach($chunks as $chunk){
                $embedding = $this->get_embedding($chunk,$api_key);
                if($embedding){
                    $wpdb->insert($table,['url'=>$url,'chunk'=>$chunk,'embedding'=>json_encode($embedding)]);
                }
            }
        }
        return ['success'=>true,'message'=>'URL-ovi indeksirani.'];
    }

    private function update_existing_index(){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        if(!$api_key) return ['success'=>false,'message'=>'API key not set'];

        $urls = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        foreach($urls as $url){
            $res = wp_remote_get($url,['timeout'=>15]);
            if(is_wp_error($res)) continue;
            $text = wp_strip_all_tags(wp_remote_retrieve_body($res));
            $chunks = str_split($text,800);

            $wpdb->delete($table,['url'=>$url]);

            foreach($chunks as $chunk){
                $embedding = $this->get_embedding($chunk,$api_key);
                if($embedding){
                    $wpdb->insert($table,['url'=>$url,'chunk'=>$chunk,'embedding'=>json_encode($embedding)]);
                }
            }
        }
        return ['success'=>true,'message'=>'Existing indexed pages updated.'];
    }

    private function add_new_pages_index(){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        if(!$api_key) return ['success'=>false,'message'=>'API key not set'];

        $sources = array_filter(array_map('trim', explode("\n", get_option(self::OPT_SOURCES))));
        $existing = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        $added = 0;

        foreach($sources as $url){
            if(in_array($url,$existing)) continue;
            $res = wp_remote_get($url,['timeout'=>15]);
            if(is_wp_error($res)) continue;
            $text = wp_strip_all_tags(wp_remote_retrieve_body($res));
            $chunks = str_split($text,800);
            foreach($chunks as $chunk){
                $embedding = $this->get_embedding($chunk,$api_key);
                if($embedding){
                    $wpdb->insert($table,['url'=>$url,'chunk'=>$chunk,'embedding'=>json_encode($embedding)]);
                }
            }
            $added++;
        }
        return ['success'=>true,'message'=>"Added $added new page(s) to index."];
    }

    private function get_embedding($text,$api_key){
        $body = ['input'=>$text,'model'=>'text-embedding-3-small'];
        $res = wp_remote_post('https://api.openai.com/v1/embeddings',[
            'headers'=>[
                'Authorization'=>'Bearer '.$api_key,
                'Content-Type'=>'application/json'
            ],
            'body'=>wp_json_encode($body)
        ]);
        if(is_wp_error($res)) return null;
        $data = json_decode(wp_remote_retrieve_body($res),true);
        return $data['data'][0]['embedding'] ?? null;
    }


    // -------------------------------
    // Handle chat message
    // -------------------------------
    public function handle_message($req){
        $login_only = get_option(self::OPT_LOGIN_ONLY,0);
        if($login_only && !is_user_logged_in()){
            return ['success'=>false,'reply'=>'You must be logged in to use the chat.'];
        }

        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $message = trim((string)$req->get_param('message'));
        $api_key = get_option(self::OPT_KEY);
        $system_prompt = get_option(self::OPT_PROMPT);
        $tone = get_option(self::OPT_TONE);
        $default_lang = get_option(self::OPT_LANG,'en');

        if(!$api_key) return ['success'=>false,'reply'=>'API key not set'];

        // RATE LIMIT PO IP-u
        $ip = $_SERVER['REMOTE_ADDR'];
        $transient_key = 'wpoasc_rate_'.$ip;
        $count = get_transient($transient_key) ?: 0;
        $max_requests = 5;
        $period = 30;
        if($count >= $max_requests){
            return ['success'=>false,'reply'=>'Too many requests, please wait a moment.'];
        }
        set_transient($transient_key,$count+1,$period);

        // Automatska detekcija jezika
        $detected_lang = $this->detect_language($message) ?? $default_lang;

        // Embedding
        $msg_emb = $this->get_embedding($message,$api_key);
        if(!$msg_emb) return ['success'=>false,'reply'=>'Embedding failed.'];

        // Top chunks RAG
        $rows = $wpdb->get_results("SELECT chunk, embedding FROM $table");
        $top = [];
        foreach($rows as $r){
            $emb = json_decode($r->embedding,true);
            $sim = $this->cosine_similarity($msg_emb,$emb);
            $top[] = ['chunk'=>$r->chunk,'score'=>$sim];
        }
        usort($top,function($a,$b){return $b['score']<=>$a['score'];});
        $context = implode("\n",array_column(array_slice($top,0,3),'chunk'));

        $system_text = $system_prompt."\n".$tone."\nUser language: $detected_lang\nRelevant context:\n".$context;

        $payload = [
            'model'=>get_option(self::OPT_MODEL,'gpt-4o-mini'),
            'messages'=>[
                ['role'=>'system','content'=>$system_text],
                ['role'=>'user','content'=>$message]
            ],
            'temperature'=>0.3
        ];

        $res = wp_remote_post('https://api.openai.com/v1/chat/completions',[
            'headers'=>[
                'Authorization'=>'Bearer '.$api_key,
                'Content-Type'=>'application/json'
            ],
            'body'=>wp_json_encode($payload)
        ]);

        $json = json_decode(wp_remote_retrieve_body($res),true);
        $reply = $json['choices'][0]['message']['content'] ?? 'Unable to get reply.';

        return ['success'=>true,'reply'=>$reply];
    }

    private function detect_language($text){
        $hr_words = ['i','je','na','se','za','da','što','svi','mi'];
        $matches = 0;
        foreach($hr_words as $w){ if(stripos($text,' '.$w.' ')!==false) $matches++; }
        if($matches>=2) return 'hr';
        return null;
    }

    private function cosine_similarity($a,$b){
        $dot=0;$normA=0;$normB=0;
        for($i=0;$i<count($a);$i++){$dot+=$a[$i]*$b[$i];$normA+=$a[$i]*$a[$i];$normB+=$b[$i]*$b[$i];}
        return $dot/(sqrt($normA)*sqrt($normB)+1e-10);
    }

    // -------------------------------
    // Enqueue JS i Shortcode
    // -------------------------------
public function enqueue_assets(){
    // JS za chat
    wp_enqueue_script('wpoasc-chatbot-js', plugin_dir_url(__FILE__).'chatbot.js',['jquery'],null,true);
    wp_localize_script('wpoasc-chatbot-js','wpoasc_vars',['rest_url'=>rest_url('openai-chat/v1/message')]);

    // CSS za chat box i dot loader
    wp_enqueue_style('wpoasc-chatbot-css', plugin_dir_url(__FILE__).'chatbot.css');
}


    public function shortcode(){
        ob_start(); ?>
        <div id="wpoasc-chatbot-container">
            <div id="wpoasc-messages"></div>
            <input type="text" id="wpoasc-user-input" placeholder="Type your question...">
            <button id="wpoasc-send-btn">Send</button>
        </div>
        <script>
        jQuery(document).ready(function($){
            $('#wpoasc-send-btn').on('click',function(){
                var msg = $('#wpoasc-user-input').val();
                if(!msg) return;
                $('#wpoasc-messages').append('<div><strong>You:</strong> '+msg+'</div>');
                $('#wpoasc-user-input').val('');

                var typing = $('<div id="wpoasc-typing" class="dot-loader"><em>Bot is typing </em><span></span><span></span><span></span></div>');
                $('#wpoasc-messages').append(typing);
                $('#wpoasc-messages').scrollTop($('#wpoasc-messages')[0].scrollHeight);

                $.post(wpoasc_vars.rest_url,{message: msg}, function(resp){
                    $('#wpoasc-typing').remove();
                    if(resp.success){
                        $('#wpoasc-messages').append('<div><strong>Bot:</strong> '+resp.reply+'</div>');
                    } else {
                        $('#wpoasc-messages').append('<div style="color:red;"><strong>Error:</strong> '+resp.reply+'</div>');
                    }
                    $('#wpoasc-messages').scrollTop($('#wpoasc-messages')[0].scrollHeight);
                });
            });
        });
        </script>
        <?php
        return ob_get_clean();
    }
}

new WP_OpenAI_Support_Chatbot();

