<?php
/*
Plugin Name: WP OpenAI Support Chatbot
Description: AI chat support with RAG, language detection, tone, login-only option, enhanced indexing, retry mechanism, error log, and progress bar. SHORTCODE [openai_support_chatbot]
Version: 2.0
Author: Your Name
*/

if(!defined('ABSPATH')) exit;

class WP_OpenAI_Support_Chatbot {

    const OPT_KEY = 'wpoasc_api_key';
    const OPT_PROMPT = 'wpoasc_system_prompt';
    const OPT_TONE = 'wpoasc_tone';
    const OPT_SOURCES = 'wpoasc_sources';
    const OPT_MODEL = 'wpoasc_model';
    const OPT_LANG = 'wpoasc_default_lang';
    const OPT_LOGIN_ONLY = 'wpoasc_login_only';
    const OPT_LOGS = 'wpoasc_logs';

    public function __construct(){
        register_activation_hook(__FILE__, [$this,'activate']);
        register_deactivation_hook(__FILE__, [$this,'deactivate']);

        add_action('admin_menu', [$this,'settings_page']);
        add_action('admin_init', [$this,'register_settings']);

        add_action('rest_api_init', function(){
            register_rest_route('openai-chat/v1','message',[
                'methods'=>'POST',
                'callback'=>[$this,'handle_message'],
                'permission_callback'=> '__return_true'
            ]);
        });

        add_action('wp_enqueue_scripts', [$this,'enqueue_assets']);
        add_shortcode('openai_support_chatbot', [$this,'shortcode']);
    }

    public function activate(){
        global $wpdb;
        $charset = $wpdb->get_charset_collate();
        $table_emb = $wpdb->prefix.'wpoasc_embeddings';
        $table_logs = $wpdb->prefix.'wpoasc_logs';
        $sql_emb = "CREATE TABLE IF NOT EXISTS $table_emb (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            url TEXT NOT NULL,
            chunk LONGTEXT NOT NULL,
            embedding LONGTEXT NOT NULL,
            PRIMARY KEY (id)
        ) $charset;";
        $sql_logs = "CREATE TABLE IF NOT EXISTS $table_logs (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            time DATETIME NOT NULL,
            message TEXT NOT NULL,
            PRIMARY KEY (id)
        ) $charset;";
        require_once(ABSPATH.'wp-admin/includes/upgrade.php');
        dbDelta($sql_emb);
        dbDelta($sql_logs);
    }

    public function deactivate(){}

    // -------------------------------
    // Settings page & registration
    // -------------------------------
    public function settings_page(){
        add_options_page('OpenAI Chatbot','OpenAI Chatbot','manage_options','wpoasc_settings',[$this,'render_settings']);
    }

    public function register_settings(){
        register_setting('wpoasc_settings_group', self::OPT_KEY);
        register_setting('wpoasc_settings_group', self::OPT_PROMPT);
        register_setting('wpoasc_settings_group', self::OPT_TONE);
        register_setting('wpoasc_settings_group', self::OPT_SOURCES);
        register_setting('wpoasc_settings_group', self::OPT_MODEL);
        register_setting('wpoasc_settings_group', self::OPT_LANG);
        register_setting('wpoasc_settings_group', self::OPT_LOGIN_ONLY);

        add_settings_section('wpoasc_main','Main Settings', null, 'wpoasc_settings');

        add_settings_field(self::OPT_KEY,'OpenAI API Key', function(){
            $val = get_option(self::OPT_KEY,'');
            echo '<input type="text" name="'.self::OPT_KEY.'" value="'.esc_attr($val).'" size="50" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_PROMPT,'System Prompt', function(){
            $val = get_option(self::OPT_PROMPT,'');
            echo '<textarea name="'.self::OPT_PROMPT.'" rows="4" cols="50">'.esc_textarea($val).'</textarea>';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_TONE,'Tone / Style', function(){
            $val = get_option(self::OPT_TONE,'');
            echo '<input type="text" name="'.self::OPT_TONE.'" value="'.esc_attr($val).'" size="50" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_SOURCES,'Source URLs (one per line)', function(){
            $val = get_option(self::OPT_SOURCES,'');
            echo '<textarea name="'.self::OPT_SOURCES.'" rows="6" cols="50">'.esc_textarea($val).'</textarea>';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_MODEL,'Model', function(){
            $val = get_option(self::OPT_MODEL,'gpt-4o-mini');
            echo '<input type="text" name="'.self::OPT_MODEL.'" value="'.esc_attr($val).'" size="20" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_LANG,'Default Language', function(){
            $val = get_option(self::OPT_LANG,'en');
            echo '<input type="text" name="'.self::OPT_LANG.'" value="'.esc_attr($val).'" size="5" />';
        },'wpoasc_settings','wpoasc_main');

        add_settings_field(self::OPT_LOGIN_ONLY,'Login Only Chat', function(){
            $val = get_option(self::OPT_LOGIN_ONLY,0);
            echo '<input type="checkbox" name="'.self::OPT_LOGIN_ONLY.'" value="1" '.checked(1,$val,false).' /> Allow chat only for logged-in users';
        },'wpoasc_settings','wpoasc_main');
    }


    // -------------------------------
    // Render settings, action buttons, indexed URLs & error logs
    // -------------------------------
    public function render_settings(){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $table_logs = $wpdb->prefix.'wpoasc_logs';

        // Handle actions
        if(isset($_POST['wpoasc_delete_all'])){
            $wpdb->query("TRUNCATE TABLE $table");
            $this->log_event("All indexed pages deleted manually.");
            echo '<div class="notice notice-success is-dismissible"><p>All indexed pages deleted.</p></div>';
        }

        if(isset($_POST['wpoasc_update_existing'])){
            echo '<div style="margin-bottom:10px;">Updating existing pages...<progress id="wpoasc-progress" max="100" value="0" style="width:100%;"></progress></div>';
            flush();
            $result = $this->update_existing_index(true);
            echo '<div class="notice notice-success is-dismissible"><p>'.$result['message'].'</p></div>';
        }

        if(isset($_POST['wpoasc_add_new'])){
            echo '<div style="margin-bottom:10px;">Adding new pages...<progress id="wpoasc-progress" max="100" value="0" style="width:100%;"></progress></div>';
            flush();
            $result = $this->add_new_pages_index(true);
            echo '<div class="notice notice-success is-dismissible"><p>'.$result['message'].'</p></div>';
        }

        // Standard settings form
        echo '<form method="post" action="options.php">';
        settings_fields('wpoasc_settings_group');
        do_settings_sections('wpoasc_settings');
        submit_button();
        echo '</form>';

        // Indexing action buttons
        echo '<hr>';
        echo '<form method="post">';
        echo '<input type="submit" name="wpoasc_delete_all" class="button button-secondary" value="Delete All Indexed Pages">';
        echo ' ';
        echo '<input type="submit" name="wpoasc_update_existing" class="button button-primary" value="Update Existing Indexed Pages">';
        echo ' ';
        echo '<input type="submit" name="wpoasc_add_new" class="button button-primary" value="Add New Pages to Index">';
        echo '</form>';

        // -------------------------------
        // Indexed URLs table with update/delete
        // -------------------------------
        $urls = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        echo '<h3>Indexed URLs</h3>';
        echo '<table class="widefat"><tr><th>URL</th><th>Actions</th></tr>';
        foreach($urls as $url){
            echo '<tr>';
            echo '<td>'.esc_html($url).'</td>';
            echo '<td>';
            echo '<form method="post" style="display:inline-block;">';
            echo '<input type="hidden" name="wpoasc_single_url" value="'.esc_attr($url).'">';
            echo '<input type="submit" name="wpoasc_update_single" class="button button-primary" value="Update">';
            echo '</form> ';
            echo '<form method="post" style="display:inline-block;">';
            echo '<input type="hidden" name="wpoasc_single_url" value="'.esc_attr($url).'">';
            echo '<input type="submit" name="wpoasc_delete_single" class="button button-secondary" value="Delete">';
            echo '</form>';
            echo '</td></tr>';
        }
        echo '</table>';

        // Handle single URL update/delete
        if(isset($_POST['wpoasc_delete_single']) && !empty($_POST['wpoasc_single_url'])){
            $wpdb->delete($table,['url'=>sanitize_text_field($_POST['wpoasc_single_url'])]);
            $this->log_event("Deleted indexed URL: ".$_POST['wpoasc_single_url']);
            echo '<div class="notice notice-success is-dismissible"><p>URL deleted.</p></div>';
        }
        if(isset($_POST['wpoasc_update_single']) && !empty($_POST['wpoasc_single_url'])){
            $this->update_single_url($_POST['wpoasc_single_url'],true);
            echo '<div class="notice notice-success is-dismissible"><p>URL updated.</p></div>';
        }

        // -------------------------------
        // Show last 500 logs
        // -------------------------------
        $logs = $wpdb->get_results("SELECT time,message FROM $table_logs ORDER BY id DESC LIMIT 500");
        echo '<h3>Chatbot Retry/Error Logs (last 500)</h3>';
        echo '<div style="max-height:300px;overflow:auto;background:#f1f1f1;padding:5px;">';
        foreach($logs as $log){
            echo '<div>['.$log->time.'] '.esc_html($log->message).'</div>';
        }
        echo '</div>';
    }

    private function log_event($msg){
        global $wpdb;
        $table_logs = $wpdb->prefix.'wpoasc_logs';
        $wpdb->insert($table_logs,['time'=>current_time('mysql'),'message'=>$msg]);
    }


    // -------------------------------
    // Handle chat message with retry
    // -------------------------------
    public function handle_message($req){
        $login_only = get_option(self::OPT_LOGIN_ONLY,0);
        if($login_only && !is_user_logged_in()){
            return ['success'=>false,'reply'=>'You must be logged in to use the chat.'];
        }

        $message = trim((string)$req->get_param('message'));
        $api_key = get_option(self::OPT_KEY);
        $system_prompt = get_option(self::OPT_PROMPT);
        $tone = get_option(self::OPT_TONE);
        $default_lang = get_option(self::OPT_LANG,'en');

        if(!$api_key) return ['success'=>false,'reply'=>'API key not set'];

        // RATE LIMIT
        $ip = $_SERVER['REMOTE_ADDR'];
        $transient_key = 'wpoasc_rate_'.$ip;
        $count = get_transient($transient_key) ?: 0;
        if($count >= 5) return ['success'=>false,'reply'=>'Too many requests, please wait a moment.'];
        set_transient($transient_key,$count+1,30);

        // Language detect
        $detected_lang = $this->detect_language($message) ?? $default_lang;

        // Embedding
        $msg_emb = $this->get_embedding($message,$api_key);
        if(!$msg_emb) return ['success'=>false,'reply'=>'Embedding failed.'];

        // Top chunks RAG
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $rows = $wpdb->get_results("SELECT chunk, embedding FROM $table");
        $top = [];
        foreach($rows as $r){
            $emb = json_decode($r->embedding,true);
            $sim = $this->cosine_similarity($msg_emb,$emb);
            $top[] = ['chunk'=>$r->chunk,'score'=>$sim];
        }
        usort($top,function($a,$b){return $b['score']<=>$a['score'];});
        $context = implode("\n",array_column(array_slice($top,0,3),'chunk'));

        $system_text = $system_prompt."\n".$tone."\nUser language: $detected_lang\nRelevant context:\n".$context;

        $payload = [
            'model'=>get_option(self::OPT_MODEL,'gpt-4o-mini'),
            'messages'=>[
                ['role'=>'system','content'=>$system_text],
                ['role'=>'user','content'=>$message]
            ],
            'temperature'=>0.3
        ];

        // Retry mechanism
        $reply = $this->call_chat_api($api_key,$payload);

        return ['success'=>true,'reply'=>$reply];
    }

    private function call_chat_api($api_key,$payload){
        $attempts = 0; $max_attempts = 2;
        $reply = 'Unable to get reply.';
        while($attempts<$max_attempts){
            $res = wp_remote_post('https://api.openai.com/v1/chat/completions',[
                'headers'=>['Authorization'=>'Bearer '.$api_key,'Content-Type'=>'application/json'],
                'body'=>wp_json_encode($payload),
                'timeout'=>30
            ]);
            $json = json_decode(wp_remote_retrieve_body($res),true);
            if(isset($json['choices'][0]['message']['content'])){
                $reply = $json['choices'][0]['message']['content'];
                break;
            } else {
                $this->log_event("Retry #".($attempts+1)." due to API failure.");
                $attempts++;
            }
        }
        return $reply;
    }

    // -------------------------------
    // Helper functions
    // -------------------------------
    private function detect_language($text){
        $hr_words = ['i','je','na','se','za','da','što','svi','mi'];
        $matches = 0; foreach($hr_words as $w){ if(stripos($text,' '.$w.' ')!==false) $matches++; }
        if($matches>=2) return 'hr';
        return null;
    }

    private function cosine_similarity($a,$b){
        $dot=0;$normA=0;$normB=0;
        for($i=0;$i<count($a);$i++){$dot+=$a[$i]*$b[$i];$normA+=$a[$i]*$a[$i];$normB+=$b[$i]*$b[$i];}
        return $dot/(sqrt($normA)*sqrt($normB)+1e-10);
    }

    // -------------------------------
    // Indexing helpers
    // -------------------------------
    private function update_single_url($url,$show_progress=false){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        if(!$api_key) return;
        $res = wp_remote_get($url,['timeout'=>15]);
        if(is_wp_error($res)) return;
        $text = wp_strip_all_tags(wp_remote_retrieve_body($res));
        $chunks = str_split($text,800);
        $wpdb->delete($table,['url'=>$url]);
        $total = count($chunks); $i=0;
        foreach($chunks as $chunk){
            $embedding = $this->get_embedding($chunk,$api_key);
            if($embedding) $wpdb->insert($table,['url'=>$url,'chunk'=>$chunk,'embedding'=>json_encode($embedding)]);
            $i++; if($show_progress) echo "<script>document.getElementById('wpoasc-progress').value=".intval($i/$total*100).";</script>"; flush();
        }
        $this->log_event("Updated single URL: $url");
    }

    private function update_existing_index($show_progress=false){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        $urls = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        $total_urls = count($urls); $idx=0;
        foreach($urls as $url){
            $this->update_single_url($url,$show_progress);
            $idx++; if($show_progress) echo "<script>document.getElementById('wpoasc-progress').value=".intval($idx/$total_urls*100).";</script>"; flush();
        }
        return ['success'=>true,'message'=>'Existing indexed pages updated.'];
    }

    private function add_new_pages_index($show_progress=false){
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        $sources = array_filter(array_map('trim', explode("\n", get_option(self::OPT_SOURCES))));
        $existing = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        $added = 0; $total_urls = count($sources); $idx=0;
        foreach($sources as $url){
            if(in_array($url,$existing)) { $idx++; continue; }
            $this->update_single_url($url,$show_progress);
            $added++; $idx++; if($show_progress) echo "<script>document.getElementById('wpoasc-progress').value=".intval($idx/$total_urls*100).";</script>"; flush();
        }
        return ['success'=>true,'message'=>"Added $added new page(s) to index."];
    }

    private function get_embedding($text,$api_key){
        $body = ['input'=>$text,'model'=>'text-embedding-3-small'];
        $res = wp_remote_post('https://api.openai.com/v1/embeddings',[
            'headers'=>['Authorization'=>'Bearer '.$api_key,'Content-Type'=>'application/json'],
            'body'=>wp_json_encode($body),
            'timeout'=>30
        ]);
        if(is_wp_error($res)) return null;
        $data = json_decode(wp_remote_retrieve_body($res),true);
        return $data['data'][0]['embedding'] ?? null;
    }

    // -------------------------------
    // Enqueue JS/CSS & shortcode
    // -------------------------------
    public function enqueue_assets(){
        wp_enqueue_script('wpoasc-chatbot-js', plugin_dir_url(__FILE__).'chatbot.js',['jquery'],null,true);
        wp_localize_script('wpoasc-chatbot-js','wpoasc_vars',['rest_url'=>rest_url('openai-chat/v1/message')]);
        wp_enqueue_style('wpoasc-chatbot-css', plugin_dir_url(__FILE__).'chatbot.css');
    }

    public function shortcode(){
        ob_start(); ?>
        <div id="wpoasc-chatbot-container">
            <div id="wpoasc-messages"></div>
            <input type="text" id="wpoasc-user-input" placeholder="Type your question...">
            <button id="wpoasc-send-btn">Send</button>
        </div>
        <?php
        return ob_get_clean();
    }
}

new WP_OpenAI_Support_Chatbot();

