<?php
/*
Plugin Name: WP OpenAI Support Chatbot
Description: AI chat support with RAG, language detection, tone, error log, and progress bar. SHORTCODE [openai_support_chatbot]
Version: 2.4.7
Author: Molly9
*/

if(!defined('ABSPATH')) exit;

class WP_OpenAI_Support_Chatbot {

    const OPT_KEY          = 'wpoasc_api_key';
    const OPT_PROMPT       = 'wpoasc_system_prompt';
    const OPT_TONE         = 'wpoasc_tone';
    const OPT_SOURCES      = 'wpoasc_sources';
    const OPT_MODEL        = 'wpoasc_model';
    const OPT_LANG         = 'wpoasc_default_lang';
    const OPT_LOGIN_ONLY   = 'wpoasc_login_only';

    // UI / chatbot appearance
    const OPT_BOT_NAME      = 'wpoasc_bot_name';
    const OPT_SEND_LABEL    = 'wpoasc_send_label';
    const OPT_PRIMARY_COLOR = 'wpoasc_primary_color';
    const OPT_THEME_MODE    = 'wpoasc_theme_mode'; // dark|light
    const OPT_HEADER_TEXT   = 'wpoasc_header_text'; // subtitle in header

    // Rate limiting
    const OPT_RATE_MAX     = 'wpoasc_rate_max';
    const OPT_RATE_WINDOW  = 'wpoasc_rate_window'; // seconds

    public function __construct(){
        register_activation_hook(__FILE__, [$this,'activate']);
        register_deactivation_hook(__FILE__, [$this,'deactivate']);

        add_action('admin_menu', [$this,'settings_page']);
        add_action('admin_init', [$this,'register_settings']);

        add_action('rest_api_init', function(){
            register_rest_route('openai-chat/v1','message',[
                'methods'=>'POST',
                'callback'=>[$this,'handle_message'],
                'permission_callback'=> '__return_true'
            ]);
        });

        add_action('wp_enqueue_scripts', [$this,'enqueue_assets']);
        add_shortcode('openai_support_chatbot', [$this,'shortcode']);
    }

    // -------------------------------
    // Table creation helper
    // -------------------------------
    private function maybe_create_tables(){
        global $wpdb;
        $charset    = $wpdb->get_charset_collate();
        $table_emb  = $wpdb->prefix.'wpoasc_embeddings';
        $table_logs = $wpdb->prefix.'wpoasc_logs';

        require_once(ABSPATH.'wp-admin/includes/upgrade.php');

        // Embeddings table
        $sql_emb = "CREATE TABLE IF NOT EXISTS $table_emb (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            url TEXT NOT NULL,
            chunk LONGTEXT NOT NULL,
            embedding LONGTEXT NOT NULL,
            PRIMARY KEY (id)
        ) $charset;";
        dbDelta($sql_emb);

        // LOGS – čista schema s log_time
        $logs_exists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $table_logs)
        );

        if ($logs_exists === $table_logs) {
            $cols = $wpdb->get_col("SHOW COLUMNS FROM $table_logs", 0);
            $needed = ['log_time','message','session_id','user_id','role'];

            $missing = false;
            foreach($needed as $c){
                if(!in_array($c,$cols,true)){
                    $missing = true;
                    break;
                }
            }

            // ako nema log_time, ali ima time -> rename
            if(!$missing && !in_array('log_time',$cols,true) && in_array('time',$cols,true)){
                $wpdb->query("ALTER TABLE $table_logs CHANGE `time` `log_time` DATETIME NOT NULL");
                $cols = $wpdb->get_col("SHOW COLUMNS FROM $table_logs", 0);
                if(!in_array('log_time',$cols,true)){
                    $missing = true;
                }
            }

            if($missing){
                // kriva struktura – drop pa napravi novu
                $wpdb->query("DROP TABLE $table_logs");
                $logs_exists = null;
            }
        }

        $sql_logs = "CREATE TABLE $table_logs (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            log_time DATETIME NOT NULL,
            message TEXT NOT NULL,
            session_id VARCHAR(64) NULL,
            user_id BIGINT UNSIGNED NULL,
            role VARCHAR(20) NULL,
            PRIMARY KEY (id),
            KEY session_idx (session_id)
        ) $charset;";

        dbDelta($sql_logs);
    }

    public function activate(){
        $this->maybe_create_tables();
    }

    public function deactivate(){}

    // -------------------------------
    // Settings page & registration
    // -------------------------------
    public function settings_page(){
        add_options_page(
            'OpenAI Chatbot',
            'OpenAI Chatbot',
            'manage_options',
            'wpoasc_settings',
            [$this,'render_settings']
        );
    }

    public function register_settings(){
        // Register options (storage)
        register_setting('wpoasc_settings_group', self::OPT_KEY);
        register_setting('wpoasc_settings_group', self::OPT_PROMPT);
        register_setting('wpoasc_settings_group', self::OPT_TONE);
        register_setting('wpoasc_settings_group', self::OPT_SOURCES);
        register_setting('wpoasc_settings_group', self::OPT_MODEL);
        register_setting('wpoasc_settings_group', self::OPT_LANG);
        register_setting('wpoasc_settings_group', self::OPT_LOGIN_ONLY);

        // UI / theme
        register_setting('wpoasc_settings_group', self::OPT_BOT_NAME);
        register_setting('wpoasc_settings_group', self::OPT_SEND_LABEL);
        register_setting('wpoasc_settings_group', self::OPT_PRIMARY_COLOR);
        register_setting('wpoasc_settings_group', self::OPT_THEME_MODE);
        register_setting('wpoasc_settings_group', self::OPT_HEADER_TEXT);

        // Rate limiting
        register_setting('wpoasc_settings_group', self::OPT_RATE_MAX);
        register_setting('wpoasc_settings_group', self::OPT_RATE_WINDOW);

        // Sections: General, Appearance, Rate limiting
        add_settings_section('wpoasc_general','General Settings', function(){
            echo '<p>Core configuration for the chatbot engine and data sources.</p>';
        }, 'wpoasc_settings');

        add_settings_section('wpoasc_ui','Appearance', function(){
            echo '<p>Customize how the chat widget looks and feels on the front-end.</p>';
        }, 'wpoasc_settings');

        add_settings_section('wpoasc_rate','Rate Limiting', function(){
            echo '<p>Protect your API quota by limiting how many messages users can send in a short period.</p>';
        }, 'wpoasc_settings');

        // ---------------------------
        // GENERAL SECTION FIELDS
        // ---------------------------
        add_settings_field(self::OPT_KEY,'OpenAI API Key', function(){
            $val = get_option(self::OPT_KEY,'');
            echo '<input type="text" name="'.self::OPT_KEY.'" value="'.esc_attr($val).'" size="50" />';
            echo '<p class="description">Enter your OpenAI API key. Required for chatbot functionality.</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_PROMPT,'System Prompt', function(){
            $val = get_option(self::OPT_PROMPT,'');
            echo '<textarea name="'.self::OPT_PROMPT.'" rows="4" cols="60">'.esc_textarea($val).'</textarea>';
            echo '<p class="description">Defines the base personality and behavior of the assistant.</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_TONE,'Tone / Style', function(){
            $val = get_option(self::OPT_TONE,'');
            echo '<input type="text" name="'.self::OPT_TONE.'" value="'.esc_attr($val).'" size="40" />';
            echo '<p class="description">Optional tone instructions (friendly, professional, humorous, etc.).</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_SOURCES,'Source URLs (one per line)', function(){
            $val = get_option(self::OPT_SOURCES,'');
            echo '<textarea name="'.self::OPT_SOURCES.'" rows="6" cols="60">'.esc_textarea($val).'</textarea>';
            echo '<p class="description">List URLs (one per line) that should be indexed for RAG search.</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_MODEL,'Model', function(){
            $val = get_option(self::OPT_MODEL,'gpt-4o-mini');
            echo '<input type="text" name="'.self::OPT_MODEL.'" value="'.esc_attr($val).'" size="20" />';
            echo '<p class="description">OpenAI model to use for chat responses.</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_LANG,'Default Language (2-letter code)', function(){
            $val = get_option(self::OPT_LANG,'en');
            echo '<input type="text" name="'.self::OPT_LANG.'" value="'.esc_attr($val).'" size="5" />';
            echo '<p class="description">Two-letter language code (e.g., en, hr, de, fr). Used for the welcome message and default header text.</p>';
        },'wpoasc_settings','wpoasc_general');

        add_settings_field(self::OPT_LOGIN_ONLY,'Login Only Chat', function(){
            $val = get_option(self::OPT_LOGIN_ONLY,0);
            echo '<label><input type="checkbox" name="'.self::OPT_LOGIN_ONLY.'" value="1" '.checked(1,$val,false).' /> Restrict chatbot usage to logged-in users only.</label>';
        },'wpoasc_settings','wpoasc_general');

        // ---------------------------
        // APPEARANCE SECTION FIELDS
        // ---------------------------
        add_settings_field(self::OPT_BOT_NAME,'Bot Name', function(){
            $val = get_option(self::OPT_BOT_NAME,'Support Bot');
            echo '<input type="text" name="'.self::OPT_BOT_NAME.'" value="'.esc_attr($val).'" size="30" />';
            echo '<p class="description">Displayed as the bot’s name in the chat header.</p>';
        },'wpoasc_settings','wpoasc_ui');

        add_settings_field(self::OPT_SEND_LABEL,'Send Button Text', function(){
            $val = get_option(self::OPT_SEND_LABEL,'Send');
            echo '<input type="text" name="'.self::OPT_SEND_LABEL.'" value="'.esc_attr($val).'" size="20" />';
            echo '<p class="description">Text displayed on the send button.</p>';
        },'wpoasc_settings','wpoasc_ui');

        add_settings_field(self::OPT_PRIMARY_COLOR,'Primary Color (hex)', function(){
            $val = get_option(self::OPT_PRIMARY_COLOR,'#22d3ee');
            echo '<input type="text" name="'.self::OPT_PRIMARY_COLOR.'" value="'.esc_attr($val).'" size="10" placeholder="#22d3ee" />';
            echo '<p class="description">Main accent color used for highlights and buttons.</p>';
        },'wpoasc_settings','wpoasc_ui');

        add_settings_field(self::OPT_THEME_MODE,'Theme Mode', function(){
            $val = get_option(self::OPT_THEME_MODE,'dark');
            echo '<select name="'.self::OPT_THEME_MODE.'">';
            echo '<option value="dark"'.selected($val,'dark',false).'>Dark</option>';
            echo '<option value="light"'.selected($val,'light',false).'>Light</option>';
            echo '</select>';
            echo '<p class="description">Choose between light or dark theme for the chat widget.</p>';
        },'wpoasc_settings','wpoasc_ui');

        add_settings_field(self::OPT_HEADER_TEXT,'Header Subtitle Text', function(){
            $saved = get_option(self::OPT_HEADER_TEXT,'');
            $lang  = get_option(self::OPT_LANG,'en');
            $default = '';

            if($saved === ''){
                switch(strtolower($lang)){
                    case 'hr':
                        $default = 'Postavite mi bilo koje pitanje o ovoj stranici.';
                        break;
                    case 'de':
                        $default = 'Stellen Sie mir jede Frage zu dieser Website.';
                        break;
                    case 'es':
                        $default = 'Hazme cualquier pregunta sobre este sitio.';
                        break;
                    case 'fr':
                        $default = 'Posez-moi n\'importe quelle question sur ce site.';
                        break;
                    case 'it':
                        $default = 'Fammi qualsiasi domanda su questo sito.';
                        break;
                    default:
                        $default = 'Ask me anything about this site.';
                        break;
                }
                $val = $default;
            } else {
                $val = $saved;
            }

            echo '<input type="text" name="'.self::OPT_HEADER_TEXT.'" value="'.esc_attr($val).'" size="60" />';
            echo '<p class="description">This text appears in the header below the bot’s name. You can change it manually.</p>';
        },'wpoasc_settings','wpoasc_ui');

        // ---------------------------
        // RATE LIMIT SECTION FIELDS
        // ---------------------------
        add_settings_field(self::OPT_RATE_MAX,'Max Messages per Window', function(){
            $val = (int) get_option(self::OPT_RATE_MAX,5);
            echo '<input type="number" min="1" name="'.self::OPT_RATE_MAX.'" value="'.esc_attr($val).'" />';
            echo '<p class="description">Maximum number of messages a user can send within the rate-limit window.</p>';
        },'wpoasc_settings','wpoasc_rate');

        add_settings_field(self::OPT_RATE_WINDOW,'Rate Limit Window (seconds)', function(){
            $val = (int) get_option(self::OPT_RATE_WINDOW,30);
            echo '<input type="number" min="1" name="'.self::OPT_RATE_WINDOW.'" value="'.esc_attr($val).'" />';
            echo '<p class="description">Duration of the rate-limit window in seconds.</p>';
        },'wpoasc_settings','wpoasc_rate');
    }


    // -------------------------------
    // Render settings & admin UI
    // -------------------------------
    public function render_settings(){
        if(!current_user_can('manage_options')) return;

        // Ensure tables exist
        $this->maybe_create_tables();

        global $wpdb;
        $table      = $wpdb->prefix.'wpoasc_embeddings';
        $table_logs = $wpdb->prefix.'wpoasc_logs';

        echo '<div class="wrap wpoasc-settings">';
        echo '<h1>WP OpenAI Support Chatbot</h1>';

        // Light styling for a more modern layout
        echo '<style>
            .wpoasc-settings .wpoasc-settings-grid{
                display:grid;
                grid-template-columns:minmax(0, 2fr);
                gap:20px;
                margin-top:15px;
                max-width:960px;
            }
            .wpoasc-settings .wpoasc-card{
                background:#fff;
                border:1px solid #e5e7eb;
                border-radius:8px;
                padding:18px 20px;
                box-shadow:0 1px 2px rgba(15,23,42,.08);
            }
            .wpoasc-settings .wpoasc-card h2{
                margin-top:0;
                margin-bottom:8px;
            }
            .wpoasc-settings .wpoasc-card p.description{
                margin-top:4px;
                color:#6b7280;
            }
            .wpoasc-settings .form-table{
                margin-top:10px;
            }
            .wpoasc-settings .form-table th{
                width:220px;
                padding-top:12px;
            }
            .wpoasc-settings .form-table td{
                padding-top:8px;
            }
            .wpoasc-settings textarea{
                width:100%;
                max-width:580px;
            }
            .wpoasc-settings input[type="text"],
            .wpoasc-settings input[type="number"],
            .wpoasc-settings select{
                max-width:320px;
            }
            .wpoasc-settings .wpoasc-index-table{
                max-height:260px;
                overflow:auto;
            }
            .wpoasc-settings .wpoasc-logs-box{
                max-height:300px;
                overflow:auto;
                background:#ffffff;
                border:1px solid #e5e7eb;
                padding:10px;
                font-family:monospace;
                font-size:12px;
            }
            .wpoasc-settings .wpoasc-debug{
                margin-top:8px;
                font-size:11px;
                color:#6b7280;
            }
            .wpoasc-settings .wpoasc-logs-actions{
                margin-top:8px;
            }
        </style>';

        echo '<div class="wpoasc-settings-grid">';

        // SETTINGS FORM CARD
        echo '<div class="wpoasc-card">';
        echo '<h2>Configuration</h2>';
        echo '<p class="description">Control how the chatbot behaves, looks, and protects your API usage.</p>';

        echo '<form method="post" action="options.php">';
        settings_fields('wpoasc_settings_group');
        do_settings_sections('wpoasc_settings');
        submit_button('Save Settings');
        echo '</form>';
        echo '</div>';

        // INDEX + LOGS CARD
        echo '<div class="wpoasc-card">';
        echo '<h2>Index Management & Logs</h2>';
        echo '<p class="description">Manage indexed URLs for RAG and inspect recent events and chat transcripts.</p>';

        // Handle actions (index operations)
        if(isset($_POST['wpoasc_full_reindex'])){
            check_admin_referer('wpoasc_reindex_action','wpoasc_reindex_nonce');
            $res = $this->full_reindex(true);
            echo '<div class="notice notice-success is-dismissible"><p>'.esc_html($res['message']).'</p></div>';
        }

        if(isset($_POST['wpoasc_update_existing'])){
            check_admin_referer('wpoasc_update_action','wpoasc_update_nonce');
            $res = $this->update_existing_index(true);
            echo '<div class="notice notice-success is-dismissible"><p>'.esc_html($res['message']).'</p></div>';
        }

        if(isset($_POST['wpoasc_add_new'])){
            check_admin_referer('wpoasc_add_action','wpoasc_add_nonce');
            $res = $this->add_new_pages_index(true);
            echo '<div class="notice notice-success is-dismissible"><p>'.esc_html($res['message']).'</p></div>';
        }

        if(isset($_POST['wpoasc_delete_index'])){
            check_admin_referer('wpoasc_delete_action','wpoasc_delete_nonce');
            $wpdb->query("TRUNCATE TABLE $table");
            $this->log_event("All indexed pages deleted manually.");
            echo '<div class="notice notice-success is-dismissible"><p>All indexed pages deleted.</p></div>';
        }

        // NEW: clear logs
        if(isset($_POST['wpoasc_clear_logs'])){
            check_admin_referer('wpoasc_clear_logs_action','wpoasc_clear_logs_nonce');
            $wpdb->query("TRUNCATE TABLE $table_logs");
            echo '<div class="notice notice-success is-dismissible"><p>All logs have been cleared.</p></div>';
        }

        // Index action buttons
        echo '<h3>Index Management</h3>';

        echo '<form method="post" style="margin-bottom:8px;">';
        wp_nonce_field('wpoasc_reindex_action','wpoasc_reindex_nonce');
        echo '<input type="submit" name="wpoasc_full_reindex" class="button button-primary" value="Full Reindex (drop + index all sources)" />';
        echo '</form>';

        echo '<form method="post" style="margin-bottom:8px;">';
        wp_nonce_field('wpoasc_update_action','wpoasc_update_nonce');
        echo '<input type="submit" name="wpoasc_update_existing" class="button" value="Update Existing Indexed Pages" />';
        echo '</form>';

        echo '<form method="post" style="margin-bottom:8px;">';
        wp_nonce_field('wpoasc_add_action','wpoasc_add_nonce');
        echo '<input type="submit" name="wpoasc_add_new" class="button" value="Add New Pages Only" />';
        echo '</form>';

        echo '<form method="post" style="margin-bottom:8px;">';
        wp_nonce_field('wpoasc_delete_action','wpoasc_delete_nonce');
        echo '<input type="submit" name="wpoasc_delete_index" class="button button-secondary" value="Delete Entire Index" />';
        echo '</form>';

        echo '<div style="margin-top:10px;margin-bottom:12px;">';
        echo '<label>Indexing Progress:</label><br />';
        echo '<progress id="wpoasc_progress" value="0" max="100" style="width:100%;max-width:300px;"></progress>';
        echo '</div>';

        // Per-URL management
        $urls = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        echo '<h3>Indexed URLs</h3>';
        echo '<div class="wpoasc-index-table">';
        echo '<table class="widefat striped"><thead><tr><th>URL</th><th style="width:160px;">Actions</th></tr></thead><tbody>';
        if($urls){
            foreach($urls as $url){
                echo '<tr>';
                echo '<td>'.esc_html($url).'</td>';
                echo '<td>';
                echo '<form method="post" style="display:inline-block;margin-right:4px;">';
                echo '<input type="hidden" name="wpoasc_single_url" value="'.esc_attr($url).'">';
                echo '<input type="submit" name="wpoasc_update_single" class="button button-small button-primary" value="Update" />';
                echo '</form>';
                echo '<form method="post" style="display:inline-block;">';
                echo '<input type="hidden" name="wpoasc_single_url" value="'.esc_attr($url).'">';
                echo '<input type="submit" name="wpoasc_delete_single" class="button button-small button-secondary" value="Delete" />';
                echo '</form>';
                echo '</td></tr>';
            }
        } else {
            echo '<tr><td colspan="2">No URLs indexed yet.</td></tr>';
        }
        echo '</tbody></table>';
        echo '</div>';

        if(isset($_POST['wpoasc_delete_single']) && !empty($_POST['wpoasc_single_url'])){
            $wpdb->delete($table,['url'=>sanitize_text_field($_POST['wpoasc_single_url'])]);
            $this->log_event("Deleted indexed URL: ".$_POST['wpoasc_single_url']);
            echo '<div class="notice notice-success is-dismissible" style="margin-top:8px;"><p>URL deleted.</p></div>';
        }

        if(isset($_POST['wpoasc_update_single']) && !empty($_POST['wpoasc_single_url'])){
            $url = esc_url_raw($_POST['wpoasc_single_url']);
            $this->update_single_url($url,true);
            $this->log_event("Updated single URL: $url");
            echo '<div class="notice notice-success is-dismissible" style="margin-top:8px;"><p>URL updated.</p></div>';
        }

        // Logs
        echo '<h3 style="margin-top:18px;">Logs (events + chat transcripts)</h3>';

        $logs_table_exists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $table_logs)
        );

        if($logs_table_exists !== $table_logs){
            echo '<div class="wpoasc-logs-box"><div>Logs table does not exist ('.$table_logs.').</div></div>';
        } else {
            $logs = $wpdb->get_results("SELECT log_time, message, session_id, user_id, role FROM $table_logs ORDER BY id DESC LIMIT 500");
            echo '<div class="wpoasc-logs-box">';
            if($logs){
                foreach($logs as $log){
                    $meta = [];
                    if($log->session_id){
                        $meta[] = 'session: '.$log->session_id;
                    }
                    if($log->user_id){
                        $meta[] = 'user_id: '.$log->user_id;
                    }
                    if($log->role){
                        $meta[] = 'role: '.$log->role;
                    }
                    $meta_str = $meta ? ' ['.esc_html(implode(' | ',$meta)).']' : '';
                    echo '<div>['.$log->log_time.'] '.esc_html($log->message).$meta_str.'</div>';
                }
            } else {
                echo '<div>No logs yet.</div>';
            }
            echo '</div>';

            // Clear logs button
            echo '<div class="wpoasc-logs-actions">';
            echo '<form method="post">';
            wp_nonce_field('wpoasc_clear_logs_action','wpoasc_clear_logs_nonce');
            echo '<input type="submit" name="wpoasc_clear_logs" class="button button-secondary" value="Clear all logs" />';
            echo '</form>';
            echo '</div>';

            // debug info
            $count_logs = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table_logs");
            $last_error = $wpdb->last_error;
            echo '<div class="wpoasc-debug">';
            echo 'Log rows in DB: '.esc_html($count_logs).'. ';
            if($last_error){
                echo 'Last DB error: '.esc_html($last_error);
            }
            echo '</div>';
        }

        echo '</div>'; // card
        echo '</div>'; // settings grid
        echo '</div>'; // wrap
    }

    private function log_event($msg, $session_id = null, $role = null, $user_id = null){
        global $wpdb;
        $this->maybe_create_tables();

        $table_logs = $wpdb->prefix.'wpoasc_logs';

        $data = [
            'log_time' => current_time('mysql'),
            'message'  => $msg,
        ];

        if( $session_id ){
            $data['session_id'] = substr( (string) $session_id, 0, 64 );
        }

        if( null !== $user_id ){
            $data['user_id'] = (int) $user_id;
        }

        if( $role ){
            $data['role'] = substr( (string) $role, 0, 20 );
        }

        $wpdb->insert($table_logs, $data);
    }


    // -------------------------------
    // Handle chat message
    // -------------------------------
    public function handle_message($req){
        $login_only = get_option(self::OPT_LOGIN_ONLY,0);
        if($login_only && !is_user_logged_in()){
            return ['success'=>false,'reply'=>'You must be logged in to use the chat.'];
        }

        // Ensure tables exist
        $this->maybe_create_tables();

        $message      = trim((string)$req->get_param('message'));
        $api_key      = get_option(self::OPT_KEY);
        $system_prompt= get_option(self::OPT_PROMPT);
        $tone         = get_option(self::OPT_TONE);
        $default_lang = get_option(self::OPT_LANG,'en');

        // Session / user info
        $session_id = sanitize_text_field( $req->get_param('session_id') ?: '' );
        if( ! $session_id ){
            $session_id = wp_generate_uuid4();
        }
        $user_id = get_current_user_id();

        if(!$api_key) return ['success'=>false,'reply'=>'API key not set'];

        if( $message ){
            $this->log_event('User: '.$message, $session_id, 'user', $user_id);
        }

        // RATE LIMIT
        $ip      = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'unknown';
        $max     = (int) get_option(self::OPT_RATE_MAX, 5);
        $window  = (int) get_option(self::OPT_RATE_WINDOW, 30);
        if($max < 1){ $max = 5; }
        if($window < 1){ $window = 30; }

        $transient_key = 'wpoasc_rate_'.$ip;
        $count = (int) get_transient($transient_key);
        if($count >= $max){
            $this->log_event('Rate limit hit from IP '.$ip, $session_id, 'system', $user_id);
            return ['success'=>false,'reply'=>'Too many requests, please wait a moment.'];
        }
        set_transient($transient_key, $count+1, $window);

        // Language detect
        $detected_lang = $this->detect_language($message) ?? $default_lang;

        // Embedding
        $msg_emb = $this->get_embedding($message,$api_key);
        if(!$msg_emb) return ['success'=>false,'reply'=>'Embedding failed.'];

        // RAG
        global $wpdb;
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $rows = $wpdb->get_results("SELECT chunk, embedding FROM $table");
        $top = [];
        foreach($rows as $r){
            $emb = json_decode($r->embedding,true);
            $sim = $this->cosine_similarity($msg_emb,$emb);
            $top[] = ['chunk'=>$r->chunk,'score'=>$sim];
        }
        usort($top,function($a,$b){return $b['score']<=>$a['score'];});
        $context = implode("\n",array_column(array_slice($top,0,3),'chunk'));

        $system_text = $system_prompt."\n".$tone."\nUser language: $detected_lang\nRelevant context:\n".$context;

        $payload = [
            'model'=>get_option(self::OPT_MODEL,'gpt-4o-mini'),
            'messages'=>[
                ['role'=>'system','content'=>$system_text],
                ['role'=>'user','content'=>$message]
            ],
            'temperature'=>0.3
        ];

        $reply = $this->call_chat_api($api_key,$payload);

        if( $reply ){
            $this->log_event('Bot: '.$reply, $session_id, 'bot', $user_id);
        }

        return ['success'=>true,'reply'=>$reply];
    }

    private function call_chat_api($api_key,$payload){
        $attempts = 0; $max_attempts = 2;
        $reply = 'Unable to get reply.';
        while($attempts<$max_attempts){
            $attempts++;
            $ch = curl_init('https://api.openai.com/v1/chat/completions');
            curl_setopt($ch,CURLOPT_HTTPHEADER,[
                'Content-Type: application/json',
                'Authorization: Bearer '.$api_key
            ]);
            curl_setopt($ch,CURLOPT_RETURNTRANSFER,true);
            curl_setopt($ch,CURLOPT_POST,true);
            curl_setopt($ch,CURLOPT_POSTFIELDS,json_encode($payload));
            $res = curl_exec($ch);
            if(curl_errno($ch)){
                $this->log_event('cURL error: '.curl_error($ch));
                curl_close($ch);
                continue;
            }
            $code = curl_getinfo($ch,CURLINFO_HTTP_CODE);
            curl_close($ch);
            $data = json_decode($res,true);
            if($code>=200 && $code<300 && isset($data['choices'][0]['message']['content'])){
                $reply = $data['choices'][0]['message']['content'];
                break;
            } else {
                $this->log_event('API error: HTTP '.$code.' - '.$res);
            }
        }
        return $reply;
    }

    // -------------------------------
    // Language detection via OpenAI
    // -------------------------------
    private function detect_language($text){
        $api_key = get_option(self::OPT_KEY);
        if(!$api_key || !$text) return null;
        $payload = [
            'model'=>get_option(self::OPT_MODEL,'gpt-4o-mini'),
            'messages'=>[
                ['role'=>'system','content'=>'Detect the language of the user text. Reply ONLY with the 2-letter language code, e.g. "en","hr","de".'],
                ['role'=>'user','content'=>$text]
            ],
            'temperature'=>0
        ];
        $ch = curl_init('https://api.openai.com/v1/chat/completions');
        curl_setopt($ch,CURLOPT_HTTPHEADER,[
            'Content-Type: application/json',
            'Authorization: Bearer '.$api_key
        ]);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER,true);
        curl_setopt($ch,CURLOPT_POST,true);
        curl_setopt($ch,CURLOPT_POSTFIELDS,json_encode($payload));
        $res = curl_exec($ch);
        if(curl_errno($ch)){
            curl_close($ch);
            return null;
        }
        $code = curl_getinfo($ch,CURLINFO_HTTP_CODE);
        curl_close($ch);
        $data = json_decode($res,true);
        if($code>=200 && $code<300 && isset($data['choices'][0]['message']['content'])){
            $lang = trim($data['choices'][0]['message']['content']);
            return substr($lang,0,2);
        }
        return null;
    }

    // -------------------------------
    // Embedding + cosine similarity
    // -------------------------------
    private function get_embedding($text,$api_key){
        $ch = curl_init('https://api.openai.com/v1/embeddings');
        curl_setopt($ch,CURLOPT_HTTPHEADER,[
            'Content-Type: application/json',
            'Authorization: Bearer '.$api_key
        ]);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER,true);
        curl_setopt($ch,CURLOPT_POST,true);
        curl_setopt($ch,CURLOPT_POSTFIELDS,json_encode([
            'input'=>$text,
            'model'=>'text-embedding-3-small'
        ]));
        $res = curl_exec($ch);
        if(curl_errno($ch)){
            $this->log_event('Embedding cURL error: '.curl_error($ch));
            curl_close($ch);
            return null;
        }
        $code = curl_getinfo($ch,CURLINFO_HTTP_CODE);
        curl_close($ch);
        $data = json_decode($res,true);
        if($code>=200 && $code<300 && isset($data['data'][0]['embedding'])){
            return $data['data'][0]['embedding'];
        }
        $this->log_event('Embedding API error: HTTP '.$code.' - '.$res);
        return null;
    }

    private function cosine_similarity($a,$b){
        if(!is_array($a) || !is_array($b) || count($a)!==count($b)) return 0;
        $dot=0; $na=0; $nb=0;
        $len = count($a);
        for($i=0;$i<$len;$i++){
            $dot += $a[$i]*$b[$i];
            $na  += $a[$i]*$a[$i];
            $nb  += $b[$i]*$b[$i];
        }
        if($na==0 || $nb==0) return 0;
        return $dot/(sqrt($na)*sqrt($nb));
    }

    // -------------------------------
    // Indexing helpers
    // -------------------------------
    private function fetch_url_content($url){
        $res = wp_remote_get($url);
        if(is_wp_error($res)) return '';
        $body = wp_remote_retrieve_body($res);
        $body = wp_strip_all_tags($body);
        $body = preg_replace('/\s+/',' ',$body);
        return trim($body);
    }

    private function chunk_text($text,$max_len=800){
        $words = preg_split('/\s+/', $text);
        $chunks = [];
        $current = '';
        foreach($words as $w){
            if(strlen($current.' '.$w)>$max_len){
                $chunks[] = trim($current);
                $current = $w;
            } else {
                $current .= ' '.$w;
            }
        }
        if(trim($current) !== ''){
            $chunks[] = trim($current);
        }
        return $chunks;
    }

    private function full_reindex($show_progress=false){
        global $wpdb;
        $this->maybe_create_tables();
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $wpdb->query("TRUNCATE TABLE $table");
        return $this->update_existing_index($show_progress);
    }

    private function update_existing_index($show_progress=false){
        global $wpdb;
        $this->maybe_create_tables();
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        $sources = array_filter(array_map('trim', explode("\n", get_option(self::OPT_SOURCES))));
        $existing = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        $total_urls = count($sources);
        $idx = 0;
        foreach($sources as $url){
            if(!in_array($url,$existing)){
                $idx++;
                continue;
            }
            $this->update_single_url($url,$show_progress,$idx,$total_urls);
            $idx++;
            if($show_progress) echo "<script>document.getElementById('wpoasc_progress').value=".intval($idx/$total_urls*100).";</script>"; flush();
        }
        return ['success'=>true,'message'=>'Existing indexed pages updated.'];
    }

    private function add_new_pages_index($show_progress=false){
        global $wpdb;
        $this->maybe_create_tables();
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        $sources = array_filter(array_map('trim', explode("\n", get_option(self::OPT_SOURCES))));
        $existing = $wpdb->get_col("SELECT DISTINCT url FROM $table");
        $added = 0; $total_urls = count($sources); $idx=0;
        foreach($sources as $url){
            if(in_array($url,$existing)) { $idx++; continue; }
            $this->update_single_url($url,$show_progress,$idx,$total_urls);
            $idx++; $added++;
            if($show_progress) echo "<script>document.getElementById('wpoasc_progress').value=".intval($idx/$total_urls*100).";</script>"; flush();
        }
        return ['success'=>true,'message'=>"Added $added new URLs to index."];
    }

    private function update_single_url($url,$show_progress=false,$idx=0,$total_urls=1){
        global $wpdb;
        $this->maybe_create_tables();
        $table = $wpdb->prefix.'wpoasc_embeddings';
        $api_key = get_option(self::OPT_KEY);
        $content = $this->fetch_url_content($url);
        if(!$content){
            $this->log_event("Empty content for URL: $url");
            return;
        }
        $chunks = $this->chunk_text($content);
        $wpdb->delete($table,['url'=>$url]);
        foreach($chunks as $chunk){
            $emb = $this->get_embedding($chunk,$api_key);
            if(!$emb) continue;
            $wpdb->insert($table,[
                'url'=>$url,
                'chunk'=>$chunk,
                'embedding'=>json_encode($emb)
            ]);
        }
        if($show_progress && $total_urls > 0){
            echo "<script>document.getElementById('wpoasc_progress').value=".intval(($idx+1)/$total_urls*100).";</script>"; flush();
        }
    }

    // -------------------------------
    // Front-end assets & shortcode
    // -------------------------------
    public function enqueue_assets(){
        $bot_name     = get_option(self::OPT_BOT_NAME, 'Support Bot');
        $send_label   = get_option(self::OPT_SEND_LABEL, 'Send');
        $primary      = get_option(self::OPT_PRIMARY_COLOR, '#22d3ee');
        $theme_mode   = get_option(self::OPT_THEME_MODE, 'dark');
        $default_lang = get_option(self::OPT_LANG, 'en');

        wp_enqueue_script(
            'wpoasc-chatbot-js',
            plugin_dir_url(__FILE__).'chatbot.js',
            ['jquery'],
            null,
            true
        );

        wp_localize_script(
            'wpoasc-chatbot-js',
            'wpoasc_vars',
            [
                'rest_url'      => rest_url('openai-chat/v1/message'),
                'bot_name'      => $bot_name,
                'send_label'    => $send_label,
                'primary_color' => $primary,
                'theme_mode'    => $theme_mode,
                'default_lang'  => $default_lang,
            ]
        );

        wp_enqueue_style(
            'wpoasc-chatbot-css',
            plugin_dir_url(__FILE__).'chatbot.css'
        );

        // Theme via CSS variables
        $style = '';
        if( $primary ){
            $style .= '#wpoasc-chatbot-container.wpoasc-chatbot{--wpoasc-primary:'.esc_attr($primary).';}';
        }

        if( $theme_mode === 'light' ){
            // Light theme – “website style”
            $style .= '#wpoasc-chatbot-container.wpoasc-chatbot{'.
                        '--wpoasc-bg:#ffffff;'.
                        '--wpoasc-bg-header:#ffffff;'.
                        '--wpoasc-bg-input:#f9fafb;'.
                        '--wpoasc-messages-bg:#f9fafb;'.
                        '--wpoasc-text:#111827;'.
                        '--wpoasc-text-muted:#6b7280;'.
                        '--wpoasc-bot-bg:#e5edf5;'.
                        '--wpoasc-bot-text:#111827;'.
                      '}';
        } else {
            // Dark theme – original glassy look
            $style .= '#wpoasc-chatbot-container.wpoasc-chatbot{'.
                        '--wpoasc-bg:#0f172a;'.
                        '--wpoasc-bg-header:#020617;'.
                        '--wpoasc-bg-input:#020617;'.
                        '--wpoasc-messages-bg:transparent;'.
                        '--wpoasc-text:#e5e7eb;'.
                        '--wpoasc-text-muted:#9ca3af;'.
                      '}';
        }

        if( $style ){
            wp_add_inline_style('wpoasc-chatbot-css', $style);
        }
    }

    public function shortcode(){
        $bot_name     = get_option(self::OPT_BOT_NAME, 'Support Bot');
        $lang         = get_option(self::OPT_LANG, 'en');
        $saved_header = get_option(self::OPT_HEADER_TEXT, '');
        $header_text  = '';

        if($saved_header !== ''){
            $header_text = $saved_header;
        } else {
            switch(strtolower($lang)){
                case 'hr':
                    $header_text = 'Postavite mi bilo koje pitanje o ovoj stranici.';
                    break;
                case 'de':
                    $header_text = 'Stellen Sie mir jede Frage zu dieser Website.';
                    break;
                case 'es':
                    $header_text = 'Hazme cualquier pregunta sobre este sitio.';
                    break;
                case 'fr':
                    $header_text = 'Posez-moi n\'importe quelle question sur ce site.';
                    break;
                case 'it':
                    $header_text = 'Fammi qualsiasi domanda su questo sito.';
                    break;
                default:
                    $header_text = 'Ask me anything about this site.';
                    break;
            }
        }

        $send_label = get_option(self::OPT_SEND_LABEL, 'Send');

        ob_start(); ?>
        <div id="wpoasc-chatbot-container" class="wpoasc-chatbot" aria-label="<?php esc_attr_e('Support chatbot','wpoasc'); ?>">

            <div class="wpoasc-chat-header">
                <div class="wpoasc-chat-avatar">
                    <?php echo esc_html( strtoupper( mb_substr( $bot_name, 0, 1 ) ) ); ?>
                </div>
                <div class="wpoasc-chat-header-text">
                    <div class="wpoasc-chat-title">
                        <?php echo esc_html( $bot_name ); ?>
                    </div>
                    <div class="wpoasc-chat-subtitle">
                        <?php echo esc_html( $header_text ); ?>
                    </div>
                </div>
            </div>

            <div id="wpoasc-messages"
                 class="wpoasc-messages"
                 role="log"
                 aria-live="polite"
                 aria-label="<?php esc_attr_e('Chat messages','wpoasc'); ?>">
            </div>

            <div class="wpoasc-chat-input-row">
                <label for="wpoasc-user-input" class="wpoasc-visually-hidden">
                    <?php esc_html_e('Your message','wpoasc'); ?>
                </label>
                <textarea
                    id="wpoasc-user-input"
                    class="wpoasc-input"
                    rows="1"
                    placeholder="<?php esc_attr_e('Type your question and press Enter...','wpoasc'); ?>"
                ></textarea>

                <button id="wpoasc-send-btn" class="wpoasc-send-btn" type="button">
                    <?php echo esc_html( $send_label ); ?>
                </button>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

}

new WP_OpenAI_Support_Chatbot();

