(function($) {
    'use strict';

    // Safety: ako shortcode nije na stranici, ne radi ništa
    $(document).ready(function() {
        var $container = $('#wpoasc-chatbot-container');
        if (!$container.length) {
            return;
        }

        var $messages  = $('#wpoasc-messages');
        var $input     = $('#wpoasc-user-input');
        var $sendBtn   = $('#wpoasc-send-btn');

        var botName    = (window.wpoasc_vars && wpoasc_vars.bot_name)      || 'Support Bot';
        var restUrl    = (window.wpoasc_vars && wpoasc_vars.rest_url)      || '';
        var defLang    = (window.wpoasc_vars && wpoasc_vars.default_lang)  || 'en';

        // Jedan session_id po browseru (sprema se u localStorage)
        var sessionKey = 'wpoasc_session_id';
        var sessionId  = window.localStorage ? window.localStorage.getItem(sessionKey) : null;
        if (!sessionId) {
            sessionId = generateUUID();
            if (window.localStorage) {
                window.localStorage.setItem(sessionKey, sessionId);
            }
        }

        // -------------------------
        // Helper: UUID v4 light
        // -------------------------
        function generateUUID() {
            // simple UUID v4
            return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
                var r = Math.random() * 16 | 0;
                var v = c === 'x' ? r : (r & 0x3 | 0x8);
                return v.toString(16);
            });
        }

        // -------------------------
        // Helper: welcome poruka po jeziku
        // -------------------------
        function getWelcomeText(lang, name) {
            lang = (lang || 'en').toLowerCase();
            switch (lang) {
                case 'hr':
                    return 'Bok, ja sam ' + name + ' – kako mogu pomoći?';
                case 'de':
                    return 'Hallo, ich bin ' + name + ' – wie kann ich helfen?';
                case 'fr':
                    return 'Salut, je suis ' + name + ' – comment puis-je vous aider ?';
                case 'es':
                    return 'Hola, soy ' + name + ' – ¿en qué puedo ayudarte?';
                case 'it':
                    return 'Ciao, sono ' + name + ' – come posso aiutarti?';
                default:
                    return 'Hi, I\'m ' + name + ' – how can I help?';
            }
        }

        // -------------------------
        // Render poruke
        // -------------------------
        function appendMessage(role, text, opts) {
            opts = opts || {};
            var isTyping = !!opts.typing;

            var $msg = $('<div/>', {
                'class': 'wpoasc-message ' + (role === 'user' ? 'user' : 'bot')
            });

            var bubbleClass = 'wpoasc-bubble';
            if (isTyping && role === 'bot') {
                bubbleClass += ' wpoasc-typing-bubble';
            }

            var $bubble = $('<div/>', {
                'class': bubbleClass
            });

            if (isTyping && role === 'bot') {
                // Typing indikator – tri točkice
                var $typing = $('<div/>', { 'class': 'wpoasc-typing' });
                $typing.append($('<span/>', { 'class': 'wpoasc-typing-dot' }));
                $typing.append($('<span/>', { 'class': 'wpoasc-typing-dot' }));
                $typing.append($('<span/>', { 'class': 'wpoasc-typing-dot' }));
                $bubble.append($typing);
            } else {
                $bubble.text(text);
            }

            $msg.append($bubble);
            $messages.append($msg);

            scrollToBottom();
        }

        function scrollToBottom() {
            var el = $messages[0];
            if (!el) return;
            el.scrollTop = el.scrollHeight;
        }

        // -------------------------
        // Typing indikator
        // -------------------------
        function showTyping() {
            // ukloni stare, ako su ostali
            hideTyping();
            appendMessage('bot', '', { typing: true });
        }

        function hideTyping() {
            $messages.find('.wpoasc-typing-bubble').closest('.wpoasc-message').remove();
        }

        // -------------------------
        // Slanje poruke
        // -------------------------
        function sendMessage() {
            var text = ($input.val() || '').trim();
            if (!text || !restUrl) {
                return;
            }

            // Locally prikaži user poruku
            appendMessage('user', text);

            // očisti i resetiraj textarea
            $input.val('');
            autoResizeTextarea();

            // disable dok čekamo odgovor
            setSendingState(true);
            showTyping();

            $.ajax({
                url: restUrl,
                method: 'POST',
                contentType: 'application/json',
                dataType: 'json',
                data: JSON.stringify({
                    message: text,
                    session_id: sessionId
                })
            }).done(function(res) {
                hideTyping();
                setSendingState(false);

                if (res && res.success && res.reply) {
                    appendMessage('bot', res.reply);
                } else if (res && res.reply) {
                    // npr. rate limit message
                    appendMessage('bot', res.reply);
                } else {
                    appendMessage('bot', 'Unexpected response from server.');
                }
            }).fail(function(xhr) {
                hideTyping();
                setSendingState(false);

                var msg = 'Error contacting server.';
                try {
                    if (xhr && xhr.responseJSON && xhr.responseJSON.message) {
                        msg = xhr.responseJSON.message;
                    }
                } catch(e) {}
                appendMessage('bot', msg);
            });
        }

        function setSendingState(isSending) {
            $input.prop('disabled', isSending);
            $sendBtn.prop('disabled', isSending);
        }

        // -------------------------
        // Auto-resize textarea
        // -------------------------
        function autoResizeTextarea() {
            var el = $input[0];
            if (!el) return;
            el.style.height = 'auto';
            var max = 80; // px
            el.style.height = Math.min(el.scrollHeight, max) + 'px';
        }

        // -------------------------
        // Event binding
        // -------------------------
        $sendBtn.on('click', function(e) {
            e.preventDefault();
            sendMessage();
        });

        $input.on('keydown', function(e) {
            // Enter = pošalji, Shift+Enter = novi red
            if (e.key === 'Enter' || e.keyCode === 13) {
                if (!e.shiftKey) {
                    e.preventDefault();
                    sendMessage();
                }
            }
        });

        $input.on('input', function() {
            autoResizeTextarea();
        });

        // -------------------------
        // Initial welcome poruka
        // -------------------------
        if ($messages.children().length === 0) {
            var welcome = getWelcomeText(defLang, botName);
            appendMessage('bot', welcome);
        }
    });

})(jQuery);

