<?php
/**
 * Plugin Name: Bulk OpenAI Article Generator
 * Description: Generate SEO articles in bulk using the OpenAI API and push them into WordPress as draft posts.
 * Version: 1.2.0
 * Author: Your Name
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Bulk_OpenAI_Article_Generator {

    const CPT_SLUG = 'boag_article';

    public function __construct() {
        add_action( 'init', [ $this, 'register_custom_post_type' ] );
        add_action( 'admin_menu', [ $this, 'register_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );

        // AJAX for per-article generation with status/progress
        add_action( 'wp_ajax_boag_generate_article', [ $this, 'ajax_generate_article' ] );
    }

    public function register_custom_post_type() {
        $labels = [
            'name'               => 'AI Articles',
            'singular_name'      => 'AI Article',
            'menu_name'          => 'AI Articles',
            'name_admin_bar'     => 'AI Article',
            'add_new'            => 'Add New',
            'add_new_item'       => 'Add New AI Article',
            'new_item'           => 'New AI Article',
            'edit_item'          => 'Edit AI Article',
            'view_item'          => 'View AI Article',
            'all_items'          => 'All AI Articles',
            'search_items'       => 'Search AI Articles',
            'parent_item_colon'  => 'Parent AI Articles:',
            'not_found'          => 'No AI articles found.',
            'not_found_in_trash' => 'No AI articles found in Trash.',
        ];

        $args = [
            'labels'             => $labels,
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'            => false,
            'show_in_menu'       => false,
            'query_var'          => false,
            'rewrite'            => false,
            'capability_type'    => 'post',
            'has_archive'        => false,
            'hierarchical'       => false,
            'supports'           => [ 'title', 'editor' ],
        ];

        register_post_type( self::CPT_SLUG, $args );
    }

    public function register_admin_menu() {
        // Main menu points to Generate page
        add_menu_page(
            'AI Bulk Articles',
            'AI Bulk Articles',
            'manage_options',
            'boag_generate',
            [ $this, 'render_generate_page' ],
            'dashicons-edit',
            81
        );

        add_submenu_page(
            'boag_generate',
            'Generate Articles',
            'Generate Articles',
            'manage_options',
            'boag_generate',
            [ $this, 'render_generate_page' ]
        );

        add_submenu_page(
            'boag_generate',
            'Generated Articles',
            'Generated Articles',
            'manage_options',
            'boag_generated_articles',
            [ $this, 'render_generated_articles_page' ]
        );

        add_submenu_page(
            'boag_generate',
            'Settings',
            'Settings',
            'manage_options',
            'boag_settings',
            [ $this, 'render_settings_page' ]
        );
    }

    public function register_settings() {
        register_setting( 'boag_settings_group', 'boag_api_key' );
        register_setting( 'boag_settings_group', 'boag_default_model' );
        register_setting( 'boag_settings_group', 'boag_default_language' );
        register_setting( 'boag_settings_group', 'boag_delete_on_uninstall' );
    }

    /* ========== SETTINGS PAGE ========== */

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $delete_on_uninstall = (int) get_option( 'boag_delete_on_uninstall', 0 );
        ?>
        <div class="wrap">
            <h1>AI Bulk Articles – Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields( 'boag_settings_group' ); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_api_key">OpenAI API Key</label></th>
                        <td>
                            <input type="password" name="boag_api_key" id="boag_api_key"
                                   value="<?php echo esc_attr( get_option( 'boag_api_key', '' ) ); ?>"
                                   class="regular-text" />
                            <p class="description">Enter your OpenAI secret key (starts with <code>sk-</code>).</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_default_model">Default Model</label></th>
                        <td>
                            <input type="text" name="boag_default_model" id="boag_default_model"
                                   value="<?php echo esc_attr( get_option( 'boag_default_model', 'gpt-4.1-mini' ) ); ?>"
                                   class="regular-text" />
                            <p class="description">
                                Example: <code>gpt-4.1-mini</code>, <code>gpt-4o-mini</code>, etc.
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_default_language">Default Language</label></th>
                        <td>
                            <input type="text" name="boag_default_language" id="boag_default_language"
                                   value="<?php echo esc_attr( get_option( 'boag_default_language', 'English' ) ); ?>"
                                   class="regular-text" />
                            <p class="description">Example: <code>English</code>, <code>Croatian</code>, <code>German</code>…</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_delete_on_uninstall">Delete data on uninstall</label></th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_delete_on_uninstall" id="boag_delete_on_uninstall"
                                       value="1" <?php checked( $delete_on_uninstall, 1 ); ?> />
                                Delete all generated AI articles and plugin settings when the plugin is deleted.
                            </label>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    /* ========== GENERATE PAGE ========== */

    public function render_generate_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $api_key          = get_option( 'boag_api_key', '' );
        $default_language = get_option( 'boag_default_language', 'English' );
        $ajax_nonce       = wp_create_nonce( 'boag_generate_ajax' );

        // Fallback non-AJAX generation
        if ( isset( $_POST['boag_generate_submit'] ) && empty( $_POST['boag_ajax_run'] ) ) {
            check_admin_referer( 'boag_generate_articles', 'boag_generate_nonce' );
            $this->handle_generation_form();
        }

        $categories = get_categories( [ 'hide_empty' => false ] );
        ?>
        <div class="wrap">
            <h1>AI Bulk Article Generator</h1>

            <?php if ( empty( $api_key ) ) : ?>
                <div class="notice notice-error">
                    <p><strong>OpenAI API key is not set.</strong> Please go to the Settings tab and add it before generating articles.</p>
                </div>
            <?php endif; ?>

            <form method="post" id="boag-generate-form">
                <?php wp_nonce_field( 'boag_generate_articles', 'boag_generate_nonce' ); ?>
                <input type="hidden" name="boag_ajax_run" value="1" />

                <h2 class="title">Generation Mode</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Mode</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio" name="boag_mode" value="topic" checked="checked" />
                                    Generate X articles from a topic keyword
                                </label><br />
                                <label>
                                    <input type="radio" name="boag_mode" value="titles" />
                                    Generate articles from provided titles (one per line)
                                </label>
                            </fieldset>
                        </td>
                    </tr>

                    <tr class="boag-mode-topic">
                        <th scope="row"><label for="boag_topic_keyword">Topic keyword</label></th>
                        <td>
                            <input type="text" name="boag_topic_keyword" id="boag_topic_keyword"
                                   class="regular-text" />
                            <p class="description">Example: <code>vegan recipes</code>, <code>WordPress SEO</code>.</p>
                        </td>
                    </tr>

                    <tr class="boag-mode-topic">
                        <th scope="row"><label for="boag_article_count">Number of articles</label></th>
                        <td>
                            <input type="number" name="boag_article_count" id="boag_article_count"
                                   class="small-text" value="3" min="1" />
                            <p class="description">No hard-coded limit, but very large batches may take a while.</p>
                        </td>
                    </tr>

                    <tr class="boag-mode-titles" style="display:none;">
                        <th scope="row"><label for="boag_titles">Article titles</label></th>
                        <td>
                            <textarea name="boag_titles" id="boag_titles" class="large-text code" rows="8"
                                      placeholder="Title 1&#10;Title 2&#10;Title 3"></textarea>
                            <p class="description">Enter one title per line. Each title will become a separate article.</p>
                        </td>
                    </tr>
                </table>

                <h2 class="title">Article Options</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_language">Language</label></th>
                        <td>
                            <input type="text" name="boag_language" id="boag_language"
                                   value="<?php echo esc_attr( $default_language ); ?>" class="regular-text" />
                            <p class="description">Overrides default language for this batch if changed.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_length">Length (words)</label></th>
                        <td>
                            <input type="number" name="boag_length" id="boag_length" class="small-text" value="1200" min="200" />
                            <p class="description">Approximate target word count per article.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_tone">Tone</label></th>
                        <td>
                            <input type="text" name="boag_tone" id="boag_tone" class="regular-text"
                                   placeholder="informative, friendly, professional, casual..." />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_instructions">Additional instructions</label></th>
                        <td>
                            <textarea name="boag_instructions" id="boag_instructions"
                                      class="large-text code" rows="5"
                                      placeholder="E.g. Use H2 and H3 headings, include bullet lists, write SEO-friendly introduction and conclusion, etc."></textarea>
                        </td>
                    </tr>
                </table>

                <h2 class="title">Taxonomy (optional)</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Categories</th>
                        <td>
                            <?php if ( ! empty( $categories ) ) : ?>
                                <?php foreach ( $categories as $cat ) : ?>
                                    <label>
                                        <input type="checkbox" name="boag_categories[]" value="<?php echo esc_attr( $cat->term_id ); ?>" />
                                        <?php echo esc_html( $cat->name ); ?>
                                    </label><br />
                                <?php endforeach; ?>
                                <p class="description">Selected categories will be assigned when sending generated articles to WordPress draft posts.</p>
                            <?php else : ?>
                                <p class="description">No categories found. You can create some in Posts → Categories.</p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_tags">Tags</label></th>
                        <td>
                            <input type="text" name="boag_tags" id="boag_tags" class="regular-text"
                                   placeholder="tag1, tag2, tag3" />
                            <p class="description">Comma-separated tags to assign when sending to WordPress draft posts.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( 'Generate Articles', 'primary', 'boag_generate_submit' ); ?>
            </form>

            <div id="boag-generation-status" style="margin-top:20px;"></div>
            <div id="boag-progress-bar-wrap" style="width:100%;max-width:400px;background:#f1f1f1;border:1px solid #ccd0d4;height:20px;border-radius:3px;display:none;">
                <div id="boag-progress-bar" style="height:100%;width:0%;background:#0073aa;"></div>
            </div>
        </div>

        <script>
        (function($) {
            var boagGenerateAjaxNonce = '<?php echo esc_js( $ajax_nonce ); ?>';

            const modeRadios = document.querySelectorAll('input[name="boag_mode"]');
            const topicRows = document.querySelectorAll('.boag-mode-topic');
            const titleRows = document.querySelectorAll('.boag-mode-titles');

            function updateMode() {
                let mode = 'topic';
                modeRadios.forEach(function(r) { if (r.checked) mode = r.value; });
                if (mode === 'topic') {
                    topicRows.forEach(function(el) { el.style.display = ''; });
                    titleRows.forEach(function(el) { el.style.display = 'none'; });
                } else {
                    topicRows.forEach(function(el) { el.style.display = 'none'; });
                    titleRows.forEach(function(el) { el.style.display = ''; });
                }
            }

            modeRadios.forEach(function(r) { r.addEventListener('change', updateMode); });
            updateMode();

            // AJAX generation with progress/status
            const form = document.getElementById('boag-generate-form');
            const statusBox = document.getElementById('boag-generation-status');
            const progressWrap = document.getElementById('boag-progress-bar-wrap');
            const progressBar = document.getElementById('boag-progress-bar');

            if (form) {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();

                    statusBox.innerHTML = '';
                    progressWrap.style.display = 'none';
                    progressBar.style.width = '0%';

                    const submitBtn = form.querySelector('input[type="submit"], button[type="submit"]');
                    if (submitBtn) {
                        submitBtn.disabled = true;
                    }

                    let mode = 'topic';
                    modeRadios.forEach(function(r) { if (r.checked) mode = r.value; });

                    const language     = form.querySelector('#boag_language').value.trim();
                    const length       = parseInt(form.querySelector('#boag_length').value, 10) || 1200;
                    const tone         = form.querySelector('#boag_tone').value.trim();
                    const instructions = form.querySelector('#boag_instructions').value.trim();

                    const catEls = form.querySelectorAll('input[name="boag_categories[]"]:checked');
                    let categories = [];
                    catEls.forEach(function(el) { categories.push(el.value); });

                    const tags = (form.querySelector('#boag_tags').value || '').trim();

                    let tasks = [];

                    if (mode === 'topic') {
                        const topic = form.querySelector('#boag_topic_keyword').value.trim();
                        const count = parseInt(form.querySelector('#boag_article_count').value, 10) || 1;

                        if (!topic || count < 1) {
                            statusBox.innerHTML = '<div class="notice notice-error"><p>Please provide a topic keyword and a valid article count.</p></div>';
                            if (submitBtn) submitBtn.disabled = false;
                            return;
                        }

                        for (let i = 0; i < count; i++) {
                            tasks.push({ mode: 'topic', topic: topic });
                        }
                    } else {
                        const titlesRaw = form.querySelector('#boag_titles').value || '';
                        const lines = titlesRaw.split(/\r?\n/).map(function(line) { return line.trim(); }).filter(Boolean);

                        if (!lines.length) {
                            statusBox.innerHTML = '<div class="notice notice-error"><p>Please provide at least one title.</p></div>';
                            if (submitBtn) submitBtn.disabled = false;
                            return;
                        }

                        lines.forEach(function(title) {
                            tasks.push({ mode: 'titles', title: title });
                        });
                    }

                    if (!tasks.length) {
                        statusBox.innerHTML = '<div class="notice notice-error"><p>No tasks to run.</p></div>';
                        if (submitBtn) submitBtn.disabled = false;
                        return;
                    }

                    let total = tasks.length;
                    let completed = 0;
                    let failed = 0;

                    progressWrap.style.display = 'block';
                    statusBox.innerHTML = '<p>Starting generation of ' + total + ' article(s)...</p><ul id="boag-status-list"></ul>';
                    const statusList = document.getElementById('boag-status-list');

                    function updateProgress() {
                        const percent = Math.round((completed / total) * 100);
                        progressBar.style.width = percent + '%';
                    }

                    function addStatus(message, success) {
                        if (!statusList) return;
                        const li = document.createElement('li');
                        li.textContent = message;
                        li.style.color = success ? 'green' : 'red';
                        statusList.appendChild(li);
                    }

                    function runTask(index) {
                        if (index >= total) {
                            let summary = '<p><strong>Done.</strong> Generated ' + (total - failed) +
                                ' article(s), ' + failed + ' failed.</p>' +
                                '<p><a href="<?php echo esc_url( admin_url( 'admin.php?page=boag_generated_articles' ) ); ?>">View generated articles</a></p>';
                            statusBox.insertAdjacentHTML('beforeend', summary);
                            if (submitBtn) submitBtn.disabled = false;
                            return;
                        }

                        const task = tasks[index];

                        const data = {
                            action: 'boag_generate_article',
                            security: boagGenerateAjaxNonce,
                            mode: task.mode,
                            topic: task.topic || '',
                            title: task.title || '',
                            language: language,
                            length: length,
                            tone: tone,
                            instructions: instructions,
                            categories: categories,
                            tags: tags
                        };

                        addStatus('Generating article ' + (index + 1) + ' of ' + total + '...', true);

                        $.post(ajaxurl, data)
                            .done(function(response) {
                                completed++;
                                if (response && response.success && response.data) {
                                    const t = response.data.title || ('Article #' + (index + 1));
                                    addStatus('Article ' + (index + 1) + ' generated: "' + t + '"', true);
                                } else {
                                    failed++;
                                    let msg = (response && response.data && response.data.message) ? response.data.message : 'Unknown error';
                                    addStatus('Article ' + (index + 1) + ' failed: ' + msg, false);
                                }
                                updateProgress();
                                runTask(index + 1);
                            })
                            .fail(function() {
                                completed++;
                                failed++;
                                addStatus('Article ' + (index + 1) + ' failed: AJAX error.', false);
                                updateProgress();
                                runTask(index + 1);
                            });
                    }

                    updateProgress();
                    runTask(0);
                });
            }
        })(jQuery);
        </script>
        <?php
    }

    private function handle_generation_form() {
        // Old synchronous generation logic as fallback
        $api_key = get_option( 'boag_api_key', '' );
        if ( empty( $api_key ) ) {
            echo '<div class="notice notice-error"><p>OpenAI API key is missing. Set it in the Settings page.</p></div>';
            return;
        }

        $mode         = isset( $_POST['boag_mode'] ) ? sanitize_text_field( $_POST['boag_mode'] ) : 'topic';
        $language     = isset( $_POST['boag_language'] ) ? sanitize_text_field( $_POST['boag_language'] ) : get_option( 'boag_default_language', 'English' );
        $length       = isset( $_POST['boag_length'] ) ? max( 100, intval( $_POST['boag_length'] ) ) : 1200;
        $tone         = isset( $_POST['boag_tone'] ) ? sanitize_text_field( $_POST['boag_tone'] ) : '';
        $instructions = isset( $_POST['boag_instructions'] ) ? sanitize_textarea_field( $_POST['boag_instructions'] ) : '';
        $model        = get_option( 'boag_default_model', 'gpt-4.1-mini' );
        $categories   = isset( $_POST['boag_categories'] ) ? array_map( 'intval', (array) $_POST['boag_categories'] ) : [];
        $tags         = isset( $_POST['boag_tags'] ) ? sanitize_text_field( $_POST['boag_tags'] ) : '';

        $articles_to_generate = [];

        if ( $mode === 'topic' ) {
            $topic = isset( $_POST['boag_topic_keyword'] ) ? sanitize_text_field( $_POST['boag_topic_keyword'] ) : '';
            $count = isset( $_POST['boag_article_count'] ) ? intval( $_POST['boag_article_count'] ) : 1;

            if ( empty( $topic ) || $count < 1 ) {
                echo '<div class="notice notice-error"><p>Please provide a topic keyword and a valid article count.</p></div>';
                return;
            }

            for ( $i = 0; $i < $count; $i++ ) {
                $articles_to_generate[] = [
                    'mode'  => 'topic',
                    'topic' => $topic,
                ];
            }
        } else {
            $titles_raw = isset( $_POST['boag_titles'] ) ? wp_unslash( $_POST['boag_titles'] ) : '';
            $titles     = array_filter( array_map( 'trim', explode( "\n", $titles_raw ) ) );

            if ( empty( $titles ) ) {
                echo '<div class="notice notice-error"><p>Please provide at least one title (one per line).</p></div>';
                return;
            }

            foreach ( $titles as $title ) {
                $articles_to_generate[] = [
                    'mode'  => 'titles',
                    'title' => $title,
                ];
            }
        }

        $generated_count = 0;
        $errors          = [];

        foreach ( $articles_to_generate as $index => $item ) {

            if ( $item['mode'] === 'topic' ) {
                $prompt = $this->build_topic_prompt(
                    $item['topic'],
                    $language,
                    $length,
                    $tone,
                    $instructions
                );
                $expected_title = null;
            } else {
                $prompt = $this->build_title_prompt(
                    $item['title'],
                    $language,
                    $length,
                    $tone,
                    $instructions
                );
                $expected_title = $item['title'];
            }

            $result = $this->call_openai_chat( $api_key, $model, $prompt );

            if ( is_wp_error( $result ) ) {
                $errors[] = 'Article #' . ( $index + 1 ) . ': ' . $result->get_error_message();
                continue;
            }

            $content_raw = trim( $result );

            if ( $item['mode'] === 'topic' ) {
                list( $title, $body ) = $this->extract_title_and_body_from_response( $content_raw, $item['topic'] );
            } else {
                $title = $expected_title;
                $body  = $content_raw;
            }

            $post_id = wp_insert_post( [
                'post_type'    => self::CPT_SLUG,
                'post_status'  => 'draft',
                'post_title'   => $title,
                'post_content' => $body,
            ] );

            if ( is_wp_error( $post_id ) || ! $post_id ) {
                $errors[] = 'Article #' . ( $index + 1 ) . ': Failed to save article to database.';
                continue;
            }

            update_post_meta( $post_id, '_boag_language', $language );
            update_post_meta( $post_id, '_boag_tone', $tone );
            update_post_meta( $post_id, '_boag_length', $length );
            update_post_meta( $post_id, '_boag_instructions', $instructions );
            update_post_meta( $post_id, '_boag_categories', $categories );
            update_post_meta( $post_id, '_boag_tags', $tags );

            $generated_count++;
        }

        if ( $generated_count > 0 ) {
            echo '<div class="notice notice-success"><p>Successfully generated ' . intval( $generated_count ) . ' article(s).</p></div>';
        }

        if ( ! empty( $errors ) ) {
            echo '<div class="notice notice-error"><p><strong>Some articles failed:</strong><br />' .
                 implode( '<br />', array_map( 'esc_html', $errors ) ) . '</p></div>';
        }
    }

    /* ========== PROMPT BUILDERS (HTML-READY, PRE-FORMATTED) ========== */

    private function build_topic_prompt( $topic, $language, $length, $tone, $instructions ) {
        $base = "You are an expert SEO content writer.\n\n";
        $base .= "Write a high-quality blog article in {$language} about the topic: \"{$topic}\".\n";
        $base .= "Target length: approximately {$length} words.\n";

        if ( ! empty( $tone ) ) {
            $base .= "Tone: {$tone}.\n";
        }

        if ( ! empty( $instructions ) ) {
            $base .= "Additional instructions:\n{$instructions}\n";
        }

        $base .= "\nStructure and formatting requirements:\n";
        $base .= "- Return the article as HTML suitable for direct use in a WordPress post body.\n";
        $base .= "- Do NOT include <html>, <head>, or <body> tags.\n";
        $base .= "- Start with a short introductory paragraph (<p>), not a heading.\n";
        $base .= "- Use <h2> for main sections and <h3> for subsections where appropriate.\n";
        $base .= "- Use <p> for paragraphs.\n";
        $base .= "- Use <ul><li> or <ol><li> for bullet/numbered lists where relevant.\n";
        $base .= "- Do NOT include literal text like \"H2:\" or \"H3:\" inside any headings.\n";
        $base .= "- Do not wrap the entire article in a single container <div>.\n";
        $base .= "- Do not output markdown; use HTML only.\n";

        $base .= "\nAt the very beginning, before the HTML article body, output a single line in the exact format:\n";
        $base .= "TITLE: Your SEO-optimized title here\n";
        $base .= "Then a blank line and then the full HTML article body.\n";

        return $base;
    }

    private function build_title_prompt( $title, $language, $length, $tone, $instructions ) {
        $base = "You are an expert SEO content writer.\n\n";
        $base .= "Write a high-quality blog article in {$language} with the exact title: \"{$title}\".\n";
        $base .= "Target length: approximately {$length} words.\n";

        if ( ! empty( $tone ) ) {
            $base .= "Tone: {$tone}.\n";
        }

        if ( ! empty( $instructions ) ) {
            $base .= "Additional instructions:\n{$instructions}\n";
        }

        $base .= "\nStructure and formatting requirements:\n";
        $base .= "- Return the article as HTML suitable for direct use in a WordPress post body.\n";
        $base .= "- Do NOT include <html>, <head>, or <body> tags.\n";
        $base .= "- Start with a short introductory paragraph (<p>), not a heading.\n";
        $base .= "- Use <h2> for main sections and <h3> for subsections where appropriate.\n";
        $base .= "- Use <p> for paragraphs.\n";
        $base .= "- Use <ul><li> or <ol><li> for bullet/numbered lists where relevant.\n";
        $base .= "- Do NOT include literal text like \"H2:\" or \"H3:\" inside any headings.\n";
        $base .= "- Do not wrap the entire article in a single container <div>.\n";
        $base .= "- Do not output markdown; use HTML only.\n";
        $base .= "- Do not prepend any meta line like \"TITLE:\"; the title is already known.\n";

        $base .= "\nReturn only the HTML article body (no extra explanations).\n";

        return $base;
    }

    private function extract_title_and_body_from_response( $content, $fallback_topic ) {
        $title = $fallback_topic . ' Article';
        $body  = $content;

        if ( preg_match( '/^TITLE:\s*(.+)$/mi', $content, $matches ) ) {
            $title = trim( $matches[1] );
            $body  = trim( str_replace( $matches[0], '', $content ) );
        }

        return [ $title, $body ];
    }

    private function call_openai_chat( $api_key, $model, $prompt ) {
        $endpoint = 'https://api.openai.com/v1/chat/completions';

        $body = [
            'model'    => $model,
            'messages' => [
                [
                    'role'    => 'system',
                    'content' => 'You are a helpful AI writing assistant.',
                ],
                [
                    'role'    => 'user',
                    'content' => $prompt,
                ],
            ],
            'temperature' => 0.7,
        ];

        $args = [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => 'Bearer ' . $api_key,
            ],
            'body'    => wp_json_encode( $body ),
            'timeout' => 60,
        ];

        $response = wp_remote_post( $endpoint, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $data = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code < 200 || $code >= 300 ) {
            $message = isset( $data['error']['message'] ) ? $data['error']['message'] : 'Unknown API error.';
            return new WP_Error( 'openai_api_error', 'OpenAI API error: ' . $message );
        }

        if ( empty( $data['choices'][0]['message']['content'] ) ) {
            return new WP_Error( 'openai_no_content', 'OpenAI did not return any content.' );
        }

        return $data['choices'][0]['message']['content'];
    }

    /* ========== AJAX HANDLER FOR PER-ARTICLE GENERATION ========== */

    public function ajax_generate_article() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Insufficient permissions.' ] );
        }

        check_ajax_referer( 'boag_generate_ajax', 'security' );

        $api_key = get_option( 'boag_api_key', '' );
        if ( empty( $api_key ) ) {
            wp_send_json_error( [ 'message' => 'OpenAI API key is missing.' ] );
        }

        $mode         = isset( $_POST['mode'] ) ? sanitize_text_field( $_POST['mode'] ) : 'topic';
        $language     = isset( $_POST['language'] ) ? sanitize_text_field( $_POST['language'] ) : get_option( 'boag_default_language', 'English' );
        $length       = isset( $_POST['length'] ) ? max( 100, intval( $_POST['length'] ) ) : 1200;
        $tone         = isset( $_POST['tone'] ) ? sanitize_text_field( $_POST['tone'] ) : '';
        $instructions = isset( $_POST['instructions'] ) ? sanitize_textarea_field( $_POST['instructions'] ) : '';
        $model        = get_option( 'boag_default_model', 'gpt-4.1-mini' );
        $categories   = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : [];
        $tags         = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';

        if ( $mode === 'topic' ) {
            $topic = isset( $_POST['topic'] ) ? sanitize_text_field( $_POST['topic'] ) : '';
            if ( empty( $topic ) ) {
                wp_send_json_error( [ 'message' => 'Topic is required for topic mode.' ] );
            }
            $prompt         = $this->build_topic_prompt( $topic, $language, $length, $tone, $instructions );
            $expected_title = null;
        } else {
            $title = isset( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
            if ( empty( $title ) ) {
                wp_send_json_error( [ 'message' => 'Title is required for titles mode.' ] );
            }
            $prompt         = $this->build_title_prompt( $title, $language, $length, $tone, $instructions );
            $expected_title = $title;
        }

        $result = $this->call_openai_chat( $api_key, $model, $prompt );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( [ 'message' => $result->get_error_message() ] );
        }

        $content_raw = trim( $result );

        if ( $mode === 'topic' ) {
            list( $title, $body ) = $this->extract_title_and_body_from_response( $content_raw, $topic );
        } else {
            $title = $expected_title;
            $body  = $content_raw;
        }

        $post_id = wp_insert_post( [
            'post_type'    => self::CPT_SLUG,
            'post_status'  => 'draft',
            'post_title'   => $title,
            'post_content' => $body,
        ] );

        if ( is_wp_error( $post_id ) || ! $post_id ) {
            wp_send_json_error( [ 'message' => 'Failed to save generated article.' ] );
        }

        update_post_meta( $post_id, '_boag_language', $language );
        update_post_meta( $post_id, '_boag_tone', $tone );
        update_post_meta( $post_id, '_boag_length', $length );
        update_post_meta( $post_id, '_boag_instructions', $instructions );
        update_post_meta( $post_id, '_boag_categories', $categories );
        update_post_meta( $post_id, '_boag_tags', $tags );

        wp_send_json_success( [
            'article_id' => $post_id,
            'title'      => $title,
        ] );
    }

    /* ========== GENERATED ARTICLES PAGE (WITH PAGINATION) ========== */

    public function render_generated_articles_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Handle actions: delete / send
        if ( isset( $_GET['boag_action'], $_GET['article_id'] ) ) {
            $action     = sanitize_text_field( wp_unslash( $_GET['boag_action'] ) );
            $article_id = intval( $_GET['article_id'] );

            if ( $action === 'delete' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'boag_delete_article_' . $article_id ) ) {
                wp_delete_post( $article_id, true );
                echo '<div class="notice notice-success"><p>Article deleted.</p></div>';
            }

            if ( $action === 'send' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'boag_send_article_' . $article_id ) ) {
                $this->send_article_to_draft_post( $article_id );
            }
        }

        $view_article_id = isset( $_GET['view_article_id'] ) ? intval( $_GET['view_article_id'] ) : 0;

        $paged = isset( $_GET['boag_paged'] ) ? max( 1, intval( $_GET['boag_paged'] ) ) : 1;

        $query = new WP_Query( [
            'post_type'      => self::CPT_SLUG,
            'post_status'    => 'draft',
            'posts_per_page' => 20,
            'paged'          => $paged,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ] );
        ?>
        <div class="wrap">
            <h1>Generated AI Articles</h1>

            <?php if ( ! $query->have_posts() ) : ?>
                <p>No AI articles generated yet.</p>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Title</th>
                            <th>Language</th>
                            <th>Created</th>
                            <th>Sent to WP Draft</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        while ( $query->have_posts() ) :
                            $query->the_post();
                            $article_id     = get_the_ID();
                            $language       = get_post_meta( $article_id, '_boag_language', true );
                            $sent_post_id   = get_post_meta( $article_id, '_boag_sent_post_id', true );
                            $sent_post_link = $sent_post_id ? get_edit_post_link( $sent_post_id ) : '';
                            ?>
                            <tr>
                                <td><?php echo esc_html( get_the_title() ); ?></td>
                                <td><?php echo esc_html( $language ); ?></td>
                                <td><?php echo esc_html( get_the_date( 'Y-m-d H:i' ) ); ?></td>
                                <td>
                                    <?php
                                    if ( $sent_post_id && $sent_post_link ) {
                                        echo '<a href="' . esc_url( $sent_post_link ) . '">View Draft</a>';
                                    } else {
                                        echo 'No';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $base_args = [
                                        'page' => 'boag_generated_articles',
                                    ];

                                    $view_url = add_query_arg(
                                        array_merge(
                                            $base_args,
                                            [ 'view_article_id' => $article_id ]
                                        ),
                                        admin_url( 'admin.php' )
                                    );

                                    $delete_url = wp_nonce_url(
                                        add_query_arg(
                                            array_merge(
                                                $base_args,
                                                [
                                                    'boag_action' => 'delete',
                                                    'article_id'  => $article_id,
                                                    'boag_paged'  => $paged,
                                                ]
                                            ),
                                            admin_url( 'admin.php' )
                                        ),
                                        'boag_delete_article_' . $article_id
                                    );

                                    if ( $sent_post_id ) {
                                        $send_button = '<span>Already sent</span>';
                                    } else {
                                        $send_url = wp_nonce_url(
                                            add_query_arg(
                                                array_merge(
                                                    $base_args,
                                                    [
                                                        'boag_action' => 'send',
                                                        'article_id'  => $article_id,
                                                        'boag_paged'  => $paged,
                                                    ]
                                                ),
                                                admin_url( 'admin.php' )
                                            ),
                                            'boag_send_article_' . $article_id
                                        );
                                        $send_button = '<a class="button button-primary" href="' . esc_url( $send_url ) . '">Send to Draft Posts</a>';
                                    }
                                    ?>
                                    <a class="button" href="<?php echo esc_url( $view_url ); ?>">View</a>
                                    <?php echo $send_button; ?>
                                    <a class="button button-link-delete" href="<?php echo esc_url( $delete_url ); ?>" onclick="return confirm('Delete this generated article?');">Delete</a>
                                </td>
                            </tr>
                        <?php endwhile; wp_reset_postdata(); ?>
                    </tbody>
                </table>

                <?php
                $total_pages = $query->max_num_pages;
                if ( $total_pages > 1 ) {
                    $current_url = remove_query_arg( 'boag_paged', admin_url( 'admin.php?page=boag_generated_articles' ) );
                    echo '<div class="tablenav"><div class="tablenav-pages">';
                    echo paginate_links( [
                        'base'      => add_query_arg( 'boag_paged', '%#%', $current_url ),
                        'format'    => '',
                        'current'   => $paged,
                        'total'     => $total_pages,
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                    ] );
                    echo '</div></div>';
                }
                ?>
            <?php endif; ?>

            <?php if ( $view_article_id ) : ?>
                <?php
                $article = get_post( $view_article_id );
                if ( $article && $article->post_type === self::CPT_SLUG ) :
                    ?>
                    <hr />
                    <h2>Preview: <?php echo esc_html( get_the_title( $view_article_id ) ); ?></h2>
                    <div style="background:#fff; border:1px solid #ccd0d4; padding:20px; max-height:500px; overflow:auto;">
                        <?php
                        // Content is HTML already; don't auto-wrap with <p>
                        echo wp_kses_post( $article->post_content );
                        ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    private function send_article_to_draft_post( $article_id ) {
        $article = get_post( $article_id );
        if ( ! $article || $article->post_type !== self::CPT_SLUG ) {
            echo '<div class="notice notice-error"><p>Invalid article.</p></div>';
            return;
        }

        $existing = get_post_meta( $article_id, '_boag_sent_post_id', true );
        if ( $existing ) {
            echo '<div class="notice notice-warning"><p>This article has already been sent to draft posts.</p></div>';
            return;
        }

        $categories = (array) get_post_meta( $article_id, '_boag_categories', true );
        $tags_raw   = (string) get_post_meta( $article_id, '_boag_tags', true );
        $tags_input = [];

        if ( ! empty( $tags_raw ) ) {
            $tags_input = array_filter( array_map( 'trim', explode( ',', $tags_raw ) ) );
        }

        $new_post_id = wp_insert_post( [
            'post_type'    => 'post',
            'post_status'  => 'draft',
            'post_title'   => $article->post_title,
            'post_content' => $article->post_content,
            'post_category'=> $categories,
            'tags_input'   => $tags_input,
        ] );

        if ( is_wp_error( $new_post_id ) || ! $new_post_id ) {
            echo '<div class="notice notice-error"><p>Failed to create WordPress draft post.</p></div>';
            return;
        }

        update_post_meta( $article_id, '_boag_sent_post_id', $new_post_id );

        $edit_link = get_edit_post_link( $new_post_id );
        if ( $edit_link ) {
            echo '<div class="notice notice-success"><p>Article sent to WordPress draft posts. <a href="' . esc_url( $edit_link ) . '">Edit draft</a></p></div>';
        } else {
            echo '<div class="notice notice-success"><p>Article sent to WordPress draft posts.</p></div>';
        }
    }

    /* ========== UNINSTALL CLEANUP ========== */

    public static function uninstall() {
        if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
            return;
        }

        $delete_on_uninstall = (int) get_option( 'boag_delete_on_uninstall', 0 );
        if ( ! $delete_on_uninstall ) {
            return;
        }

        // Delete all custom post type entries
        $posts = get_posts( [
            'post_type'      => self::CPT_SLUG,
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ] );

        if ( $posts ) {
            foreach ( $posts as $post_id ) {
                wp_delete_post( $post_id, true );
            }
        }

        // Delete plugin options
        delete_option( 'boag_api_key' );
        delete_option( 'boag_default_model' );
        delete_option( 'boag_default_language' );
        delete_option( 'boag_delete_on_uninstall' );
    }

}

new Bulk_OpenAI_Article_Generator();

// Register uninstall hook
register_uninstall_hook( __FILE__, [ 'Bulk_OpenAI_Article_Generator', 'uninstall' ] );

