// Helper: tabovi
function m9pgpShowTab(index) {
    const tabs = document.querySelectorAll('.m9pgp-tool .m9pgp-tab');
    const contents = document.querySelectorAll('.m9pgp-tool .m9pgp-tab-content');

    tabs.forEach((tab, i) => {
        if (i === index) {
            tab.classList.add('m9pgp-tab-active');
        } else {
            tab.classList.remove('m9pgp-tab-active');
        }
    });

    contents.forEach((content, i) => {
        if (i === index) {
            content.classList.add('m9pgp-tab-content-active');
        } else {
            content.classList.remove('m9pgp-tab-content-active');
        }
    });
}

function m9pgpToggleLoader(id, show) {
    const el = document.getElementById(id);
    if (!el) return;
    el.style.display = show ? 'inline-block' : 'none';
}

function m9pgpShowToast(message) {
    const toast = document.getElementById('m9pgp-toast');
    if (!toast) return;

    toast.textContent = message;
    toast.classList.add('m9pgp-toast-show');

    setTimeout(() => {
        toast.classList.remove('m9pgp-toast-show');
    }, 2500);
}

async function m9pgpCopyToClipboard(id) {
    const el = document.getElementById(id);
    if (!el) return;

    const text = el.value !== undefined ? el.value : el.textContent;

    try {
        if (navigator.clipboard && navigator.clipboard.writeText) {
            await navigator.clipboard.writeText(text);
        } else {
            // Fallback za starije browsere
            el.select();
            document.execCommand('copy');
        }
        m9pgpShowToast('Copied to clipboard!');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Copy failed: ' + e.message);
    }
}

/* ===========================
   Generate Keys (RSA 2048–8192)
   =========================== */
async function m9pgpGenerateKeys() {
    m9pgpToggleLoader('m9pgp-key-loader', true);
    const progress = document.getElementById('m9pgp-key-progress');

    if (progress) progress.style.width = '10%';

    try {
        const name = (document.getElementById('m9pgp-name').value || 'Anonymous').trim();
        const email = (document.getElementById('m9pgp-email').value || 'anon@example.com').trim();
        const pass = document.getElementById('m9pgp-password').value || '';
        const bits = parseInt(document.getElementById('m9pgp-key-size').value, 10) || 2048;

        if (progress) progress.style.width = '25%';

        const { privateKey, publicKey } = await openpgp.generateKey({
            type: 'rsa',
            rsaBits: bits,
            userIDs: [{ name, email }],
            passphrase: pass || undefined
        });

        if (progress) progress.style.width = '80%';

        document.getElementById('m9pgp-public-key').value = publicKey;
        document.getElementById('m9pgp-private-key').value = privateKey;

        if (progress) progress.style.width = '100%';
        m9pgpShowToast('Key pair generated!');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Error: ' + e.message);
    }

    setTimeout(() => {
        if (progress) progress.style.width = '0';
    }, 1200);

    m9pgpToggleLoader('m9pgp-key-loader', false);
}

/* =============
   Encrypt
   ============= */
async function m9pgpEncryptMessage() {
    m9pgpToggleLoader('m9pgp-encrypt-loader', true);

    try {
        const pubArmored = document.getElementById('m9pgp-encrypt-key').value.trim();
        const text = document.getElementById('m9pgp-encrypt-text').value;

        if (!pubArmored || !text) {
            throw new Error('Public key and message are required.');
        }

        const publicKey = await openpgp.readKey({ armoredKey: pubArmored });
        const message = await openpgp.createMessage({ text });

        const encrypted = await openpgp.encrypt({
            message,
            encryptionKeys: publicKey
        });

        document.getElementById('m9pgp-ciphertext').value = encrypted;
        m9pgpShowToast('Message encrypted!');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Error: ' + e.message);
    }

    m9pgpToggleLoader('m9pgp-encrypt-loader', false);
}

/* =============
   Decrypt
   ============= */
async function m9pgpDecryptMessage() {
    m9pgpToggleLoader('m9pgp-decrypt-loader', true);

    try {
        const privArmored = document.getElementById('m9pgp-decrypt-key').value.trim();
        const pass = document.getElementById('m9pgp-decrypt-password').value || '';
        const ciphertext = document.getElementById('m9pgp-decrypt-ciphertext').value.trim();

        if (!privArmored || !ciphertext) {
            throw new Error('Private key and encrypted message are required.');
        }

        const privateKey = await openpgp.readPrivateKey({ armoredKey: privArmored });
        const decryptedKey = pass
            ? await openpgp.decryptKey({ privateKey, passphrase: pass })
            : privateKey;

        const message = await openpgp.readMessage({ armoredMessage: ciphertext });
        const { data } = await openpgp.decrypt({
            message,
            decryptionKeys: decryptedKey
        });

        document.getElementById('m9pgp-decrypted').value = data;
        m9pgpShowToast('Message decrypted!');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Error: ' + e.message);
    }

    m9pgpToggleLoader('m9pgp-decrypt-loader', false);
}

/* =============
   Sign (Cleartext)
   ============= */
async function m9pgpSignMessage() {
    m9pgpToggleLoader('m9pgp-sign-loader', true);

    try {
        const privArmored = document.getElementById('m9pgp-sign-key').value.trim();
        const pass = document.getElementById('m9pgp-sign-password').value || '';
        const text = document.getElementById('m9pgp-sign-text').value;

        if (!privArmored || !text) {
            throw new Error('Private key and message are required.');
        }

        const privateKey = await openpgp.readPrivateKey({ armoredKey: privArmored });
        const decryptedKey = pass
            ? await openpgp.decryptKey({ privateKey, passphrase: pass })
            : privateKey;

        const cleartextMessage = await openpgp.createCleartextMessage({ text });

        const signed = await openpgp.sign({
            message: cleartextMessage,
            signingKeys: decryptedKey
        });

        document.getElementById('m9pgp-signed-message').value = signed;
        m9pgpShowToast('Message signed (cleartext)!');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Error: ' + e.message);
    }

    m9pgpToggleLoader('m9pgp-sign-loader', false);
}

/* =============
   Verify (Cleartext)
   ============= */
async function m9pgpVerifyMessage() {
    m9pgpToggleLoader('m9pgp-verify-loader', true);

    try {
        const pubArmored = document.getElementById('m9pgp-verify-key').value.trim();
        const signedText = document.getElementById('m9pgp-verify-message').value;

        if (!pubArmored || !signedText) {
            throw new Error('Public key and signed message are required.');
        }

        const publicKey = await openpgp.readKey({ armoredKey: pubArmored });
        const cleartextMessage = await openpgp.readCleartextMessage({
            cleartextMessage: signedText
        });

        const verificationResult = await openpgp.verify({
            message: cleartextMessage,
            verificationKeys: publicKey
        });

        const sig = verificationResult.signatures[0];
        let resultText = 'No signature found.';

        if (sig) {
            try {
                const validity = await sig.verified;
                if (validity) {
                    resultText = 'Signature is VALID.';
                    if (sig.keyID) {
                        resultText += '\nKey ID: 0x' + sig.keyID.toHex().toUpperCase();
                    }
                } else {
                    resultText = 'Signature is INVALID.';
                }
            } catch (e) {
                resultText = 'Signature verification failed: ' + e.message;
            }
        }

        document.getElementById('m9pgp-verify-result').value = resultText;
        m9pgpShowToast('Verification finished.');
    } catch (e) {
        console.error(e);
        m9pgpShowToast('Error: ' + e.message);
    }

    m9pgpToggleLoader('m9pgp-verify-loader', false);
}

