<?php
/**
 * Plugin Name: Bulk OpenAI Article Generator
 * Description: Generate SEO articles in bulk using the OpenAI API and push them into WordPress as draft posts (with optional featured images). Includes queue + cron autopublishing.
 * Version: 1.6.1
 * Author: Molly9
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Bulk_OpenAI_Article_Generator {

    const CPT_SLUG  = 'boag_article';
    const DB_VER    = '1.0';
    const CRON_HOOK = 'boag_process_queue_cron';

    public function __construct() {
        add_action( 'init', [ $this, 'register_custom_post_type' ] );
        add_action( 'admin_menu', [ $this, 'register_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );

        // Make sure custom cron schedule exists globally (not only inside admin_init context)
        add_filter( 'cron_schedules', [ $this, 'add_cron_schedules' ] );

        // AJAX for per-article generation with status/progress (small batches)
        add_action( 'wp_ajax_boag_generate_article', [ $this, 'ajax_generate_article' ] );

        // Queue actions
        add_action( 'wp_ajax_boag_enqueue_batch', [ $this, 'ajax_enqueue_batch' ] );

        // Cron worker
        add_action( self::CRON_HOOK, [ $this, 'process_queue' ] );
    }

    /* ===========================
     *  ACTIVATION / DB
     * =========================== */

    public static function activate() {
        self::create_or_update_tables();
        self::ensure_cron_scheduled();
    }

    public static function deactivate() {
        // Unschedule cron
        $timestamp = wp_next_scheduled( self::CRON_HOOK );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, self::CRON_HOOK );
        }
    }

    private static function ensure_cron_scheduled() {
        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            // Schedule first run in 1 minute
            wp_schedule_event( time() + 60, 'minute', self::CRON_HOOK );
        }
    }

    public static function create_or_update_tables() {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';
        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            status VARCHAR(20) NOT NULL DEFAULT 'queued',
            job_type VARCHAR(30) NOT NULL DEFAULT 'generate_article',
            payload LONGTEXT NOT NULL,
            attempts INT(11) NOT NULL DEFAULT 0,
            last_error TEXT NULL,
            created_at DATETIME NOT NULL,
            updated_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY status (status),
            KEY job_type (job_type),
            KEY created_at (created_at)
        ) {$charset_collate};";

        dbDelta( $sql );

        update_option( 'boag_db_version', self::DB_VER, false );

        // Ensure cron schedule exists
        self::ensure_cron_scheduled();
    }

    public function add_cron_schedules( $schedules ) {
        if ( ! isset( $schedules['minute'] ) ) {
            $schedules['minute'] = [
                'interval' => 60,
                'display'  => 'Every Minute (BOAG)'
            ];
        }
        return $schedules;
    }

    /* ===========================
     *  LOGGING
     * =========================== */

    private function log_error( $context, $message ) {
        $log = get_option( 'boag_error_log', [] );
        if ( ! is_array( $log ) ) {
            $log = [];
        }

        $log[] = [
            'time'    => current_time( 'mysql' ),
            'context' => (string) $context,
            'message' => (string) $message,
        ];

        // Keep last 500 entries
        if ( count( $log ) > 500 ) {
            $log = array_slice( $log, -500 );
        }

        update_option( 'boag_error_log', $log, false );
    }

    /* ===========================
     *  CPT
     * =========================== */

    public function register_custom_post_type() {
        $args = [
            'labels'             => [ 'name' => 'AI Articles', 'singular_name' => 'AI Article' ],
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'            => false,
            'show_in_menu'       => false,
            'query_var'          => false,
            'rewrite'            => false,
            'capability_type'    => 'post',
            'has_archive'        => false,
            'hierarchical'       => false,
            'supports'           => [ 'title', 'editor' ],
        ];

        register_post_type( self::CPT_SLUG, $args );
    }

    /* ===========================
     *  ADMIN MENUS
     * =========================== */

    public function register_admin_menu() {
        add_menu_page(
            'AI Bulk Articles',
            'AI Bulk Articles',
            'manage_options',
            'boag_generate',
            [ $this, 'render_generate_page' ],
            'dashicons-edit',
            81
        );

        add_submenu_page(
            'boag_generate',
            'Generate Articles',
            'Generate Articles',
            'manage_options',
            'boag_generate',
            [ $this, 'render_generate_page' ]
        );

        add_submenu_page(
            'boag_generate',
            'Generated Articles',
            'Generated Articles',
            'manage_options',
            'boag_generated_articles',
            [ $this, 'render_generated_articles_page' ]
        );

        add_submenu_page(
            'boag_generate',
            'Settings',
            'Settings',
            'manage_options',
            'boag_settings',
            [ $this, 'render_settings_page' ]
        );
    }

    /* ===========================
     *  SETTINGS
     * =========================== */

    public function register_settings() {
        register_setting( 'boag_settings_group', 'boag_api_key' );
        register_setting( 'boag_settings_group', 'boag_default_model' );
        register_setting( 'boag_settings_group', 'boag_default_language' );

        // Default image model (free text)
        register_setting( 'boag_settings_group', 'boag_default_image_model' );

        // Queue + cron
        register_setting( 'boag_settings_group', 'boag_queue_enabled' );
        register_setting( 'boag_settings_group', 'boag_queue_items_per_run' );
        register_setting( 'boag_settings_group', 'boag_queue_max_attempts' );

        // Auto-publishing
        register_setting( 'boag_settings_group', 'boag_autopublish_enabled' );
        register_setting( 'boag_settings_group', 'boag_autopublish_posts_per_run' );
        register_setting( 'boag_settings_group', 'boag_autopublish_interval_minutes' );

        register_setting( 'boag_settings_group', 'boag_delete_on_uninstall' );
    }

    private function get_default_image_model() {
        $m = get_option( 'boag_default_image_model', 'gpt-image-1' );
        return $m ? $m : 'gpt-image-1';
    }

    private function is_queue_enabled() {
        return (int) get_option( 'boag_queue_enabled', 1 ) === 1;
    }

    private function get_queue_items_per_run() {
        $n = (int) get_option( 'boag_queue_items_per_run', 5 );
        return $n > 0 ? $n : 5;
    }

    private function get_queue_max_attempts() {
        $n = (int) get_option( 'boag_queue_max_attempts', 3 );
        return $n > 0 ? $n : 3;
    }

    private function is_autopublish_enabled() {
        return (int) get_option( 'boag_autopublish_enabled', 0 ) === 1;
    }

    private function get_autopublish_posts_per_run() {
        $n = (int) get_option( 'boag_autopublish_posts_per_run', 3 );
        return $n > 0 ? $n : 3;
    }

    private function get_autopublish_interval_minutes() {
        $n = (int) get_option( 'boag_autopublish_interval_minutes', 10 );
        return $n > 0 ? $n : 10;
    }

    /* ===========================
     *  SETTINGS PAGE (incl. error log)
     * =========================== */

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Clear log action
        if ( isset( $_POST['boag_clear_log'] ) ) {
            check_admin_referer( 'boag_clear_log_action' );
            delete_option( 'boag_error_log' );
            echo '<div class="notice notice-success"><p>Error log cleared.</p></div>';
        }

        // Ensure tables exist (if user updated plugin without deactivate/activate)
        self::create_or_update_tables();

        $delete_on_uninstall = (int) get_option( 'boag_delete_on_uninstall', 0 );
        $log                 = get_option( 'boag_error_log', [] );
        if ( ! is_array( $log ) ) $log = [];

        $log_lines = [];
        foreach ( $log as $entry ) {
            $time    = isset( $entry['time'] ) ? $entry['time'] : '';
            $context = isset( $entry['context'] ) ? $entry['context'] : '';
            $message = isset( $entry['message'] ) ? $entry['message'] : '';
            $log_lines[] = sprintf( '[%s] [%s] %s', $time, $context, $message );
        }
        $log_text = implode( "\n", $log_lines );

        $queue_enabled  = (int) get_option( 'boag_queue_enabled', 1 );
        $items_per_run  = (int) get_option( 'boag_queue_items_per_run', 5 );
        $max_attempts   = (int) get_option( 'boag_queue_max_attempts', 3 );

        $ap_enabled     = (int) get_option( 'boag_autopublish_enabled', 0 );
        $ap_per_run     = (int) get_option( 'boag_autopublish_posts_per_run', 3 );
        $ap_interval    = (int) get_option( 'boag_autopublish_interval_minutes', 10 );

        $default_image_model = esc_attr( $this->get_default_image_model() );
        ?>
        <div class="wrap">
            <h1>AI Bulk Articles – Settings</h1>

            <form method="post" action="options.php">
                <?php settings_fields( 'boag_settings_group' ); ?>

                <h2>OpenAI</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_api_key">OpenAI API Key</label></th>
                        <td>
                            <input type="password" name="boag_api_key" id="boag_api_key"
                                   value="<?php echo esc_attr( get_option( 'boag_api_key', '' ) ); ?>"
                                   class="regular-text" />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_default_model">Default text model</label></th>
                        <td>
                            <input type="text" name="boag_default_model" id="boag_default_model"
                                   value="<?php echo esc_attr( get_option( 'boag_default_model', 'gpt-4.1-mini' ) ); ?>"
                                   class="regular-text" />
                            <p class="description">Examples: <code>gpt-4.1-mini</code>, <code>gpt-4.1</code>, <code>gpt-4o-mini</code>.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_default_language">Default language</label></th>
                        <td>
                            <input type="text" name="boag_default_language" id="boag_default_language"
                                   value="<?php echo esc_attr( get_option( 'boag_default_language', 'English' ) ); ?>"
                                   class="regular-text" />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_default_image_model">Default image model</label></th>
                        <td>
                            <input type="text" name="boag_default_image_model" id="boag_default_image_model"
                                   value="<?php echo $default_image_model; ?>"
                                   class="regular-text" list="boag-image-model-suggestions" />
                            <datalist id="boag-image-model-suggestions">
                                <option value="gpt-image-1"></option>
                                <option value="dall-e-3"></option>
                                <option value="dall-e-2"></option>
                            </datalist>
                            <p class="description">
                                You can type any image model supported by your account.
                                Recommended: <code>gpt-image-1</code>.
                            </p>
                        </td>
                    </tr>
                </table>

                <h2>Queue (background processing)</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_queue_enabled">Enable queue</label></th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_queue_enabled" id="boag_queue_enabled" value="1" <?php checked( $queue_enabled, 1 ); ?> />
                                Process large batches via queue (recommended for 100+ articles)
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_queue_items_per_run">Items per cron run</label></th>
                        <td>
                            <input type="number" class="small-text" name="boag_queue_items_per_run" id="boag_queue_items_per_run"
                                   value="<?php echo esc_attr( $items_per_run ); ?>" min="1" max="100" />
                            <p class="description">How many jobs to process per cron run. Example: 5–20.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_queue_max_attempts">Max attempts</label></th>
                        <td>
                            <input type="number" class="small-text" name="boag_queue_max_attempts" id="boag_queue_max_attempts"
                                   value="<?php echo esc_attr( $max_attempts ); ?>" min="1" max="10" />
                            <p class="description">If OpenAI fails, the job will be retried up to this many times.</p>
                        </td>
                    </tr>
                </table>

                <h2>Cron auto-publishing</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_autopublish_enabled">Enable auto publishing</label></th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_autopublish_enabled" id="boag_autopublish_enabled" value="1" <?php checked( $ap_enabled, 1 ); ?> />
                                Automatically send generated articles to WordPress drafts and publish them gradually
                            </label>
                            <p class="description">
                                Auto-publishing flow: generated AI article → create WP draft (taxonomy + featured image if enabled) → publish at a controlled pace.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_autopublish_posts_per_run">Posts per run</label></th>
                        <td>
                            <input type="number" class="small-text" name="boag_autopublish_posts_per_run" id="boag_autopublish_posts_per_run"
                                   value="<?php echo esc_attr( $ap_per_run ); ?>" min="1" max="50" />
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_autopublish_interval_minutes">Interval minutes</label></th>
                        <td>
                            <input type="number" class="small-text" name="boag_autopublish_interval_minutes" id="boag_autopublish_interval_minutes"
                                   value="<?php echo esc_attr( $ap_interval ); ?>" min="1" max="1440" />
                            <p class="description">Example: 10 minutes = every 10 minutes publish up to “Posts per run”.</p>
                        </td>
                    </tr>
                </table>

                <h2>Uninstall</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_delete_on_uninstall">Delete data on uninstall</label></th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_delete_on_uninstall" id="boag_delete_on_uninstall"
                                       value="1" <?php checked( $delete_on_uninstall, 1 ); ?> />
                                Delete all generated AI articles, queue data, error log, and plugin settings when the plugin is deleted.
                            </label>
                        </td>
                    </tr>
                </table>

                <?php submit_button(); ?>
            </form>

            <hr />

            <h2>Error log</h2>
            <p class="description">OpenAI / image / upload errors are logged here.</p>

            <textarea readonly rows="15" style="width:100%;max-width:900px;"><?php echo esc_textarea( $log_text ); ?></textarea>

            <form method="post" style="margin-top:10px;">
                <?php wp_nonce_field( 'boag_clear_log_action' ); ?>
                <input type="hidden" name="boag_clear_log" value="1" />
                <?php submit_button( 'Clear Error Log', 'secondary', 'submit', false ); ?>
            </form>
        </div>
        <?php
    }

    /* ===========================
     *  GENERATE PAGE
     * =========================== */

    public function render_generate_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Ensure tables + cron exist
        self::create_or_update_tables();

        $api_key          = get_option( 'boag_api_key', '' );
        $default_language = get_option( 'boag_default_language', 'English' );
        $ajax_nonce       = wp_create_nonce( 'boag_generate_ajax' );

        $default_img_model = esc_attr( $this->get_default_image_model() );

        $categories = get_categories( [ 'hide_empty' => false ] );

        // Queue summary
        $queue_counts = $this->get_queue_counts();

        ?>
        <div class="wrap">
            <h1>AI Bulk Article Generator</h1>

            <?php if ( empty( $api_key ) ) : ?>
                <div class="notice notice-error">
                    <p><strong>OpenAI API key is not set.</strong> Please go to the Settings tab and add it before generating articles.</p>
                </div>
            <?php endif; ?>

            <div class="notice notice-info">
                <p>
                    <strong>Queue status:</strong>
                    Queued: <?php echo (int) $queue_counts['queued']; ?>,
                    Processing: <?php echo (int) $queue_counts['processing']; ?>,
                    Done: <?php echo (int) $queue_counts['done']; ?>,
                    Failed: <?php echo (int) $queue_counts['failed']; ?>
                </p>
            </div>

            <form method="post" id="boag-generate-form">
                <input type="hidden" name="boag_ajax_run" value="1" />

                <h2 class="title">Generation Mode</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Mode</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio" name="boag_mode" value="topic" checked="checked" />
                                    Generate X articles from a topic keyword
                                </label><br />
                                <label>
                                    <input type="radio" name="boag_mode" value="titles" />
                                    Generate articles from provided titles (one per line)
                                </label>
                            </fieldset>
                        </td>
                    </tr>

                    <tr class="boag-mode-topic">
                        <th scope="row"><label for="boag_topic_keyword">Topic keyword</label></th>
                        <td>
                            <input type="text" name="boag_topic_keyword" id="boag_topic_keyword"
                                   class="regular-text" />
                        </td>
                    </tr>

                    <tr class="boag-mode-topic">
                        <th scope="row"><label for="boag_article_count">Number of articles</label></th>
                        <td>
                            <input type="number" name="boag_article_count" id="boag_article_count"
                                   class="small-text" value="3" min="1" />
                            <p class="description">For 100+ articles, using the queue is recommended.</p>
                        </td>
                    </tr>

                    <tr class="boag-mode-titles" style="display:none;">
                        <th scope="row"><label for="boag_titles">Article titles</label></th>
                        <td>
                            <textarea name="boag_titles" id="boag_titles" class="large-text code" rows="8"
                                      placeholder="Title 1&#10;Title 2&#10;Title 3"></textarea>
                        </td>
                    </tr>
                </table>

                <h2 class="title">Article Options</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="boag_language">Language</label></th>
                        <td>
                            <input type="text" name="boag_language" id="boag_language"
                                   value="<?php echo esc_attr( $default_language ); ?>" class="regular-text" />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_length">Length (words)</label></th>
                        <td>
                            <input type="number" name="boag_length" id="boag_length" class="small-text" value="1200" min="200" />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_tone">Tone</label></th>
                        <td>
                            <input type="text" name="boag_tone" id="boag_tone" class="regular-text"
                                   placeholder="informative, friendly, professional, casual..." />
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_instructions">Additional instructions</label></th>
                        <td>
                            <textarea name="boag_instructions" id="boag_instructions"
                                      class="large-text code" rows="5"
                                      placeholder="Use H2/H3, bullet lists, FAQ section, etc. HTML only."></textarea>
                        </td>
                    </tr>
                </table>

                <h2 class="title">Taxonomy (optional)</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Categories</th>
                        <td>
                            <?php if ( ! empty( $categories ) ) : ?>
                                <?php foreach ( $categories as $cat ) : ?>
                                    <label>
                                        <input type="checkbox" name="boag_categories[]" value="<?php echo esc_attr( $cat->term_id ); ?>" />
                                        <?php echo esc_html( $cat->name ); ?>
                                    </label><br />
                                <?php endforeach; ?>
                            <?php else : ?>
                                <p class="description">No categories found. Create some under Posts → Categories.</p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_tags">Tags</label></th>
                        <td>
                            <input type="text" name="boag_tags" id="boag_tags" class="regular-text"
                                   placeholder="tag1, tag2, tag3" />
                        </td>
                    </tr>
                </table>

                <h2 class="title">Featured Image (optional)</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Featured image</th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_generate_featured_image" value="1" />
                                Generate and set featured image
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_image_prompt">Image prompt (style)</label></th>
                        <td>
                            <textarea name="boag_image_prompt" id="boag_image_prompt" rows="3" class="large-text"
                                      placeholder="e.g. realistic photo, warm tones, minimal background, no text..."></textarea>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="boag_image_aspect">Image aspect</label></th>
                        <td>
                            <select name="boag_image_aspect" id="boag_image_aspect">
                                <option value="square">Square (1024x1024)</option>
                                <option value="landscape">Landscape (1536x1024)</option>
                            </select>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="boag_image_model">Image model (optional override)</label></th>
                        <td>
                            <input type="text" name="boag_image_model" id="boag_image_model"
                                   class="regular-text" value="<?php echo $default_img_model; ?>"
                                   list="boag-image-model-suggestions-gen" />
                            <datalist id="boag-image-model-suggestions-gen">
                                <option value="gpt-image-1"></option>
                                <option value="dall-e-3"></option>
                                <option value="dall-e-2"></option>
                            </datalist>
                            <p class="description">
                                Type any model name you want (e.g. <code>gpt-image-1</code>). If changed, it applies only to this batch.
                            </p>
                        </td>
                    </tr>
                </table>

                <h2 class="title">Batch processing</h2>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Use queue</th>
                        <td>
                            <label>
                                <input type="checkbox" name="boag_use_queue" id="boag_use_queue" value="1" <?php checked( $this->is_queue_enabled(), true ); ?> />
                                Put this batch into the queue (recommended for very large batches)
                            </label>
                            <p class="description">If enabled, the batch will be processed in the background via cron, without freezing your browser.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button( 'Generate Articles', 'primary', 'boag_generate_submit' ); ?>
            </form>

            <div id="boag-generation-status" style="margin-top:20px;"></div>
            <div id="boag-progress-bar-wrap" style="width:100%;max-width:400px;background:#f1f1f1;border:1px solid #ccd0d4;height:20px;border-radius:3px;display:none;">
                <div id="boag-progress-bar" style="height:100%;width:0%;background:#0073aa;"></div>
            </div>
        </div>

        <script>
        (function($) {
            var boagGenerateAjaxNonce = '<?php echo esc_js( $ajax_nonce ); ?>';

            const modeRadios = document.querySelectorAll('input[name="boag_mode"]');
            const topicRows = document.querySelectorAll('.boag-mode-topic');
            const titleRows = document.querySelectorAll('.boag-mode-titles');

            function updateMode() {
                let mode = 'topic';
                modeRadios.forEach(function(r) { if (r.checked) mode = r.value; });
                if (mode === 'topic') {
                    topicRows.forEach(function(el) { el.style.display = ''; });
                    titleRows.forEach(function(el) { el.style.display = 'none'; });
                } else {
                    topicRows.forEach(function(el) { el.style.display = 'none'; });
                    titleRows.forEach(function(el) { el.style.display = ''; });
                }
            }

            modeRadios.forEach(function(r) { r.addEventListener('change', updateMode); });
            updateMode();

            const form = document.getElementById('boag-generate-form');
            const statusBox = document.getElementById('boag-generation-status');
            const progressWrap = document.getElementById('boag-progress-bar-wrap');
            const progressBar = document.getElementById('boag-progress-bar');

            function gatherCommon() {
                const language     = (form.querySelector('#boag_language').value || '').trim();
                const length       = parseInt(form.querySelector('#boag_length').value, 10) || 1200;
                const tone         = (form.querySelector('#boag_tone').value || '').trim();
                const instructions = (form.querySelector('#boag_instructions').value || '').trim();

                const catEls = form.querySelectorAll('input[name="boag_categories[]"]:checked');
                let categories = [];
                catEls.forEach(function(el) { categories.push(el.value); });

                const tags = (form.querySelector('#boag_tags').value || '').trim();

                const fiEl = form.querySelector('input[name="boag_generate_featured_image"]');
                const generateFeaturedImage = fiEl && fiEl.checked ? '1' : '0';

                const imagePrompt = (form.querySelector('#boag_image_prompt').value || '').trim();
                const imageAspect = (form.querySelector('#boag_image_aspect').value || 'square');
                const imageModel  = (form.querySelector('#boag_image_model').value || '').trim();

                return { language, length, tone, instructions, categories, tags, generateFeaturedImage, imagePrompt, imageAspect, imageModel };
            }

            function buildTasks() {
                let mode = 'topic';
                modeRadios.forEach(function(r) { if (r.checked) mode = r.value; });

                let tasks = [];

                if (mode === 'topic') {
                    const topic = (form.querySelector('#boag_topic_keyword').value || '').trim();
                    const count = parseInt(form.querySelector('#boag_article_count').value, 10) || 1;
                    if (!topic || count < 1) {
                        return { error: 'Please provide a topic keyword and a valid article count.' };
                    }
                    for (let i = 0; i < count; i++) tasks.push({ mode: 'topic', topic: topic });
                } else {
                    const titlesRaw = form.querySelector('#boag_titles').value || '';
                    const lines = titlesRaw.split(/\r?\n/).map(s => s.trim()).filter(Boolean);
                    if (!lines.length) {
                        return { error: 'Please provide at least one title.' };
                    }
                    lines.forEach(title => tasks.push({ mode: 'titles', title: title }));
                }

                return { tasks };
            }

            function disableBtn(disabled) {
                const submitBtn = form.querySelector('input[type="submit"], button[type="submit"]');
                if (submitBtn) submitBtn.disabled = disabled;
            }

            if (form) {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();

                    statusBox.innerHTML = '';
                    progressWrap.style.display = 'none';
                    progressBar.style.width = '0%';

                    disableBtn(true);

                    const useQueueEl = form.querySelector('#boag_use_queue');
                    const useQueue = useQueueEl && useQueueEl.checked;

                    const common = gatherCommon();
                    const built = buildTasks();
                    if (built.error) {
                        statusBox.innerHTML = '<div class="notice notice-error"><p>' + built.error + '</p></div>';
                        disableBtn(false);
                        return;
                    }

                    const tasks = built.tasks || [];
                    if (!tasks.length) {
                        statusBox.innerHTML = '<div class="notice notice-error"><p>No tasks to run.</p></div>';
                        disableBtn(false);
                        return;
                    }

                    if (useQueue) {
                        // Enqueue whole batch in one call
                        statusBox.innerHTML = '<p>Enqueuing ' + tasks.length + ' job(s) into the queue...</p>';

                        $.post(ajaxurl, {
                            action: 'boag_enqueue_batch',
                            security: boagGenerateAjaxNonce,
                            tasks: tasks,
                            common: common
                        }).done(function(resp){
                            if (resp && resp.success) {
                                statusBox.innerHTML = '<div class="notice notice-success"><p>Queued ' + resp.data.enqueued + ' job(s). The queue will be processed by cron.</p></div>' +
                                                     '<p><a href="<?php echo esc_url( admin_url('admin.php?page=boag_generated_articles') ); ?>">View generated articles</a></p>';
                            } else {
                                let msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Unknown error';
                                statusBox.innerHTML = '<div class="notice notice-error"><p>Queue enqueue failed: ' + msg + '</p></div>';
                            }
                            disableBtn(false);
                        }).fail(function(){
                            statusBox.innerHTML = '<div class="notice notice-error"><p>Queue enqueue failed: AJAX error.</p></div>';
                            disableBtn(false);
                        });

                        return;
                    }

                    // Otherwise run sequential AJAX (good for smaller batches)
                    let total = tasks.length, completed = 0, failed = 0;
                    progressWrap.style.display = 'block';

                    statusBox.innerHTML = '<p>Starting generation of ' + total + ' article(s)...</p><ul id="boag-status-list"></ul>';
                    const statusList = document.getElementById('boag-status-list');

                    function updateProgress() {
                        const percent = Math.round((completed / total) * 100);
                        progressBar.style.width = percent + '%';
                    }

                    function addStatus(message, success) {
                        if (!statusList) return;
                        const li = document.createElement('li');
                        li.textContent = message;
                        li.style.color = success ? 'green' : 'red';
                        statusList.appendChild(li);
                    }

                    function runTask(index) {
                        if (index >= total) {
                            statusBox.insertAdjacentHTML('beforeend',
                                '<p><strong>Done.</strong> Generated ' + (total - failed) + ' article(s), ' + failed + ' failed.</p>' +
                                '<p><a href="<?php echo esc_url( admin_url('admin.php?page=boag_generated_articles') ); ?>">View generated articles</a></p>'
                            );
                            disableBtn(false);
                            return;
                        }

                        const t = tasks[index];

                        const data = {
                            action: 'boag_generate_article',
                            security: boagGenerateAjaxNonce,
                            mode: t.mode,
                            topic: t.topic || '',
                            title: t.title || '',
                            language: common.language,
                            length: common.length,
                            tone: common.tone,
                            instructions: common.instructions,
                            categories: common.categories,
                            tags: common.tags,
                            generate_featured_image: common.generateFeaturedImage,
                            image_prompt: common.imagePrompt,
                            image_aspect: common.imageAspect,
                            image_model: common.imageModel
                        };

                        addStatus('Generating article ' + (index + 1) + ' of ' + total + '...', true);

                        $.post(ajaxurl, data).done(function(resp){
                            completed++;
                            if (resp && resp.success && resp.data) {
                                addStatus('Generated: "' + (resp.data.title || 'Untitled') + '"', true);
                            } else {
                                failed++;
                                let msg = (resp && resp.data && resp.data.message) ? resp.data.message : 'Unknown error';
                                addStatus('Failed: ' + msg, false);
                            }
                            updateProgress();
                            runTask(index + 1);
                        }).fail(function(){
                            completed++;
                            failed++;
                            addStatus('Failed: AJAX error.', false);
                            updateProgress();
                            runTask(index + 1);
                        });
                    }

                    updateProgress();
                    runTask(0);
                });
            }
        })(jQuery);
        </script>
        <?php
    }

    /* ===========================
     *  QUEUE: enqueue batch
     * =========================== */

    public function ajax_enqueue_batch() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Insufficient permissions.' ] );
        }
        check_ajax_referer( 'boag_generate_ajax', 'security' );

        if ( ! $this->is_queue_enabled() ) {
            wp_send_json_error( [ 'message' => 'Queue is disabled in settings.' ] );
        }

        $tasks  = isset( $_POST['tasks'] ) ? (array) $_POST['tasks'] : [];
        $common = isset( $_POST['common'] ) ? (array) $_POST['common'] : [];

        if ( empty( $tasks ) ) {
            wp_send_json_error( [ 'message' => 'No tasks provided.' ] );
        }

        $enqueued = 0;
        foreach ( $tasks as $t ) {
            $mode = isset( $t['mode'] ) ? sanitize_text_field( $t['mode'] ) : 'topic';
            $topic = isset( $t['topic'] ) ? sanitize_text_field( $t['topic'] ) : '';
            $title = isset( $t['title'] ) ? sanitize_text_field( $t['title'] ) : '';

            $payload = [
                'mode'         => $mode,
                'topic'        => $topic,
                'title'        => $title,
                'language'     => isset( $common['language'] ) ? sanitize_text_field( $common['language'] ) : get_option( 'boag_default_language', 'English' ),
                'length'       => isset( $common['length'] ) ? max( 100, intval( $common['length'] ) ) : 1200,
                'tone'         => isset( $common['tone'] ) ? sanitize_text_field( $common['tone'] ) : '',
                'instructions' => isset( $common['instructions'] ) ? sanitize_textarea_field( $common['instructions'] ) : '',
                'categories'   => isset( $common['categories'] ) ? array_map( 'intval', (array) $common['categories'] ) : [],
                'tags'         => isset( $common['tags'] ) ? sanitize_text_field( $common['tags'] ) : '',
                'generate_featured_image' => ( isset( $common['generateFeaturedImage'] ) && $common['generateFeaturedImage'] === '1' ) ? 1 : 0,
                'image_prompt' => isset( $common['imagePrompt'] ) ? sanitize_textarea_field( $common['imagePrompt'] ) : '',
                'image_aspect' => isset( $common['imageAspect'] ) ? sanitize_text_field( $common['imageAspect'] ) : 'square',
                'image_model'  => isset( $common['imageModel'] ) ? sanitize_text_field( $common['imageModel'] ) : '',
                'autopublish'  => $this->is_autopublish_enabled() ? 1 : 0,
            ];

            if ( $payload['image_aspect'] !== 'landscape' ) {
                $payload['image_aspect'] = 'square';
            }

            if ( $mode === 'topic' && empty( $topic ) ) continue;
            if ( $mode === 'titles' && empty( $title ) ) continue;

            $ok = $this->queue_insert_job( 'generate_article', $payload );
            if ( $ok ) $enqueued++;
        }

        // Make sure cron is scheduled
        self::ensure_cron_scheduled();

        wp_send_json_success( [ 'enqueued' => $enqueued ] );
    }

    private function queue_insert_job( $job_type, array $payload ) {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';

        $now = current_time( 'mysql' );
        $res = $wpdb->insert(
            $table,
            [
                'status'     => 'queued',
                'job_type'   => sanitize_text_field( $job_type ),
                'payload'    => wp_json_encode( $payload ),
                'attempts'   => 0,
                'last_error' => null,
                'created_at' => $now,
                'updated_at' => $now,
            ],
            [ '%s','%s','%s','%d','%s','%s','%s' ]
        );

        if ( false === $res ) {
            $this->log_error( 'queue_insert', 'DB insert failed: ' . $wpdb->last_error );
            return false;
        }
        return true;
    }

    private function get_queue_counts() {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';
        $counts = [
            'queued'     => 0,
            'processing' => 0,
            'done'       => 0,
            'failed'     => 0,
        ];

        foreach ( array_keys( $counts ) as $st ) {
            $counts[$st] = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE status=%s", $st ) );
        }
        return $counts;
    }

    /* ===========================
     *  CRON WORKER: process queue + autopublish
     * =========================== */

    public function process_queue() {
        // Process queue jobs
        if ( $this->is_queue_enabled() ) {
            $this->process_queue_jobs();
        }

        // Auto-publish pipeline
        if ( $this->is_autopublish_enabled() ) {
            $this->process_autopublish();
        }
    }

    private function process_queue_jobs() {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';

        $limit = $this->get_queue_items_per_run();
        $max_attempts = $this->get_queue_max_attempts();

        $jobs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE status='queued' ORDER BY id ASC LIMIT %d",
                $limit
            ),
            ARRAY_A
        );

        if ( empty( $jobs ) ) {
            return;
        }

        foreach ( $jobs as $job ) {
            $id = (int) $job['id'];

            // set to processing
            $wpdb->update(
                $table,
                [ 'status' => 'processing', 'updated_at' => current_time('mysql') ],
                [ 'id' => $id, 'status' => 'queued' ],
                [ '%s','%s' ],
                [ '%d','%s' ]
            );

            $payload = json_decode( $job['payload'], true );
            if ( ! is_array( $payload ) ) {
                $this->queue_fail_job( $id, 'Invalid payload JSON.' );
                continue;
            }

            $ok = $this->execute_generate_article_job( $payload );

            if ( $ok === true ) {
                $this->queue_complete_job( $id );
            } else {
                $attempts = (int) $job['attempts'] + 1;
                if ( $attempts >= $max_attempts ) {
                    $this->queue_fail_job( $id, is_string($ok) ? $ok : 'Job failed' );
                } else {
                    // retry later
                    $wpdb->update(
                        $table,
                        [
                            'status'     => 'queued',
                            'attempts'   => $attempts,
                            'last_error' => is_string($ok) ? $ok : 'Job failed',
                            'updated_at' => current_time('mysql')
                        ],
                        [ 'id' => $id ],
                        [ '%s','%d','%s','%s' ],
                        [ '%d' ]
                    );
                }
            }
        }
    }

    private function queue_complete_job( $id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';
        $wpdb->update(
            $table,
            [ 'status' => 'done', 'updated_at' => current_time('mysql') ],
            [ 'id' => (int) $id ],
            [ '%s','%s' ],
            [ '%d' ]
        );
    }

    private function queue_fail_job( $id, $error ) {
        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';
        $wpdb->update(
            $table,
            [
                'status'     => 'failed',
                'attempts'   => (int) $wpdb->get_var( $wpdb->prepare("SELECT attempts FROM {$table} WHERE id=%d", (int)$id) ) + 1,
                'last_error' => (string) $error,
                'updated_at' => current_time('mysql'),
            ],
            [ 'id' => (int) $id ],
            [ '%s','%d','%s','%s' ],
            [ '%d' ]
        );
        $this->log_error( 'queue_job_failed', 'Job #' . (int)$id . ': ' . (string)$error );
    }

    private function execute_generate_article_job( array $payload ) {
        $api_key = get_option( 'boag_api_key', '' );
        if ( empty( $api_key ) ) {
            return 'Missing API key.';
        }

        $model = get_option( 'boag_default_model', 'gpt-4.1-mini' );

        $mode  = isset( $payload['mode'] ) ? $payload['mode'] : 'topic';
        $topic = isset( $payload['topic'] ) ? $payload['topic'] : '';
        $title = isset( $payload['title'] ) ? $payload['title'] : '';

        $language     = isset( $payload['language'] ) ? $payload['language'] : get_option( 'boag_default_language', 'English' );
        $length       = isset( $payload['length'] ) ? (int) $payload['length'] : 1200;
        $tone         = isset( $payload['tone'] ) ? $payload['tone'] : '';
        $instructions = isset( $payload['instructions'] ) ? $payload['instructions'] : '';

        $categories = isset( $payload['categories'] ) ? (array) $payload['categories'] : [];
        $tags       = isset( $payload['tags'] ) ? (string) $payload['tags'] : '';

        $generate_fi = ! empty( $payload['generate_featured_image'] ) ? 1 : 0;
        $img_prompt  = isset( $payload['image_prompt'] ) ? (string) $payload['image_prompt'] : '';
        $img_aspect  = isset( $payload['image_aspect'] ) && $payload['image_aspect'] === 'landscape' ? 'landscape' : 'square';
        $img_model   = isset( $payload['image_model'] ) ? (string) $payload['image_model'] : '';

        $autopublish = ! empty( $payload['autopublish'] ) ? 1 : 0;

        if ( $mode === 'topic' ) {
            if ( empty( $topic ) ) return 'Missing topic.';
            $prompt = $this->build_topic_prompt( $topic, $language, $length, $tone, $instructions );
            $expected_title = null;
        } else {
            if ( empty( $title ) ) return 'Missing title.';
            $prompt = $this->build_title_prompt( $title, $language, $length, $tone, $instructions );
            $expected_title = $title;
        }

        $result = $this->call_openai_chat( $api_key, $model, $prompt );
        if ( is_wp_error( $result ) ) {
            $this->log_error( 'queue_chat_generation', $result->get_error_message() );
            return $result->get_error_message();
        }

        $content_raw = trim( $result );

        if ( $mode === 'topic' ) {
            list( $final_title, $body ) = $this->extract_title_and_body_from_response( $content_raw, $topic );
        } else {
            $final_title = $expected_title;
            $body = $content_raw;
        }

        $post_id = wp_insert_post( [
            'post_type'    => self::CPT_SLUG,
            'post_status'  => 'draft',
            'post_title'   => $final_title,
            'post_content' => $body,
        ] );

        if ( is_wp_error( $post_id ) || ! $post_id ) {
            $msg = is_wp_error($post_id) ? $post_id->get_error_message() : 'Failed to save generated article.';
            $this->log_error( 'queue_article_save', $msg );
            return $msg;
        }

        update_post_meta( $post_id, '_boag_language', $language );
        update_post_meta( $post_id, '_boag_tone', $tone );
        update_post_meta( $post_id, '_boag_length', $length );
        update_post_meta( $post_id, '_boag_instructions', $instructions );
        update_post_meta( $post_id, '_boag_categories', array_map('intval', $categories ) );
        update_post_meta( $post_id, '_boag_tags', $tags );
        update_post_meta( $post_id, '_boag_generate_featured_image', $generate_fi );
        update_post_meta( $post_id, '_boag_image_prompt', $img_prompt );
        update_post_meta( $post_id, '_boag_image_aspect', $img_aspect );
        update_post_meta( $post_id, '_boag_image_model', $img_model );
        update_post_meta( $post_id, '_boag_autopublish', $autopublish );

        return true;
    }

    private function process_autopublish() {
        $last = (int) get_option( 'boag_autopublish_last_run', 0 );
        $interval = $this->get_autopublish_interval_minutes() * 60;
        if ( $last > 0 && ( time() - $last ) < $interval ) {
            return;
        }

        $limit = $this->get_autopublish_posts_per_run();

        $q = new WP_Query( [
            'post_type'      => self::CPT_SLUG,
            'post_status'    => 'draft',
            'posts_per_page' => $limit,
            'meta_query'     => [
                [
                    'key'   => '_boag_autopublish',
                    'value' => '1',
                ],
            ],
            'orderby' => 'date',
            'order'   => 'ASC',
        ] );

        if ( ! $q->have_posts() ) {
            update_option( 'boag_autopublish_last_run', time(), false );
            return;
        }

        while ( $q->have_posts() ) {
            $q->the_post();
            $article_id = get_the_ID();

            $sent_post_id = get_post_meta( $article_id, '_boag_sent_post_id', true );
            if ( $sent_post_id ) {
                continue;
            }

            $new_post_id = $this->send_article_to_draft_post_return_id( $article_id );

            if ( $new_post_id ) {
                $upd = wp_update_post( [
                    'ID'          => $new_post_id,
                    'post_status' => 'publish',
                ], true );

                if ( is_wp_error( $upd ) ) {
                    $this->log_error( 'autopublish_publish', $upd->get_error_message() );
                }
            }
        }
        wp_reset_postdata();

        update_option( 'boag_autopublish_last_run', time(), false );
    }

    /* ===========================
     *  AJAX: per-article generation (non-queue)
     * =========================== */

    public function ajax_generate_article() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Insufficient permissions.' ] );
        }
        check_ajax_referer( 'boag_generate_ajax', 'security' );

        $api_key = get_option( 'boag_api_key', '' );
        if ( empty( $api_key ) ) {
            wp_send_json_error( [ 'message' => 'OpenAI API key is missing.' ] );
        }

        $mode         = isset( $_POST['mode'] ) ? sanitize_text_field( $_POST['mode'] ) : 'topic';
        $language     = isset( $_POST['language'] ) ? sanitize_text_field( $_POST['language'] ) : get_option( 'boag_default_language', 'English' );
        $length       = isset( $_POST['length'] ) ? max( 100, intval( $_POST['length'] ) ) : 1200;
        $tone         = isset( $_POST['tone'] ) ? sanitize_text_field( $_POST['tone'] ) : '';
        $instructions = isset( $_POST['instructions'] ) ? sanitize_textarea_field( $_POST['instructions'] ) : '';
        $model        = get_option( 'boag_default_model', 'gpt-4.1-mini' );

        $categories   = isset( $_POST['categories'] ) ? array_map( 'intval', (array) $_POST['categories'] ) : [];
        $tags         = isset( $_POST['tags'] ) ? sanitize_text_field( $_POST['tags'] ) : '';

        $generate_fi  = isset( $_POST['generate_featured_image'] ) && $_POST['generate_featured_image'] === '1' ? 1 : 0;
        $image_prompt = isset( $_POST['image_prompt'] ) ? sanitize_textarea_field( $_POST['image_prompt'] ) : '';
        $image_aspect = isset( $_POST['image_aspect'] ) ? sanitize_text_field( $_POST['image_aspect'] ) : 'square';
        $image_aspect = ( $image_aspect === 'landscape' ) ? 'landscape' : 'square';

        $image_model  = isset( $_POST['image_model'] ) ? sanitize_text_field( $_POST['image_model'] ) : '';

        if ( $mode === 'topic' ) {
            $topic = isset( $_POST['topic'] ) ? sanitize_text_field( $_POST['topic'] ) : '';
            if ( empty( $topic ) ) {
                wp_send_json_error( [ 'message' => 'Topic is required for topic mode.' ] );
            }
            $prompt         = $this->build_topic_prompt( $topic, $language, $length, $tone, $instructions );
            $expected_title = null;
        } else {
            $title = isset( $_POST['title'] ) ? sanitize_text_field( $_POST['title'] ) : '';
            if ( empty( $title ) ) {
                wp_send_json_error( [ 'message' => 'Title is required for titles mode.' ] );
            }
            $prompt         = $this->build_title_prompt( $title, $language, $length, $tone, $instructions );
            $expected_title = $title;
        }

        $result = $this->call_openai_chat( $api_key, $model, $prompt );
        if ( is_wp_error( $result ) ) {
            $this->log_error( 'chat_generation_ajax', $result->get_error_message() );
            wp_send_json_error( [ 'message' => $result->get_error_message() ] );
        }

        $content_raw = trim( $result );

        if ( $mode === 'topic' ) {
            list( $final_title, $body ) = $this->extract_title_and_body_from_response( $content_raw, $topic );
        } else {
            $final_title = $expected_title;
            $body  = $content_raw;
        }

        $post_id = wp_insert_post( [
            'post_type'    => self::CPT_SLUG,
            'post_status'  => 'draft',
            'post_title'   => $final_title,
            'post_content' => $body,
        ] );

        if ( is_wp_error( $post_id ) || ! $post_id ) {
            $msg = is_wp_error($post_id) ? $post_id->get_error_message() : 'Failed to save generated article.';
            $this->log_error( 'article_save_ajax', $msg );
            wp_send_json_error( [ 'message' => $msg ] );
        }

        update_post_meta( $post_id, '_boag_language', $language );
        update_post_meta( $post_id, '_boag_tone', $tone );
        update_post_meta( $post_id, '_boag_length', $length );
        update_post_meta( $post_id, '_boag_instructions', $instructions );
        update_post_meta( $post_id, '_boag_categories', $categories );
        update_post_meta( $post_id, '_boag_tags', $tags );
        update_post_meta( $post_id, '_boag_generate_featured_image', $generate_fi );
        update_post_meta( $post_id, '_boag_image_prompt', $image_prompt );
        update_post_meta( $post_id, '_boag_image_aspect', $image_aspect );
        update_post_meta( $post_id, '_boag_image_model', $image_model );
        update_post_meta( $post_id, '_boag_autopublish', $this->is_autopublish_enabled() ? 1 : 0 );

        wp_send_json_success( [
            'article_id' => $post_id,
            'title'      => $final_title,
        ] );
    }

    /* ===========================
     *  GENERATED ARTICLES PAGE
     * =========================== */

    public function render_generated_articles_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( isset( $_GET['boag_action'], $_GET['article_id'] ) ) {
            $action     = sanitize_text_field( wp_unslash( $_GET['boag_action'] ) );
            $article_id = intval( $_GET['article_id'] );

            if ( $action === 'delete' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'boag_delete_article_' . $article_id ) ) {
                wp_delete_post( $article_id, true );
                echo '<div class="notice notice-success"><p>Article deleted.</p></div>';
            }

            if ( $action === 'send' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'boag_send_article_' . $article_id ) ) {
                $this->send_article_to_draft_post( $article_id );
            }
        }

        $view_article_id = isset( $_GET['view_article_id'] ) ? intval( $_GET['view_article_id'] ) : 0;
        $paged = isset( $_GET['boag_paged'] ) ? max( 1, intval( $_GET['boag_paged'] ) ) : 1;

        $query = new WP_Query( [
            'post_type'      => self::CPT_SLUG,
            'post_status'    => 'draft',
            'posts_per_page' => 20,
            'paged'          => $paged,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ] );
        ?>
        <div class="wrap">
            <h1>Generated AI Articles</h1>

            <?php if ( ! $query->have_posts() ) : ?>
                <p>No AI articles generated yet.</p>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Title</th>
                            <th>Language</th>
                            <th>Created</th>
                            <th>Sent to WP</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        while ( $query->have_posts() ) :
                            $query->the_post();
                            $article_id     = get_the_ID();
                            $language       = get_post_meta( $article_id, '_boag_language', true );
                            $sent_post_id   = get_post_meta( $article_id, '_boag_sent_post_id', true );
                            $sent_post_link = $sent_post_id ? get_edit_post_link( $sent_post_id ) : '';
                            ?>
                            <tr>
                                <td><?php echo esc_html( get_the_title() ); ?></td>
                                <td><?php echo esc_html( $language ); ?></td>
                                <td><?php echo esc_html( get_the_date( 'Y-m-d H:i' ) ); ?></td>
                                <td>
                                    <?php
                                    if ( $sent_post_id && $sent_post_link ) {
                                        echo '<a href="' . esc_url( $sent_post_link ) . '">View</a>';
                                    } else {
                                        echo 'No';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $base_args = [ 'page' => 'boag_generated_articles' ];

                                    $view_url = add_query_arg(
                                        array_merge( $base_args, [ 'view_article_id' => $article_id ] ),
                                        admin_url( 'admin.php' )
                                    );

                                    $delete_url = wp_nonce_url(
                                        add_query_arg(
                                            array_merge( $base_args, [ 'boag_action' => 'delete', 'article_id' => $article_id, 'boag_paged' => $paged ] ),
                                            admin_url( 'admin.php' )
                                        ),
                                        'boag_delete_article_' . $article_id
                                    );

                                    if ( $sent_post_id ) {
                                        $send_button = '<span>Already sent</span>';
                                    } else {
                                        $send_url = wp_nonce_url(
                                            add_query_arg(
                                                array_merge( $base_args, [ 'boag_action' => 'send', 'article_id' => $article_id, 'boag_paged' => $paged ] ),
                                                admin_url( 'admin.php' )
                                            ),
                                            'boag_send_article_' . $article_id
                                        );
                                        $send_button = '<a class="button button-primary" href="' . esc_url( $send_url ) . '">Send to WP Draft</a>';
                                    }
                                    ?>
                                    <a class="button" href="<?php echo esc_url( $view_url ); ?>">View</a>
                                    <?php echo $send_button; ?>
                                    <a class="button button-link-delete" href="<?php echo esc_url( $delete_url ); ?>" onclick="return confirm('Delete this generated article?');">Delete</a>
                                </td>
                            </tr>
                        <?php endwhile; wp_reset_postdata(); ?>
                    </tbody>
                </table>

                <?php
                $total_pages = $query->max_num_pages;
                if ( $total_pages > 1 ) {
                    $current_url = remove_query_arg( 'boag_paged', admin_url( 'admin.php?page=boag_generated_articles' ) );
                    echo '<div class="tablenav"><div class="tablenav-pages">';
                    echo paginate_links( [
                        'base'      => add_query_arg( 'boag_paged', '%#%', $current_url ),
                        'format'    => '',
                        'current'   => $paged,
                        'total'     => $total_pages,
                        'prev_text' => '&laquo;',
                        'next_text' => '&raquo;',
                    ] );
                    echo '</div></div>';
                }
                ?>
            <?php endif; ?>

            <?php if ( $view_article_id ) : ?>
                <?php
                $article = get_post( $view_article_id );
                if ( $article && $article->post_type === self::CPT_SLUG ) :
                    ?>
                    <hr />
                    <h2>Preview: <?php echo esc_html( get_the_title( $view_article_id ) ); ?></h2>
                    <div style="background:#fff; border:1px solid #ccd0d4; padding:20px; max-height:500px; overflow:auto;">
                        <?php echo wp_kses_post( $article->post_content ); ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    /* ===========================
     *  PROMPTS (HTML output)
     * =========================== */

    private function build_topic_prompt( $topic, $language, $length, $tone, $instructions ) {
        $base  = "You are an expert SEO content writer.\n\n";
        $base .= "Write a high-quality blog article in {$language} about the topic: \"{$topic}\".\n";
        $base .= "Target length: approximately {$length} words.\n";
        if ( ! empty( $tone ) ) $base .= "Tone: {$tone}.\n";
        if ( ! empty( $instructions ) ) $base .= "Additional instructions:\n{$instructions}\n";

        $base .= "\nStructure and formatting requirements:\n";
        $base .= "- Return the article as HTML suitable for direct use in a WordPress post body.\n";
        $base .= "- Do NOT include <html>, <head>, or <body> tags.\n";
        $base .= "- Start with a short intro paragraph (<p>), not a heading.\n";
        $base .= "- Use <h2> for main sections and <h3> for subsections where appropriate.\n";
        $base .= "- Use <p> for paragraphs, <ul><li> / <ol><li> for lists.\n";
        $base .= "- Do NOT include literal 'H2:' or 'H3:' in headings.\n";
        $base .= "- Do not output markdown; use HTML only.\n";

        $base .= "\nAt the very beginning, output a single line:\n";
        $base .= "TITLE: Your SEO-optimized title here\n";
        $base .= "Then a blank line and the full HTML body.\n";

        return $base;
    }

    private function build_title_prompt( $title, $language, $length, $tone, $instructions ) {
        $base  = "You are an expert SEO content writer.\n\n";
        $base .= "Write a high-quality blog article in {$language} with the exact title: \"{$title}\".\n";
        $base .= "Target length: approximately {$length} words.\n";
        if ( ! empty( $tone ) ) $base .= "Tone: {$tone}.\n";
        if ( ! empty( $instructions ) ) $base .= "Additional instructions:\n{$instructions}\n";

        $base .= "\nStructure and formatting requirements:\n";
        $base .= "- Return HTML only (no markdown).\n";
        $base .= "- No <html>/<head>/<body> tags.\n";
        $base .= "- Start with <p> intro.\n";
        $base .= "- Use <h2>/<h3>, bullet lists where useful.\n";
        $base .= "- No 'H2:'/'H3:' text in headings.\n";
        $base .= "- Return ONLY the HTML article body.\n";

        return $base;
    }

    private function extract_title_and_body_from_response( $content, $fallback_topic ) {
        $title = $fallback_topic . ' Article';
        $body  = $content;

        if ( preg_match( '/^TITLE:\s*(.+)$/mi', $content, $m ) ) {
            $title = trim( $m[1] );
            $body  = trim( str_replace( $m[0], '', $content ) );
        }
        return [ $title, $body ];
    }

    /* ===========================
     *  OPENAI CHAT
     * =========================== */

    private function call_openai_chat( $api_key, $model, $prompt ) {
        $endpoint = 'https://api.openai.com/v1/chat/completions';

        $body = [
            'model'    => $model,
            'messages' => [
                [ 'role' => 'system', 'content' => 'You are a helpful AI writing assistant.' ],
                [ 'role' => 'user',   'content' => $prompt ],
            ],
            'temperature' => 0.7,
        ];

        $args = [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => 'Bearer ' . $api_key,
            ],
            'body'    => wp_json_encode( $body ),
            'timeout' => 60,
        ];

        $response = wp_remote_post( $endpoint, $args );

        if ( is_wp_error( $response ) ) {
            $this->log_error( 'chat_api_http', $response->get_error_message() );
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $data = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code < 200 || $code >= 300 ) {
            $message = isset( $data['error']['message'] ) ? $data['error']['message'] : 'Unknown API error.';
            $this->log_error( 'chat_api_response', 'HTTP ' . $code . ' - ' . $message );
            return new WP_Error( 'openai_api_error', 'OpenAI API error: ' . $message );
        }

        if ( empty( $data['choices'][0]['message']['content'] ) ) {
            $this->log_error( 'chat_api_no_content', 'No content returned.' );
            return new WP_Error( 'openai_no_content', 'OpenAI did not return any content.' );
        }

        return $data['choices'][0]['message']['content'];
    }

    /* ===========================
     *  OPENAI IMAGE + FEATURED IMAGE
     * =========================== */

    private function call_openai_image_b64( $api_key, $model, $prompt, $size ) {
        $endpoint = 'https://api.openai.com/v1/images/generations';

        $payload = [
            'model'  => $model,
            'prompt' => $prompt,
            'size'   => $size,
            'n'      => 1,
        ];

        // gpt-image-1 supports output_format/quality (harmless if ignored by other models)
        $payload['output_format'] = 'jpeg';
        $payload['quality'] = 'high';

        $args = [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => 'Bearer ' . $api_key,
            ],
            'body'    => wp_json_encode( $payload ),
            'timeout' => 120,
        ];

        $response = wp_remote_post( $endpoint, $args );

        if ( is_wp_error( $response ) ) {
            $this->log_error( 'image_api_http', $response->get_error_message() );
            return $response;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $data = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code < 200 || $code >= 300 ) {
            $message = isset( $data['error']['message'] ) ? $data['error']['message'] : 'Unknown image API error.';
            $this->log_error( 'image_api_response', 'HTTP ' . $code . ' - ' . $message );
            return new WP_Error( 'openai_image_error', 'OpenAI image error: ' . $message );
        }

        if ( ! empty( $data['data'][0]['b64_json'] ) ) {
            return $data['data'][0]['b64_json'];
        }

        // Fallback: URL → download → return as base64
        if ( ! empty( $data['data'][0]['url'] ) ) {
            $url = $data['data'][0]['url'];
            $r = wp_remote_get( $url );
            if ( is_wp_error( $r ) ) {
                return $r;
            }
            $bin = wp_remote_retrieve_body( $r );
            if ( empty( $bin ) ) {
                return new WP_Error( 'image_download_failed', 'Failed to download generated image from URL.' );
            }
            return base64_encode( $bin );
        }

        return new WP_Error( 'openai_no_image', 'OpenAI did not return image data.' );
    }

    private function create_featured_image_for_title( $title, $api_key, $custom_prompt = '', $aspect = 'square', $model_override = '' ) {
        $model = $model_override ? $model_override : $this->get_default_image_model();

        $image_prompt = "Create a high-quality, clean, professional illustration for a blog post titled: \"{$title}\". " .
                        "No text in the image. Style: modern, eye-catching, suitable as a WordPress featured image.";
        if ( ! empty( $custom_prompt ) ) {
            $image_prompt .= " Additional style and details: {$custom_prompt}.";
        }

        $size = ( $aspect === 'landscape' ) ? '1536x1024' : '1024x1024';

        $b64 = $this->call_openai_image_b64( $api_key, $model, $image_prompt, $size );
        if ( is_wp_error( $b64 ) ) {
            $this->log_error( 'featured_image_create', $b64->get_error_message() );
            return $b64;
        }

        $image_data = base64_decode( $b64 );
        if ( ! $image_data ) {
            $this->log_error( 'image_decode', 'Failed to decode base64 image data.' );
            return new WP_Error( 'image_decode_failed', 'Failed to decode generated image (base64).' );
        }

        $upload_dir = wp_upload_dir();
        if ( ! empty( $upload_dir['error'] ) ) {
            $this->log_error( 'image_upload_dir', $upload_dir['error'] );
            return new WP_Error( 'upload_dir_error', $upload_dir['error'] );
        }

        $filename  = 'ai-featured-' . time() . '-' . wp_generate_uuid4() . '.jpg';
        $file_path = trailingslashit( $upload_dir['path'] ) . $filename;

        if ( ! file_put_contents( $file_path, $image_data ) ) {
            $this->log_error( 'image_file_write', 'Failed to write image file: ' . $file_path );
            return new WP_Error( 'file_write_error', 'Failed to write image file.' );
        }

        $attachment = [
            'post_mime_type' => 'image/jpeg',
            'post_title'     => sanitize_text_field( $title ),
            'post_content'   => '',
            'post_status'    => 'inherit',
        ];

        $attach_id = wp_insert_attachment( $attachment, $file_path );
        if ( is_wp_error( $attach_id ) ) {
            $this->log_error( 'image_attachment_insert', $attach_id->get_error_message() );
            return $attach_id;
        }

        if ( ! function_exists( 'wp_generate_attachment_metadata' ) ) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }

        $attach_data = wp_generate_attachment_metadata( $attach_id, $file_path );
        wp_update_attachment_metadata( $attach_id, $attach_data );

        return $attach_id;
    }

    /* ===========================
     *  SEND ARTICLE TO WP DRAFT (+featured image)
     * =========================== */

    private function send_article_to_draft_post_return_id( $article_id ) {
        $article = get_post( $article_id );
        if ( ! $article || $article->post_type !== self::CPT_SLUG ) {
            return 0;
        }

        $existing = get_post_meta( $article_id, '_boag_sent_post_id', true );
        if ( $existing ) {
            return (int) $existing;
        }

        $categories    = (array) get_post_meta( $article_id, '_boag_categories', true );
        $tags_raw      = (string) get_post_meta( $article_id, '_boag_tags', true );
        $tags_input    = [];
        $generate_fi   = (int) get_post_meta( $article_id, '_boag_generate_featured_image', true );
        $image_prompt  = (string) get_post_meta( $article_id, '_boag_image_prompt', true );
        $image_aspect  = (string) get_post_meta( $article_id, '_boag_image_aspect', true );
        $image_aspect  = ( $image_aspect === 'landscape' ) ? 'landscape' : 'square';

        $image_model   = (string) get_post_meta( $article_id, '_boag_image_model', true );

        if ( ! empty( $tags_raw ) ) {
            $tags_input = array_filter( array_map( 'trim', explode( ',', $tags_raw ) ) );
        }

        $new_post_id = wp_insert_post( [
            'post_type'    => 'post',
            'post_status'  => 'draft',
            'post_title'   => $article->post_title,
            'post_content' => $article->post_content,
            'post_category'=> array_map('intval', $categories ),
            'tags_input'   => $tags_input,
        ] );

        if ( is_wp_error( $new_post_id ) || ! $new_post_id ) {
            $msg = is_wp_error($new_post_id) ? $new_post_id->get_error_message() : 'Failed to create WordPress draft post.';
            $this->log_error( 'draft_post_create', $msg );
            return 0;
        }

        if ( $generate_fi ) {
            $api_key = get_option( 'boag_api_key', '' );
            if ( ! empty( $api_key ) ) {
                $attach_id = $this->create_featured_image_for_title(
                    $article->post_title,
                    $api_key,
                    $image_prompt,
                    $image_aspect,
                    $image_model
                );

                if ( ! is_wp_error( $attach_id ) && $attach_id ) {
                    set_post_thumbnail( $new_post_id, $attach_id );
                }
            }
        }

        update_post_meta( $article_id, '_boag_sent_post_id', $new_post_id );

        return (int) $new_post_id;
    }

    private function send_article_to_draft_post( $article_id ) {
        $new_post_id = $this->send_article_to_draft_post_return_id( $article_id );
        if ( ! $new_post_id ) {
            echo '<div class="notice notice-error"><p>Failed to create WordPress draft post.</p></div>';
            return;
        }
        $edit_link = get_edit_post_link( $new_post_id );
        echo '<div class="notice notice-success"><p>Article sent to WordPress draft posts. <a href="' . esc_url( $edit_link ) . '">Edit draft</a></p></div>';
    }

    /* ===========================
     *  UNINSTALL
     * =========================== */

    public static function uninstall() {
        if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
            return;
        }

        $delete_on_uninstall = (int) get_option( 'boag_delete_on_uninstall', 0 );
        if ( ! $delete_on_uninstall ) {
            return;
        }

        $posts = get_posts( [
            'post_type'      => self::CPT_SLUG,
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ] );

        if ( $posts ) {
            foreach ( $posts as $post_id ) {
                wp_delete_post( $post_id, true );
            }
        }

        global $wpdb;
        $table = $wpdb->prefix . 'boag_queue';
        $wpdb->query( "DROP TABLE IF EXISTS {$table}" );

        delete_option( 'boag_api_key' );
        delete_option( 'boag_default_model' );
        delete_option( 'boag_default_language' );
        delete_option( 'boag_default_image_model' );

        delete_option( 'boag_queue_enabled' );
        delete_option( 'boag_queue_items_per_run' );
        delete_option( 'boag_queue_max_attempts' );

        delete_option( 'boag_autopublish_enabled' );
        delete_option( 'boag_autopublish_posts_per_run' );
        delete_option( 'boag_autopublish_interval_minutes' );
        delete_option( 'boag_autopublish_last_run' );

        delete_option( 'boag_error_log' );
        delete_option( 'boag_delete_on_uninstall' );
        delete_option( 'boag_db_version' );
    }
}

/* ===========================
 *  Hooks
 * =========================== */

register_activation_hook( __FILE__, [ 'Bulk_OpenAI_Article_Generator', 'activate' ] );
register_deactivation_hook( __FILE__, [ 'Bulk_OpenAI_Article_Generator', 'deactivate' ] );
register_uninstall_hook( __FILE__, [ 'Bulk_OpenAI_Article_Generator', 'uninstall' ] );

// Boot
new Bulk_OpenAI_Article_Generator();

